<?php

namespace app\Model\Vendor;

use Illuminate\Notifications\Notifiable;
use Illuminate\Foundation\Auth\User as Authenticatable;

class Order extends Authenticatable
{
    use Notifiable;
    protected $table = 'orders';
    protected $guard = 'restaurant';


    public function orderStatus()
    {
        return  $this->hasOne('app\Model\Vendor\OrderStatus', 'id', 'order_status_id');
    }

    public static function list($request, $type = false)
    {
        $limit = $request->query('limit', config('admin.constants.DEFAULT_PAGE_LIMIT'));
        $search = !empty($request->search) ? $request->search : '';
        $status = !empty($request->status) ? $request->status : '';
        $start_date = !empty($request->start_date) ? $request->start_date : '';
        $end_date = !empty($request->end_date) ? $request->end_date : '';
        $query = Order::where(['restaurant_id' => $request['restaurant_id']])->with('orderStatus');

        if (!empty($type)) {
            if ($type == 'pending') {
                $query->where('order_status_id', config('admin.constants.ORDER_STATUS_PENDING'));
            } else {
                $query->where('order_status_id', '!=', config('admin.constants.ORDER_STATUS_PENDING'));
            }
        }

        if (!empty($search)) {
            $query->where(function ($q) use ($search) {
                $q->where('id', 'like', '%' . $search . '%');
                $q->orWhere('firstname', 'like', '%' . $search . '%');
                $q->orWhere('lastname', 'like', '%' . $search . '%');
            });
        }

        if (!empty($status) && $type == 'punched') {
            $query->where(function ($q) use ($status) {
                $q->where('order_status_id', $status);
            });
        }

        if (!empty($start_date)) {
            $query->where(\DB::raw('DATE_FORMAT(created_at , "%Y-%m-%d")'), '>=', $start_date);
        }

        if (!empty($end_date)) {
            $query->where(\DB::raw('DATE_FORMAT(created_at , "%Y-%m-%d")'), '<=', $end_date);
        }

        $query->orderBy('id', 'DESC');
        $records = $query->paginate($limit);
        return $records;
    }

    public static function details($orderId)
    {
        $query = Order::where(['id' => $orderId])->with('orderStatus')->with('orderItem');
        return $query->first();
    }

    public function orderItem()
    {
        return $this->hasMany('app\Model\Vendor\OrderItem', 'order_id', 'id')->with('itemDetail');
    }


    public static function getPaymentSummary($restaurantId,  $start_date = false,  $end_date = false)
    {
        // get sum of total amount which is paid & un-paid
        $wh['restaurant_id'] = $restaurantId;
        $wh['order_status_id'] = config('admin.constants.ORDER_STATUS_COMPLETED');
        $query = Order::select('vendor_payment_status', \DB::raw('SUM(sub_total) AS subTotal'))
            ->where($wh);
        if (!empty($start_date)) {
            $query->where(\DB::raw('DATE_FORMAT(created_at , "%Y-%m-%d")'), '>=', $start_date);
        }

        if (!empty($end_date)) {
            $query->where(\DB::raw('DATE_FORMAT(created_at , "%Y-%m-%d")'), '<=', $end_date);
        }

        return $query->groupBy('vendor_payment_status')->get()->pluck('subTotal', 'vendor_payment_status');
    }
}
