<?php

namespace app\Model\Api\V1;

use File;
use app\Traits\FilePathUrl;
use app\Model\Api\V1\Address;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\DB;
use Laravel\Passport\HasApiTokens;
use app\Rules\Api\ValidatePassword;
use Kyslik\ColumnSortable\Sortable;
use app\Transformers\RiderTransformer;
use Illuminate\Database\Eloquent\Model;

class User extends Model
{
    use FilePathUrl;
    use Sortable, HasApiTokens;

    protected $tables   = 'users';
    protected $hidden   = ['created_at', 'updated_at'];
    protected $fillable = ['role_id', 'user_name', 'email', 'country_code', 'phone_number', 'verification_otp', 'password', 'email_verified_at', 'first_name', 'last_name', 'is_enable_notification', 'remember_token', 'status', 'email'];
    protected $guarded  = [];

    public function avatars()
    {
        return $this->hasMany('app\Model\Api\V1\UserAvatars');
    }
    public function fcmTokens()
    {
        return $this->hasMany('app\Model\Api\V1\FcmToken');
    }
    public function setPasswordAttribute($password)
    {
        if (!empty($password)) {
            $this->attributes['password'] = bcrypt($password);
        }
    }

    public function addresses()
    {
        return $this->hasMany(Address::class);
    }

    public static function rules($id = null)
    {
        return [
            'firstName'   => 'bail|required|string|min:' . config('api.constants.user.firstName.minLength') . '|max:' . config('api.constants.user.firstName.maxLength') . '||regex:' . config('api.constants.user.firstName.regex'),
            'lastName'    => 'bail|required|string|min:' . config('api.constants.user.lastName.minLength') . '|max:' . config('api.constants.user.lastName.maxLength') . '||regex:' . config('api.constants.user.lastName.regex'),

            'email'       => 'bail|string|nullable|email|max:' . config('api.constants.user.email.maxLength') . '|unique:users,email,' . ($id ? $id : ''),

            'phone'       => 'bail|required|string|max:' . config('api.constants.user.phone_number.maxLength') . '|unique:users,phone_number,' . ($id ? $id : ''),

            'password'    => 'bail|required|string|min:' . config('api.constants.user.password.minLength') . '|max:' . config('api.constants.user.password.maxLength') . '|regex:' . config('api.constants.user.password.regex'),
            'deviceToken' => 'bail|required|',
            'deviceType'  => 'bail|required|in:1,2',
        ];
    }
    public static function loginRules()
    {
        return [
            //'email' => 'bail|required|string|email|max:' . config('api.constants.user.email.maxLength') . '|exists:users,email',
            // 'phone'       => 'bail|required|max:' . config('api.constants.user.phone_number.maxLength') . '|exists:users,phone_number',
            'phone'       => 'bail|required|max:' . config('api.constants.user.phone_number.maxLength'),
            'password'    => 'bail|required|string|min:' . config('api.constants.user.password.minLength') . '|max:' . config('api.constants.user.password.maxLength') . '|regex:' . config('api.constants.user.password.regex'),
            'deviceToken' => 'bail|required|nullable',
            'deviceType'  => 'bail|required|in:1,2',
        ];
    }

    public static function voipTokenRules()
    {
        return [
            'voipToken' => 'bail|required|string|max:' . config('api.constants.user.voipToken.maxLength'),
        ];
    }

    public static function forgotPasswordRules()
    {
        return [
            //'email' => 'bail|required|string|email|max:' . config('admin.constants.user.email.maxLength') . '|exists:users,email',
            'phone' => 'bail|required|string|max:' . config('admin.constants.user.mobile_no.maxLength'),
        ];
    }
    public static function changeMobileRules()
    {
        return [
            //'email' => 'bail|required|string|email|max:' . config('admin.constants.user.email.maxLength') . '|exists:users,email',
            'mobile_no' => 'bail|required|string|max:' . config('admin.constants.user.mobile_no.maxLength') . '|unique:users,phone_number',
        ];
    }
    public static function apiProfileRules($id = null)
    {
        return [
            'userName'  => 'bail|required|string|min:' . config('api.constants.user.userName.minLength') . '|max:' . config('api.constants.user.userName.maxLength') . '||regex:' . config('api.constants.user.userName.regex'),
            'firstName' => 'bail|required|string|min:' . config('api.constants.user.firstName.minLength') . '|max:' . config('api.constants.user.firstName.maxLength') . '||regex:' . config('api.constants.user.firstName.regex'),
            'lastName'  => 'bail|required|string|min:' . config('api.constants.user.lastName.minLength') . '|max:' . config('api.constants.user.lastName.maxLength') . '||regex:' . config('api.constants.user.lastName.regex'),
            'avatar'    => 'bail|nullable|mimes:' . config('api.constants.user.avatar.mimes') . '|max:' . config('api.constants.user.avatar.size'), //formats: jpeg, png, bmp, gif, svg
        ];
    }

    public static function apiAvatarRules()
    {
        return [
            'avatar' => 'bail|required|mimes:' . config('admin.constants.user.avatar.mimes') . 'max:' . config('admin.constants.user.avatar.size'), //formats: jpeg, png, bmp, gif, svg
        ];
    }
    public static function changePasswordRules($user)
    {
        return [
            'password'    => ['required', new ValidatePassword($user)],
            'newPassword' => 'bail|required|string|min:' . config('api.constants.user.password.minLength') . '|max:' . config('api.constants.user.password.maxLength') . '|regex:' . config('api.constants.user.password.regex') . '|different:password',
        ];
    }

    public static function changeEmailRules($user)
    {
        return [
            'email'         => ['bail', 'required', 'email:filter', 'unique:users', new \app\Rules\V1\ChangeEmailRule($user)],
            'confirm_email' => ['bail', 'same:email'],
        ];
    }

    public static function messages()
    {
        return [
            'email.required'        => __('api/validation/user.email.required'),
            'email.string'          => __('api/validation/user.email.string'),
            'email.email'           => __('api/validation/user.email.email'),
            'email.max'             => __('api/validation/user.email.max'),
            'email.unique'          => __('api/validation/user.email.unique'),
            'email.exists'          => __('api/validation/user.email.exists'),

            'password.required'     => __('api/validation/user.password.required'),
            'password.string'       => __('api/validation/user.password.string'),
            'password.min'          => __('api/validation/user.password.min'),
            'password.max'          => __('api/validation/user.password.max'),
            'password.regex'        => __('api/validation/user.password.regex'),
            'password.confirmed'    => __('api/validation/user.password.confirmed'),

            'password_confirmation' => __('api/validation/user.passwordConfirmation.same'),

            'userName.required'     => __('api/validation/user.userName.required'),
            'userName.min'          => __('api/validation/user.userName.min'),
            'userName.max'          => __('api/validation/user.userName.max'),
            'userName.regex'        => __('api/validation/user.userName.regex'),

            'firstName.required'    => __('api/validation/user.firstName.required'),
            'firstName.string'      => __('api/validation/user.firstName.string'),
            'firstName.min'         => __('api/validation/user.firstName.min'),
            'firstName.max'         => __('api/validation/user.firstName.max'),
            'firstName.regex'       => __('api/validation/user.firstName.regex'),

            'lastName.required'     => __('api/validation/user.lastName.required'),
            'lastName.string'       => __('api/validation/user.lastName.string'),
            'lastName.min'          => __('api/validation/user.lastName.min'),
            'lastName.max'          => __('api/validation/user.lastName.max'),
            'lastName.regex'        => __('api/validation/user.lastName.regex'),

            'avatar.required'       => __('api/validation/user.avatar.required'),
            'avatar.mimes'          => __('api/validation/user.avatar.mimes'),
            'avatar.max'            => __('api/validation/user.avatar.filesize'),

            'newPassword.required'  => __('api/validation/user.newPassword.required'),
            'newPassword.string'    => __('api/validation/user.newPassword.string'),
            'newPassword.min'       => __('api/validation/user.newPassword.min'),
            'newPassword.max'       => __('api/validation/user.newPassword.max'),
            'newPassword.regex'     => __('api/validation/user.newPassword.regex'),
            'newPassword.different' => __('api/validation/user.newPassword.different'),

            'phone.required'        => __('api/validation/user.phone_no.required'),
            'phone.min'             => __('api/validation/user.phone_no.min'),
            'phone.max'             => __('api/validation/user.phone_no.max'),
            'phone.exists'          => __('api/validation/user.phone_no.exists'),
            'phone.unique'          => __('api/validation/user.phone_no.unique'),

            'type.in'               => 'Type value must be one of values[1, 2, 3]',
            'deviceToken.required'  => __('api/validation/user.deviceToken.required'),
            'deviceType.required'   => __('api/validation/user.deviceType.required'),
            'deviceType.in'         => __('api/validation/user.deviceType.in'),

            'is_rider.notIn' =>     __('api/validation/user.rider.notIn'),

        ];
    }

    /**
     * @Method: fullName()
     * @Scope: public
     * @Params: userId
     * @Description: return fullName of given user
     * @Created 12/Dec/2019
     * @Updated 20/Dec/2019
     */
    public static function fullName($id = null)
    {
        if (empty($id)) {
            return false;
        }

        $user = self::select('user_name', \DB::raw('CONCAT(first_name, " ", last_name) AS full_name'))
            ->where('id', '=', "$id")->first();
        if (!empty($user)) {
            $udata = $user->toArray();
            return ['full_name' => $udata['full_name'], 'user_name' => $udata['user_name'], 'avatar' => null];
        }
        return false;
    }

    /**
     * @Method: userProfileInfo()
     * @Scope: public
     * @Params: userId
     * @Description: this function will return user profile info, facility info, department & title
     * @Created 17/July/2021
     * @Updated 17/July/2021
     */
    public static function userProfileInfo($user_id)
    {
        $data = [];
        if (!empty($user_id)) {
            $query = User::select(['*', \DB::raw('CONCAT(first_name, " ", last_name) AS full_name')])
                ->with([
                    'avatars' => function ($q) {
                        $q->latest()->first();
                    },
                ])
                ->where('id', $user_id)->with('rider');
            $userDetail = $query->first();
            if (!empty($userDetail)) {
                $data['profile']['userId'] = (int) $userDetail->id;
                $data['profile']['fullName']         = trim($userDetail->first_name . ' ' . $userDetail->last_name);
                $data['profile']['email']            = $userDetail->email;
                $data['profile']['phone']            = $userDetail->phone_number;
                $data['profile']['isPhoneVerified']  = ($userDetail->phone_number_verified == 1 ? true : false);
                $data['profile']['isDeliveryPerson'] = ($userDetail->is_delivery_person == 1 ? true : false);

                $data['profile']['userImage']           = $userDetail->profile_image;
                $data['profile']['notificationStatus'] = (bool) $userDetail->notification_status;

                $data['profile']['isRiderApplied'] = !empty($userDetail->rider) ? true : false;
                $data['profile']['isOnline'] = (bool) $userDetail->is_active;
                $data['profile']['riderData'] = !empty($userDetail->rider) ?  (new RiderTransformer)->transform($userDetail->rider) : null;
            }
        }
        return $data;
    }

    /**
     * @Method: verifyOtp()
     * @Scope: public static
     * @Params: userId
     * @Description: this function will return true / false if mention params are valid
     * @Created 03/June/2020
     * @Updated 05/June/2020
     */
    public static function verifyOtp()
    {
        return [
            'phone' => 'bail|required|max:' . config('api.constants.user.phone_number.maxLength') . '|exists:users,phone_number',
            'otp'   => 'bail|required|numeric|min:' . config('api.constants.user.otp.minLength'),
            'type'  => 'required|' . Rule::in([1, 2, 3]),
        ];
    }

    /**
     * @Method: videoRules()
     * @Scope: public static
     * @Params: userId
     * @Description: this function will return true / false if mention params are valid
     * @Created 05/June/2020
     * @Updated 05/June/2020
     */

    public static function videoRules()
    {
        return [
            'id'        => 'bail|required|numeric',
            'mediaFile' => 'bail|required|mimes:jpeg,jpg,png,gif,svg,wmv,mov,avi,mp4,mp3|max:' . config('api.constants.user.media.maxLength'),
        ];
    }

    /**
     * Author: NMG
     * function: userMedia
     * Description: Generate user media of media path by just providing user id
     * Input: user id, image_id(optional), is_url(flag)
     * output: ['key'(for s3 url generation)]
     * Dated: 08/June/2020
     */

    public static function uploadMedia($file = [], $user_id)
    {
        if (empty($file) || empty($user_id)) {
            return false;
        }
        $uploaded  = false;
        $uploadDir = config('admin.path.UPLOAD_DIR');
        $mediaDir  = config('admin.path.media.DIR');
        // getting extension
        $ext = $file->getClientOriginalExtension();
        if (empty($ext)) {
            $mime = File::mimeType($file);
            if ($mime == 'video/mp4') {
                $ext = 'mp4';
            }
            if ($mime == 'video/wmv') {
                $ext = 'wmv';
            }
            if ($mime == 'video/avi') {
                $ext = 'avi';
            }
            if ($mime == 'video/mov') {
                $ext = 'mov';
            }
        }
        //rename video
        $fileName = uniqid() . '.' . $ext;

        // check if image uploaded on aws s3
        if (env('S3_ENABLED') == true) {
            $fileDir = $uploadDir . '/' . $mediaDir . '/' . $user_id . '/';
            //Original file
            $originalFile = $uploadDir . '/' . $mediaDir . '/' . $user_id . '/' . $fileName;
            $uploaded     = \Storage::disk('s3')->put(
                $originalFile,
                file_get_contents($file),
                'public'
            );
        } else {
            $basePath = storage_path('app/public') . '/' . $uploadDir . '/' . $mediaDir . '/' . $user_id . '/';
            // create directory
            if (!File::exists($basePath)) {
                File::makeDirectory($basePath, 0755, true);
            }
            //origional image
            $uploaded = $file->move($basePath, $fileName);
        }
        //return response
        if ($uploaded) {
            $mediaDetails = [
                'user_id' => $user_id,
                'media'   => $fileName,
                'type'    => $ext,
            ];
            return $mediaDetails;
        }
        return false;
    }
    /**
     * Author: NMG
     * @param string $tmpEmail
     * Function: Veryfy Email Rules
     * Description: To check email and otp
     * Dated: 09/June/2020
     */
    public static function verifyEmail($tmpEmail = null)
    {
        $email = $tmpEmail ?? 'email';
        return [
            'email' => 'bail|required|string|email:filter|max:' . config('api.constants.user.email.maxLength') . '|exists:users,' . $email,
            'otp'   => 'bail|required|numeric|min:' . config('api.constants.user.otp.minLength'),
        ];
    }

    /**
     * Author: NMG
     * Function: Veryfy Mobile Rules
     * Description: To check mobile and otp
     * Dated: 26/Apr/2021
     */
    public static function verifyMobile($tmpPhone = null)
    {
        $phone_number = $tmpPhone ?? 'phone_number';
        return [
            'mobile_no' => 'bail|required|string|max:' . config('api.constants.user.phone_number.maxLength') . '|exists:users,' . $phone_number,
            'otp'       => 'bail|required|numeric|min:' . config('api.constants.user.otp.minLength'),
            // 'type'      => 'bail|required|numeric|min:' . config('api.constants.user.otp.type'),
        ];
    }
    /**
     * reset password rules
     *
     * @return array
     */
    public static function resetPasswordRules()
    {
        return [
            'phone'                 => 'bail|required|string|max:' . config('api.constants.user.phone_number.maxLength'),
            'password'              => 'bail|required|string|min:' . config('api.constants.user.password.minLength') . '|max:' . config('api.constants.user.password.maxLength') . '|regex:' . config('api.constants.user.password.regex'),
            'password_confirmation' => 'required|same:password',
            'otp'                   => 'bail|required|numeric|min:' . config('api.constants.user.otp.minLength'),
        ];
    }

    public static function riderProfileRules()
    {
        return [
            'is_rider'    =>       ['bail', Rule::notIn([1])],
            "age"                 => ['bail', 'required', 'numeric'],
            "id_proof"            => ['bail', 'required', 'url'],
            "vehicle_information" => ['bail', 'required', 'url'],
            "driving_licence"     => ['bail', 'required', 'url'],
            "bank"                => ['bail', 'required', 'string', 'max:' . config('api.constants.user.password.maxLength')], // TODO max => change for bank name
            "account_number"      => ['bail', 'required', 'string', 'max:' . config('api.constants.user.password.maxLength')], // TODO max => change for account],

        ];
    }
    /**
     * one to one relationship with delivery_person.
     *
     * @return void
     */
    public function rider()
    {
        return $this->hasOne(DeliveryPerson::class);
    }
}
