<?php

namespace app\Model\Api\v1;

// use app\Rules\Api\ValidatePassword;
use app\Traits\FilePathUrl;
use Illuminate\Database\Eloquent\Model;
use Kyslik\ColumnSortable\Sortable;
// use Illuminate\Validation\Rule;
use Laravel\Passport\HasApiTokens;
use Illuminate\Support\Facades\DB;
use app\Model\Admin\GeneralSetting;
use app\Model\Api\V1\FoodCategory;

class Restaurant extends Model
{
    use FilePathUrl;
    use Sortable, HasApiTokens;

    protected $tables = 'restaurants';
    protected $hidden = ['created_at', 'updated_at', 'owner_id'];
    protected $fillable = ['name', 'bio', 'resturent_type_local', 'resturent_type_continental', 'resturent_type_groceries', 'address1', 'address2', 'city', 'region', 'zipcode', 'delivery_time', 'latitude', 'longitude', 'status', 'is_include_commission'];
    protected $guarded = [];


    public static function rules($id = null)
    {
        // type => [1=>Local,2=>Continental,3=>Groceries]
        return [
            'latitude' => 'bail|required|string',
            'longitude' => 'bail|string|required',
            'type' => 'bail|string|in:1,2,3|nullable',
            'categoryId' => 'bail|string|nullable',
            'searchString' => 'bail|string|nullable',
            'page' => 'bail|numeric|required',
        ];
    }

    public static function messages()
    {
        return [
            'latitude.required' => __('api/validation/restaurant.latitude.required'),
            'latitude.string' => __('api/validation/restaurant.latitude.string'),

            'longitude.required' => __('api/validation/restaurant.longitude.required'),
            'longitude.string' => __('api/validation/restaurant.longitude.string'),

            'type.required' => __('api/validation/restaurant.type.required'),
            'type.string' => __('api/validation/restaurant.type.string'),

            'categoryId.required' => __('api/validation/restaurant.categoryId.required'),
            'categoryId.string' => __('api/validation/restaurant.categoryId.string'),

            'searchString.required' => __('api/validation/restaurant.searchString.required'),
            'searchString.string' => __('api/validation/restaurant.searchString.string'),

            'page.required' => __('api/validation/restaurant.page.required'),
            'page.string' => __('api/validation/restaurant.page.string'),
            'page.numeric' => __('api/validtion/restaurant.page.numeric')

        ];
    }

    public function latestAvatar()
    {
        return $this->hasMany('app\Model\Admin\UserAvatars');
    }

    public function activeUser()
    {
        return $this->belongsTo('app\Model\Admin\User', 'owner_id', 'id')->where(['status' => 1, 'is_active' => 1])->with([
            'latestAvatar' => function ($q) {
                $q->latest()->first();
            },
        ]);
    }

    public function user()
    {
        return $this->belongsTo('app\Model\Admin\User', 'owner_id', 'id');
    }

    public function menu()
    {
        return $this->hasMany('app\Model\Vendor\Restaurant_menu', 'restaurant_id', 'id')->with('foodCategory');
    }

    public function foodCategory()
    {
        return $this->hasMany('app\Model\Vendor\FoodCategory', 'food_category_id', 'id');
    }

    /**
     * Author : jaidev
     * email : jaidev@nmgtechnologies.com
     * method : getRestaurant
     * desc : fetch restaurant list within allowed range (in miles)
     */
    public static function getRestaurant($request)
    {
        // getting delivery range
        $deliveryRangeData = GeneralSetting::where('slug', 'delivery_range')->first();
        $deliveryRange = (!empty($deliveryRangeData) ? $deliveryRangeData->value : 0);

        $latitude = $request->latitude;
        $longitude = $request->longitude;

        $page = $request->query('page', 1);
        $limit = $request->query('limit', config('admin.constants.DEFAULT_PAGE_LIMIT'));

        $select = ['restaurants.*', DB::raw('(3959 * acos( cos( radians(' . $latitude . ') ) * cos( radians( latitude ) ) * cos( radians( longitude ) - radians(' . $longitude . ') ) + sin( radians(' . $latitude . ') ) * sin( radians( latitude ) ) ) ) as distanceInRange')];
        $query = Restaurant::select($select)->whereRaw('((3959 * acos( cos( radians(' . $latitude . ') ) * cos( radians( latitude ) ) * cos( radians( longitude ) - radians(' . $longitude . ') ) + sin( radians(' . $latitude . ') ) * sin( radians( latitude ) ) ) )  < ' . $deliveryRange . ')')->with('activeUser')->with('menu');
        $search = !empty($request->searchString) ? $request->searchString : '';


        // restaurant type filter [1=>local,2=>continent,3=>grocery]
        $restaurantType = !empty($request->type) ? $request->type : null;
        $query->where(['restaurants.status' => 1]);
        if ($restaurantType == 1) {
            $query->where(['restaurants.resturent_type_local' => 1]);
        } else if ($restaurantType == 2) {
            $query->where(['restaurants.resturent_type_continental' => 1]);
        } else if ($restaurantType == 3) {
            $query->where(['restaurants.resturent_type_groceries' => 1]);
        }

        // food category filter
        $foodCategory = !empty($request->categoryId) ? $request->categoryId : null;
        if ($foodCategory) {
            $query->where(function ($q) use ($foodCategory) {
                $q->orwhereHas('menu', function ($q) use ($foodCategory) {
                    return $q->where('restaurant_menus.food_category_id', $foodCategory);
                });
            });
        }

        $query->where(function ($q) {
            $q->orwhereHas('user', function ($q) {
                $q->where('users.status', 1);
                return $q->where('users.is_active', 1);
            });
        });

        if (!empty($search)) {
            $query->where(function ($q) use ($search) {
                $q->where('restaurants.name', 'LIKE', "%$search%");
                $q->orwhereHas('menu', function ($q) use ($search) {
                    return $q->where('name', 'LIKE', "%$search%");
                });
            });
        }

        $query->orderBy('distanceInRange', 'asc');
        $records = $query->paginate($limit);

        return $records;
    }


    public static function getDistinctFoodCategories($restaurantId)
    {
        return FoodCategory::getDistinctCategory($restaurantId);
    }



    public static function get($restaurantId)
    {
        return Restaurant::where('id', $restaurantId)->with('user')->first();
    }
}
