<?php

namespace app\Model\Api\V1;

use app\Model\Api\V1\User;
use Illuminate\Database\Eloquent\Model;

class Order extends Model
{
    /**
     * attributes of model which can not be mass-assigned.
     *
     * @var array
     */
    protected $tables   = 'orders';
    protected $guarded = [];

    public static function rules($id = null)
    {
        return [
            'txnNo' => 'bail|required|string|unique:transactions,txn_no',
            'paymentType' => 'bail|required|numeric|in:1,2',
            'paymentStatus' => 'bail|required|numeric|in:0,1',
            'addressId' => 'bail|required|numeric|exists:addresses,id',
            'deliveryNote' => 'bail|string|nullable',
        ];
    }

    public static function cancelRules($id = null)
    {
        return [
            'orderId' => 'bail|required|numeric|exists:orders,id',
        ];
    }

    public static function messages()
    {
        return [];
    }

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function restaurant()
    {
        return $this->hasOne('app\Model\Api\V1\Restaurant', 'id', 'restaurant_id')->with('user');
    }
    public function orderStatus()
    {
        return $this->hasOne('app\Model\Api\V1\OrderStatus', 'id', 'order_status_id');
    }

    public function orderItem()
    {
        return $this->hasMany('app\Model\Api\V1\OrderItem', 'order_id', 'id')->with('itemDetail');
    }


    public function deliveryPerson()
    {
        return $this->hasOne('app\Model\Api\V1\DeliveryPerson', 'id', 'delivery_person_id')->with('user');
    }




    function list($request)
    {
        $limit = $request->query('limit', config('api.constants.DEFAULT_PAGE_LIMIT'));
        $query = Order::with('orderStatus')->with('orderItem')->orderBy('id', 'desc');
        $query->when(isset($request->userId), function ($q) use ($request) {
            return $q->where('user_id', $request->userId);
        });

        $query->when(isset($request->orderStatus), function ($q) use ($request) {
            if (is_array($request->orderStatus) && count($request->orderStatus) > 0) {
                return $q->whereIn('order_status_id', $request->orderStatus);
            }
        });

        return $query->paginate($limit);
    }

    public static function details($request, $orderId)
    {
        $query = Order::where(['id' => $orderId])->with('orderStatus')->with('orderItem')->with('restaurant')->with('deliveryPerson');
        $query->when(isset($request->userId), function ($q) use ($request) {
            return $q->where('user_id', $request->userId);
        });
        return $query->first();
    }
}
