<?php

namespace app\Model\Admin;

use app\Model\Admin\UserAvatars;
use app\Rules\Api\ValidatePassword;
use app\Model\Admin\DeliveryPersons;
use app\Traits\FilePathUrl;
use File;
use Illuminate\Database\Eloquent\Model;
use Image;
use Kyslik\ColumnSortable\Sortable;

class User extends Model
{
    use FilePathUrl;

    use Sortable;

    protected $tables = 'users';
    protected $hidden = ['created_at', 'updated_at'];
    protected $fillable = [
        'role_id', 'first_name', 'last_name', 'user_name', 'profile_image', 'status', 'email', 'phone_number',
        'password', 'verification_otp', 'remember_token', 'verification_otp_sent_at', 'reset_password_sent_at'
    ];

    public $sortable = ['id', 'first_name', 'last_name', 'user_name', 'email', 'phone_number', 'created_at'];

    public static function rules($id = null)
    {
        // dd($id);
        $rules =  [
            //'user_name' => 'bail|required|string|min:' . config('admin.constants.user.user_name.minLength') . '|max:' . config('admin.constants.user.user_name.maxLength') . ($id ? $id : ''),
            //'email' => 'bail|required|string|email|max:' . config('admin.constants.user.email.maxLength')  . ($id ? $id : ''),
            'phone_number' => 'bail|required|unique:users,phone_number,' . ($id ? $id : '') . '|min:' . config('admin.constants.user.mobile_no.minLength') . '|max:' . config('admin.constants.user.mobile_no.maxLength') . ($id ? $id : ''),
            'first_name' => 'bail|required|string|min:' . config('admin.constants.user.first_name.minLength') . '|max:' . config('admin.constants.user.first_name.maxLength'),
            'last_name' => 'bail|required|string|min:' . config('admin.constants.user.last_name.minLength') . '|max:' . config('admin.constants.user.last_name.maxLength'),
            'avatar' => 'bail|nullable|mimes:' . config('admin.constants.user.avatar.mimes') . '|max:' . config('admin.constants.user.avatar.size'), //formats: jpeg, png, bmp, gif, svg
            //'role_id' => 'bail|required',

        ];
        return $rules;
    }


    public static function medicalWorkersRules($id = null)
    {
        return [
            'user_name' => 'bail|required|string|min:' . config('admin.constants.user.user_name.minLength') . '|max:' . config('admin.constants.user.user_name.maxLength') . '|unique:users,user_name,' . ($id ? $id : ''),
            'email' => 'bail|required|string|email|max:' . config('admin.constants.user.email.maxLength') . '|unique:users,email,' . ($id ? $id : ''),
            'mobile_no' => 'bail|required|string||min:' . config('admin.constants.user.mobile_no.minLength') . '|max:' . config('admin.constants.user.mobile_no.maxLength'),
            'first_name' => 'bail|required|string|min:' . config('admin.constants.user.first_name.minLength') . '|max:' . config('admin.constants.user.first_name.maxLength'),
            'last_name' => 'bail|required|string|min:' . config('admin.constants.user.last_name.minLength') . '|max:' . config('admin.constants.user.last_name.maxLength'),
            'avatar' => 'bail|nullable|mimes:' . config('admin.constants.user.avatar.mimes') . '|max:' . config('admin.constants.user.avatar.size'), //formats: jpeg, png, bmp, gif, svg
        ];
    }
    public static function loginRules()
    {
        return [
            'email' => 'bail|required|string|email|exists:users,email|max:' . config('admin.constants.user.email.maxLength'),
            'password' => 'bail|required|string|min:' . config('admin.constants.user.password.minLength') . '|max:' . config('admin.constants.user.password.maxLength'),
        ];
    }

    public static function forgotPasswordRules()
    {
        return [
            'email' => 'bail|required|string|email|max:' . config('admin.constants.user.email.maxLength') . '|exists:users,email',
        ];
    }

    public static function changeEmailRules($id = null)
    {
        return [
            'old_email' => 'bail|required|string|min:' . config('admin.constants.user.email.minLength') . '|max:' . config('admin.constants.user.email.maxLength'),
            'new_email' => 'bail|required|string|unique:users,email|min:' . config('admin.constants.user.email.minLength') . '|max:' . config('admin.constants.user.email.maxLength')
        ];
    }
    public static function otpRules($id = null)
    {
        return [
            'otp.*' => 'bail|required|numeric|digits:1',
        ];
    }

    public static function changeEmailMessages()
    {
        return [
            'old_email.required' => __('admin/validation/user.email.required'),
            'new_email.required' => __('admin/validation/user.email.required'),
        ];
    }


    public static function changePhoneNumberRules($id = null)
    {
        return [
            'old_number' =>  'bail|required|min:' . config('admin.constants.user.phone_number.minLength') . '|max:' . config('admin.constants.user.phone_number.maxLength') . ($id ? $id : ''),
            'new_number' =>  'bail|required|unique:users,phone_number,' . ($id ? $id : '') . '|min:' . config('admin.constants.user.phone_number.minLength') . '|max:' . config('admin.constants.user.phone_number.maxLength') . ($id ? $id : ''),
        ];
    }


    public static function currentPasswordRules($id = null)
    {
        return [
            'password' => 'bail|required|string|min:' . config('admin.constants.user.password.minLength') . '|max:' . config('api.constants.user.password.maxLength')
        ];
    }



    public static function changePasswordRules($user)
    {
        return [
            'password' => ['required', new ValidatePassword($user)],
            'new_password' => 'bail|required|string|min:' . config('admin.constants.user.password.minLength') . '|max:' . config('api.constants.user.password.maxLength') . '|different:password',
            'new_password_confirmation' => 'bail|required|same:new_password',
        ];
    }

    public static function changePasswordVendorRules($user)
    {
        return [
            'password' => ['required'],
            'new_password' => 'bail|required|string|min:' . config('admin.constants.user.password.minLength') . '|max:' . config('api.constants.user.password.maxLength') . '|different:password',
            'new_password_confirmation' => 'bail|required|same:new_password',
        ];
    }

    public static function profileRules($user = null)
    {
        return [
            'first_name' => 'bail|required|string|min:' . config('admin.constants.user.first_name.minLength') . '|max:' . config('admin.constants.user.first_name.maxLength'),
            'last_name' => 'bail|required|string|min:' . config('admin.constants.user.last_name.minLength') . '|max:' . config('admin.constants.user.last_name.maxLength'),
            'avatar' => 'bail|nullable|mimes:jpeg,jpg,png,gif,svg|max:' . config('admin.constants.user.avatar.size'), //formats: jpeg, png, bmp, gif, svg
        ];
    }
    public static function resetPasswordRules()
    {
        return [
            'email' => 'bail|required|string|email|max:' . config('api.constants.user.email.maxLength') . '|exists:users,email',
            'new_password' => 'bail|required|string|min:' . config('api.constants.user.password.minLength') . '|max:' . config('api.constants.user.password.maxLength'),
            'new_password_confirmation' => 'bail|required|same:new_password',
        ];
    }
    public static function messages()
    {
        return [
            'role_id.required' => __('admin/validation/user.role_id.required'),
            //'user_name.required' => __('admin/validation/user.user_name.required'),
            //'user_name.string' => __('admin/validation/user.user_name.string'),
            //'user_name.min' => __('admin/validation/user.user_name.min'),
            //'user_name.max' => __('admin/validation/user.user_name.max'),
            //'user_name.regex' => __('admin/validation/user.user_name.regex'),
            //'email.required' => __('admin/validation/user.email.required'),
            'email.string' => __('admin/validation/user.email.max'),
            'email.email' => __('admin/validation/user.email.min'),
            'email.max' => __('admin/validation/user.email.max'),
            'email.unique' => __('admin/validation/user.email.unique'),
            'email.exists' => __('admin/validation/user.email.exists'),

            'mobile_no.required' => __('admin/validation/user.mobile_no.required'),
            'mobile_no.min' => __('admin/validation/user.mobile_no.min'),
            'mobile_no.max' => __('admin/validation/user.mobile_no.max'),

            'password.required' => __('admin/validation/user.password.required'),
            'password.string' => __('admin/validation/user.password.string'),
            'password.min' => __('admin/validation/user.password.min'),
            'password.max' => __('admin/validation/user.password.max'),
            'password.regex' => __('admin/validation/user.password.regex'),
            'new_password.required' => __('admin/validation/user.new_password.required'),
            'new_password.string' => __('admin/validation/user.new_password.string'),
            'new_password.min' => __('admin/validation/user.new_password.min'),
            'new_password.max' => __('admin/validation/user.new_password.max'),
            'new_password.regex' => __('admin/validation/user.new_password.regex'),
            'new_password.different' => __('admin/validation/user.new_password.different'),
            'new_password.confirmed' => __('admin/validation/user.new_password.confirmed'),
            'new_password_confirmation.required' => __('admin/validation/user.new_password_confirmation.required'),
            'token.required' => __('admin/validation/user.token.required'),
            'first_name.required' => __('admin/validation/user.first_name.required'),
            'first_name.string' => __('admin/validation/user.first_name.string'),
            'first_name.min' => __('admin/validation/user.first_name.min'),
            'first_name.max' => __('admin/validation/user.first_name.max'),
            'first_name.regex' => __('admin/validation/user.first_name.regex'),
            'last_name.required' => __('admin/validation/user.last_name.required'),
            'last_name.string' => __('admin/validation/user.last_name.string'),
            'last_name.min' => __('admin/validation/user.last_name.min'),
            'last_name.max' => __('admin/validation/user.last_name.max'),
            'last_name.regex' => __('admin/validation/user.last_name.regex'),
            'avatar.mimes' => __('admin/validation/user.avatar.mimes'),
            'avatar.max' => __('admin/validation/user.avatar.filesize'),
        ];
    }

    public function latestAvatar()
    {
        return $this->hasOne('app\Model\Admin\UserAvatars');
    }

    public function deliveryPerson()
    {
        return $this->belongsTo('app\Model\Admin\DeliveryPersons');
    }

    public function restaurant()
    {
        return $this->hasOne('app\Model\Admin\Restaurant', 'owner_id');
    }


    public function blockedUser()
    {
        return $this->hasOne('app\Model\Api\BlockedUser', 'user_id');
    }

    public function role()
    {
        return $this->belongsTo('app\Model\Admin\Role')->latest();
    }

    public function avatars()
    {
        return $this->hasMany('app\Model\Admin\UserAvatars');
    }

    public function getFullNameAttribute()
    {

        return $this->first_name . ' ' . $this->last_name;
    }

    public function deliveryP()
    {
        return $this->belongsTo('app\Model\Admin\DeliveryPersons');
    }



    public function driverlist($request)
    {

        $page = $request->query('page', 1);
        $limit = $request->query('limit', config('admin.constants.DEFAULT_PAGE_LIMIT'));
        /*$this->const['page'] = $page;*/
        $limit = $request->limit;
        $search = !empty($request->search) ? $request->search : '';
        $query = User::select(
            'users.id',
            'users.user_name',
            'users.first_name',
            'users.last_name',
            'users.email',
            'users.status',
            'users.created_at'
        )
            ->where('users.role_id', config('admin.constants.DELIVERY_PERSON_ROLE'))
            ->with('deliveryPerson');
        if (!empty($search)) {

            $query->where(function ($q) use ($search) {
                $q->Where('users.email', 'LIKE', "%$search%");
                $q->orWhere('users.first_name', 'like', "%{$search}%");
                $q->orWhere('users.last_name', 'like', "%{$search}%");
            });
        }


        $users = $query->paginate($limit);
        return $users;
    }



    /**
     * my functions
     */
    public static function upload_avatar($file = [], $user_id = null)
    {

        if (empty($file) || empty($user_id)) {
            return false;
        }

        $uploaded = false;
        $uploadDir = config('admin.path.UPLOAD_DIR');
        $avatarDir = config('admin.path.avatar.DIR');
        //concat image name with prefixes
        $resizedPrefix = config('admin.path.avatar.RESIZE_PREFIX');
        $thumbPrefix = config('admin.path.avatar.THUMB_PREFIX');

        // getting image extension
        $ext = $file->getClientOriginalExtension();
        if (empty($ext)) {
            $mime = File::mimeType($file);
            if ($mime == 'image/jpeg') {
                $ext = 'jpeg';
            }
            if ($mime == 'image/png') {
                $ext = 'png';
            }
            if ($mime == 'image/gif') {
                $ext = 'gif';
            }
            if ($mime == 'image/svg+xml') {
                $ext = 'svg';
            }
        }
        //rename image
        $fileName = uniqid() . '.' . $ext;
        $imageObj = Image::make($file->getRealPath());
        //identify greater dimension
        $initialWidth = $imageObj->width();
        $initialHeight = $imageObj->height();

        $aspectByWidth = true;
        $aspectByHeight = false;

        if ($initialWidth < $initialHeight) {
            $aspectByHeight = true;
            $aspectByWidth = false;
        }
        // check if image uploaded on aws s3
        if (env('S3_ENABLED') == true) {

            //unlink old image
            $fileDir = $uploadDir . '/' . $avatarDir . '/' . $user_id . '/';
            \app\Helpers\CommonHelper::unlinkOldImage('users', $user_id, $fileDir);

            //Original image
            $originalFile = $uploadDir . '/' . $avatarDir . '/' . $user_id . '/' . $fileName;

            $uploaded = \Storage::disk('s3')->put(
                $originalFile,
                file_get_contents($file),
                'public'
            );

            //resized image
            $fileResize = config('admin.path.avatar.RESIZE');

            if ($aspectByWidth) {
                $resizeWidth = config('admin.path.avatar.RESIZE_WIDTH');
                $resizeHeight = null;
            }
            if ($aspectByHeight) {
                $resizeHeight = config('admin.path.avatar.RESIZE_HEIGHT');
                $resizeWidth = null;
            }
            $imageObj->resize($resizeWidth, $resizeHeight, function ($constraint) {
                $constraint->aspectRatio();
            });

            $rszResource = $imageObj->stream()->detach();

            $rszFile = $uploadDir . '/' . $avatarDir . '/' . $user_id . '/rsz_' . $fileName;

            $uploaded = \Storage::disk('s3')->put(
                $rszFile,
                $rszResource,
                'public'
            );

            //thumb image
            $isThumb = config('admin.path.avatar.THUMB');

            if ($aspectByWidth) {
                $thumbWidth = config('admin.path.avatar.THUMB_WIDTH');
                $thumbHeight = null;
            }
            if ($aspectByHeight) {
                $thumbHeight = config('admin.path.avatar.THUMB_HEIGHT');
                $thumbWidth = null;
            }
            $imageObj->resize($thumbWidth, $thumbHeight, function ($constraint) {
                $constraint->aspectRatio();
            });

            $thumbResource = $imageObj->stream()->detach();

            $thumbFile = $uploadDir . '/' . $avatarDir . '/' . $user_id . '/tmb_' . $fileName;

            $uploaded = \Storage::disk('s3')->put(
                $thumbFile,
                $thumbResource,
                'public'
            );
            $originalUrl = self::s3Uri($originalFile);
        } else {
            $basePath = storage_path('app/public') . '/' . $uploadDir . '/' . $avatarDir . '/' . $user_id . '/';
            // create directory
            if (!File::exists($basePath)) {
                File::makeDirectory($basePath, 0755, true);
            }
            //origional image
            $uploaded = $file->move($basePath, $fileName);
            //resized image
            $fileResize = config('admin.path.avatar.RESIZE');
            if ($fileResize && $uploaded) {
                if ($aspectByWidth) {
                    $resizeWidth = config('admin.path.avatar.RESIZE_WIDTH');
                    $resizeHeight = null;
                }
                if ($aspectByHeight) {
                    $resizeHeight = config('admin.path.avatar.RESIZE_HEIGHT');
                    $resizeWidth = null;
                }
                $imageObj->resize($resizeWidth, $resizeHeight, function ($constraint) {
                    $constraint->aspectRatio();
                })->save($basePath . $resizedPrefix . '_' . $fileName, 100);
            }
            //thumb image
            $isThumb = config('admin.path.avatar.THUMB');
            if ($isThumb && $uploaded) {
                if ($aspectByWidth) {
                    $thumbWidth = config('admin.path.avatar.THUMB_WIDTH');
                    $thumbHeight = null;
                }
                if ($aspectByHeight) {
                    $thumbHeight = config('admin.path.avatar.THUMB_HEIGHT');
                    $thumbWidth = null;
                }
                $imageObj->resize($thumbWidth, $thumbHeight, function ($constraint) {
                    $constraint->aspectRatio();
                })->save($basePath . $thumbPrefix . '_' . $fileName, 100);
            }
            $originalUrl = url('/storage/app/public') . '/' . $uploadDir . '/' . $avatarDir . '/' . $user_id . '/' . $fileName;
        }

        //return response
        if ($uploaded) {
            if (UserAvatars::updateOrCreate([
                'user_id' => $user_id,
            ], [
                'avatar' => $originalUrl,
                'width' => $initialWidth,
                'height' => $initialHeight,
            ])) {
                return $originalUrl;
            }
        }
        return false;
    }


    public static function upload_profile_image($file = [], $user_id = null)
    {

        if (empty($file) || empty($user_id)) {
            return false;
        }

        $uploaded = false;
        $uploadDir = config('admin.path.UPLOAD_DIR');
        $avatarDir = config('admin.path.avatar.DIR');
        //concat image name with prefixes
        $resizedPrefix = config('admin.path.avatar.RESIZE_PREFIX');
        $thumbPrefix = config('admin.path.avatar.THUMB_PREFIX');

        // getting image extension
        $ext = $file->getClientOriginalExtension();
        if (empty($ext)) {
            $mime = File::mimeType($file);
            if ($mime == 'image/jpeg') {
                $ext = 'jpeg';
            }
            if ($mime == 'image/png') {
                $ext = 'png';
            }
            if ($mime == 'image/gif') {
                $ext = 'gif';
            }
            if ($mime == 'image/svg+xml') {
                $ext = 'svg';
            }
        }
        //rename image
        $fileName = uniqid() . '.' . $ext;
        $imageObj = Image::make($file->getRealPath());
        //identify greater dimension
        $initialWidth = $imageObj->width();
        $initialHeight = $imageObj->height();

        $aspectByWidth = true;
        $aspectByHeight = false;

        if ($initialWidth < $initialHeight) {
            $aspectByHeight = true;
            $aspectByWidth = false;
        }
        // check if image uploaded on aws s3
        if (env('S3_ENABLED') == true) {

            //unlink old image
            $fileDir = $uploadDir . '/' . $avatarDir . '/' . $user_id . '/';
            \app\Helpers\CommonHelper::unlinkOldImage('users', $user_id, $fileDir);

            //Original image
            $originalFile = $uploadDir . '/' . $avatarDir . '/' . $user_id . '/' . $fileName;

            $uploaded = \Storage::disk('s3')->put(
                $originalFile,
                file_get_contents($file),
                'public'
            );

            //resized image
            $fileResize = config('admin.path.avatar.RESIZE');

            if ($aspectByWidth) {
                $resizeWidth = config('admin.path.avatar.RESIZE_WIDTH');
                $resizeHeight = null;
            }
            if ($aspectByHeight) {
                $resizeHeight = config('admin.path.avatar.RESIZE_HEIGHT');
                $resizeWidth = null;
            }
            $imageObj->resize($resizeWidth, $resizeHeight, function ($constraint) {
                $constraint->aspectRatio();
            });

            $rszResource = $imageObj->stream()->detach();

            $rszFile = $uploadDir . '/' . $avatarDir . '/' . $user_id . '/rsz_' . $fileName;

            $uploaded = \Storage::disk('s3')->put(
                $rszFile,
                $rszResource,
                'public'
            );

            //thumb image
            $isThumb = config('admin.path.avatar.THUMB');

            if ($aspectByWidth) {
                $thumbWidth = config('admin.path.avatar.THUMB_WIDTH');
                $thumbHeight = null;
            }
            if ($aspectByHeight) {
                $thumbHeight = config('admin.path.avatar.THUMB_HEIGHT');
                $thumbWidth = null;
            }
            $imageObj->resize($thumbWidth, $thumbHeight, function ($constraint) {
                $constraint->aspectRatio();
            });

            $thumbResource = $imageObj->stream()->detach();

            $thumbFile = $uploadDir . '/' . $avatarDir . '/' . $user_id . '/tmb_' . $fileName;

            $uploaded = \Storage::disk('s3')->put(
                $thumbFile,
                $thumbResource,
                'public'
            );
            $originalUrl = self::s3Uri($originalFile);
        } else {
            $basePath = storage_path('app/public') . '/' . $uploadDir . '/' . $avatarDir . '/' . $user_id . '/';
            // create directory
            if (!File::exists($basePath)) {
                File::makeDirectory($basePath, 0755, true);
            }
            //origional image
            $uploaded = $file->move($basePath, $fileName);
            //resized image
            $fileResize = config('admin.path.avatar.RESIZE');
            if ($fileResize && $uploaded) {
                if ($aspectByWidth) {
                    $resizeWidth = config('admin.path.avatar.RESIZE_WIDTH');
                    $resizeHeight = null;
                }
                if ($aspectByHeight) {
                    $resizeHeight = config('admin.path.avatar.RESIZE_HEIGHT');
                    $resizeWidth = null;
                }
                $imageObj->resize($resizeWidth, $resizeHeight, function ($constraint) {
                    $constraint->aspectRatio();
                })->save($basePath . $resizedPrefix . '_' . $fileName, 100);
            }
            //thumb image
            $isThumb = config('admin.path.avatar.THUMB');
            if ($isThumb && $uploaded) {
                if ($aspectByWidth) {
                    $thumbWidth = config('admin.path.avatar.THUMB_WIDTH');
                    $thumbHeight = null;
                }
                if ($aspectByHeight) {
                    $thumbHeight = config('admin.path.avatar.THUMB_HEIGHT');
                    $thumbWidth = null;
                }
                $imageObj->resize($thumbWidth, $thumbHeight, function ($constraint) {
                    $constraint->aspectRatio();
                })->save($basePath . $thumbPrefix . '_' . $fileName, 100);
            }
            $originalUrl = url('/storage/app/public') . '/' . $uploadDir . '/' . $avatarDir . '/' . $user_id . '/' . $fileName;
        }

        //return response
        if ($uploaded) {
            if (User::where(['id' => $user_id])->update([
                'profile_image' => $originalUrl
            ])) {
                return $originalUrl;
            }
        }
        return false;
    }

    public static function generatePassword()
    {
        $alphabet = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ1234567890';
        $pass = array(); //remember to declare $pass as an array
        $alphaLength = strlen($alphabet) - 3; //put the length -1 in cache
        for ($i = 0; $i < 8; $i++) {
            $n = rand(0, $alphaLength + 1);
            $pass[] = substr(str_shuffle($alphabet[$n]), 0, 12);
        }
        return implode($pass); //turn the array into a string
    }




    public static function getAdminEmail()
    {
        $data = User::select(['email'])->where(['role_id' => config('admin.constants.SUPER_ADMIN_ROLE')])->first();
        return $data->email;
    }
}
