<?php

namespace app\Model\Admin;

use Illuminate\Database\Eloquent\Model;
use app\Traits\FilePathUrl;
use File;
use Image;
use Kyslik\ColumnSortable\Sortable;

class Restaurant_menu extends Model
{
    use FilePathUrl;
    use Sortable;
    protected $tables = 'restaurant_menus';
    protected $hidden = ['created_at', 'updated_at'];
    protected $fillable = ['restaurant_id', 'food_category_id', 'name', 'image', 'price', 'prep_time', 'allergy_advice', 'description', 'status'];
    protected $guarded = [];


    public static function rules($id = null)
    {
        return [
            'food_category_id' => 'bail|required|exists:food_categories,id',

            'name' => 'bail|required|string|min:' . config('admin.constants.restaurant-menu.name.minLength') . '|max:' . config('admin.constants.restaurant-menu.name.maxLength'),

            'price' => 'bail|required|string|min:' . config('admin.constants.restaurant-menu.name.minLength') . '|max:' . config('admin.constants.restaurant-menu.name.maxLength'),

            'prep_time' => 'bail|required|string|min:' . config('admin.constants.restaurant-menu.prep_time.minLength') . '|max:' . config('admin.constants.delivery-person.age.maxLength'),

            'image' => 'bail|nullable|mimes:' . config('admin.constants.user.doc.mimes') . '|max:' . config('admin.constants.restaurant-menu.image.size'), //formats: jpeg, png, bmp, gif, svg 

            'description' => 'bail|string|min:' . config('admin.constants.restaurant-menu.description.minLength') . '|max:' . config('admin.constants.restaurant-menu.description.maxLength'),
        ];
    }


    public static function messages()
    {
        return [
            'food_category_id.required' => __('admin/validation/user.role_id.required'),
            'name.required' => __('admin/validation/restaurant_menu.name.required'),
            'name.string' => __('admin/validation/restaurant_menu.name.string'),
            'name.min' => __('admin/validation/restaurant_menu.name.min'),
            'name.max' => __('admin/validation/restaurant_menu.name.max'),
            'name.regex' => __('admin/validation/restaurant_menu.name.regex'),
            'prep_time.required' => __('admin/validation/user.email.required'),
            'prep_time.string' => __('admin/validation/user.email.max'),
            'prep_time.email' => __('admin/validation/user.email.min'),
            'prep_time.max' => __('admin/validation/user.email.max'),
            'prep_time.unique' => __('admin/validation/user.email.unique'),
            'prep_time.exists' => __('admin/validation/user.email.exists'),
            'mobile_no.required' => __('admin/validation/user.mobile_no.required'),
            'mobile_no.min' => __('admin/validation/user.mobile_no.min'),
            'mobile_no.max' => __('admin/validation/user.mobile_no.max'),
            'password.required' => __('admin/validation/user.password.required'),
            'password.string' => __('admin/validation/user.password.string'),
            'password.min' => __('admin/validation/user.password.min'),
            'password.max' => __('admin/validation/user.password.max'),
            'password.regex' => __('admin/validation/user.password.regex'),
            'new_password.required' => __('admin/validation/user.new_password.required'),
            'new_password.string' => __('admin/validation/user.new_password.string'),
            'new_password.min' => __('admin/validation/user.new_password.min'),
            'new_password.max' => __('admin/validation/user.new_password.max'),
            'new_password.regex' => __('admin/validation/user.new_password.regex'),
            'new_password.different' => __('admin/validation/user.new_password.different'),
            'new_password.confirmed' => __('admin/validation/user.new_password.confirmed'),
            'new_password_confirmation.required' => __('admin/validation/user.new_password_confirmation.required'),
            'token.required' => __('admin/validation/user.token.required'),
            'first_name.required' => __('admin/validation/user.first_name.required'),
            'first_name.string' => __('admin/validation/user.first_name.string'),
            'first_name.min' => __('admin/validation/user.first_name.min'),
            'first_name.max' => __('admin/validation/user.first_name.max'),
            'first_name.regex' => __('admin/validation/user.first_name.regex'),
            'last_name.required' => __('admin/validation/user.last_name.required'),
            'last_name.string' => __('admin/validation/user.last_name.string'),
            'last_name.min' => __('admin/validation/user.last_name.min'),
            'last_name.max' => __('admin/validation/user.last_name.max'),
            'last_name.regex' => __('admin/validation/user.last_name.regex'),

            'age.required' => __('admin/validation/delivery-person.age.required'),
            'age.min' => __('admin/validation/delivery-person.age.min'),
            'age.max' => __('admin/validation/delivery-person.age.max'),
            'age.regex' => __('admin/validation/delivery-person.age.regex'),

            'account_details.required' => __('admin/validation/delivery-person.account_details.required'),
            'account_details.min' => __('admin/validation/delivery-person.account_details.min'),
            'account_details.regex' => __('admin/validation/delivery-person.account_details.regex'),

            'id_proof.mimes' => __('admin/validation/user.avatar.mimes'),
            'id_proof.max' => __('admin/validation/user.avatar.filesize'),

            'vehicle_info.mimes' => __('admin/validation/user.avatar.mimes'),
            'vehicle_info.max' => __('admin/validation/user.avatar.filesize'),

            'driving_license.mimes' => __('admin/validation/user.avatar.mimes'),
            'driving_license.max' => __('admin/validation/user.avatar.filesize'),


            'avatar.mimes' => __('admin/validation/user.avatar.mimes'),
            'avatar.max' => __('admin/validation/user.avatar.filesize'),

            'image.mimes' => __('admin/validation/user.avatar.mimes'),
            'image.max' => __('admin/validation/user.avatar.filesize'),
        ];
    }

    public function user()
    {
        return $this->belongsTo('app\Model\Admin\User', 'user_id', 'id');
    }


    /**
     * records listing for Delivery Persons
     *
     * @param [type] $request
     * @return void
     */



    static function list($request)
    {
        //  dd($request->all());
        $page = $request->query('page', 1);
        $limit = $request->query('limit', config('admin.constants.DEFAULT_PAGE_LIMIT'));
        $search = !empty($request->search) ? $request->search : '';
        $status = !empty($request->status) ? $request->status : '';
        $sort = !empty($request->sort) ? $request->sort : 'created_at';
        $sortType = !empty($request->direction) ? $request->direction : 'asc';
        $query = Restaurant_menu::with('user');
        if (!empty($search)) {
            $query->where(function ($q) use ($search) {
                $q->orwhereHas('user', function ($q) use ($search) {
                    return $q->where('email', 'like', '%' . $search . '%');
                });

                $q->orwhereHas('user', function ($q) use ($search) {
                    return $q->where('phone_number', 'like', '%' . $search . '%');
                });
            });
        }
        if (!empty($status)) {

            $query->where(function ($q) use ($status) {
                $q->orwhereHas('user', function ($q) use ($status) {
                    return $q->where('status', $status);
                });
            });
        }

        $query->orderBy('id', 'DESC');

        $records = $query->paginate($limit);
        return $records;
    }



    public static function upload_doc($file = [], $user_id = null, $f = null)
    {
        if (empty($file) || empty($user_id)) {
            return false;
        }

        $uploaded = false;
        $uploadDir = config('admin.path.UPLOAD_DIR');
        $avatarDir = config('admin.path.avatar.RESTAURANT_ITEM_DIR');
        //concat image name with prefixes
        $resizedPrefix = config('admin.path.avatar.RESIZE_PREFIX');
        $thumbPrefix = config('admin.path.avatar.THUMB_PREFIX');

        // getting image extension
        $ext = $file->getClientOriginalExtension();
        if (empty($ext)) {
            $mime = File::mimeType($file);
            if ($mime == 'image/jpeg') {
                $ext = 'jpeg';
            }
            if ($mime == 'image/png') {
                $ext = 'png';
            }
            if ($mime == 'image/gif') {
                $ext = 'gif';
            }
            if ($mime == 'image/svg+xml') {
                $ext = 'svg';
            }
            if ($mime == 'application/pdf') {
                $ext = 'pdf';
            }
        }
        //rename image
        $fileName = uniqid() . '.' . $ext;
        if ($ext == 'pdf') {
            $imageObj = $file->getRealPath();
            $aspectByWidth = true;
            $aspectByHeight = false;
        } else {
            // die('else');
            $aspectByWidth = true;
            $aspectByHeight = false;
            $imageObj = Image::make($file->getRealPath());
            //identify greater dimension
            $initialWidth = $imageObj->width();
            $initialHeight = $imageObj->height();
            // die('sdf');
            if ($initialWidth < $initialHeight) {
                $aspectByHeight = true;
                $aspectByWidth = false;
            }
        }

        // dd($aspectByWidth);
        // check if image uploaded on aws s3
        if (env('S3_ENABLED') == true) {

            //unlink old image
            $fileDir = $uploadDir . '/' . $avatarDir . '/' . $user_id . '/';
            \app\Helpers\CommonHelper::unlinkOldImage('restaurant_menus', $user_id, $fileDir);

            //Original image
            $originalFile = $uploadDir . '/' . $avatarDir . '/' . $user_id . '/' . $fileName;

            $uploaded = \Storage::disk('s3')->put(
                $originalFile,
                file_get_contents($file),
                'public'
            );

            //resized image
            $fileResize = config('admin.path.avatar.RESIZE');

            if ($aspectByWidth) {
                $resizeWidth = config('admin.path.avatar.RESIZE_WIDTH');
                $resizeHeight = null;
            }
            if ($aspectByHeight) {
                $resizeHeight = config('admin.path.avatar.RESIZE_HEIGHT');
                $resizeWidth = null;
            }
            $imageObj->resize($resizeWidth, $resizeHeight, function ($constraint) {
                $constraint->aspectRatio();
            });

            $rszResource = $imageObj->stream()->detach();

            $rszFile = $uploadDir . '/' . $avatarDir . '/' . $user_id . '/rsz_' . $fileName;

            $uploaded = \Storage::disk('s3')->put(
                $rszFile,
                $rszResource,
                'public'
            );

            //thumb image
            $isThumb = config('admin.path.avatar.THUMB');

            if ($aspectByWidth) {
                $thumbWidth = config('admin.path.avatar.THUMB_WIDTH');
                $thumbHeight = null;
            }
            if ($aspectByHeight) {
                $thumbHeight = config('admin.path.avatar.THUMB_HEIGHT');
                $thumbWidth = null;
            }
            $imageObj->resize($thumbWidth, $thumbHeight, function ($constraint) {
                $constraint->aspectRatio();
            });

            $thumbResource = $imageObj->stream()->detach();

            $thumbFile = $uploadDir . '/' . $avatarDir . '/' . $user_id . '/tmb_' . $fileName;

            $uploaded = \Storage::disk('s3')->put(
                $thumbFile,
                $thumbResource,
                'public'
            );

            $originalUrl = self::s3Uri($originalFile);
        } else {
            $basePath = storage_path('app/public') . '/' . $uploadDir . '/' . $avatarDir . '/' . $user_id . '/';
            // create directory
            if (!File::exists($basePath)) {
                File::makeDirectory($basePath, 0755, true);
            }
            //origional image
            $uploaded = $file->move($basePath, $fileName);
            //resized image
            $fileResize = config('admin.path.avatar.RESIZE');
            if ($fileResize && $uploaded) {
                if ($aspectByWidth) {
                    $resizeWidth = config('admin.path.avatar.RESIZE_WIDTH');
                    $resizeHeight = null;
                }
                if ($aspectByHeight) {
                    $resizeHeight = config('admin.path.avatar.RESIZE_HEIGHT');
                    $resizeWidth = null;
                }
                if ($ext != 'pdf') {
                    $imageObj->resize($resizeWidth, $resizeHeight, function ($constraint) {
                        $constraint->aspectRatio();
                    })->save($basePath . $resizedPrefix . '_' . $fileName, 100);
                }
            }
            //thumb image
            $isThumb = config('admin.path.avatar.THUMB');
            if ($isThumb && $uploaded) {
                if ($aspectByWidth) {
                    $thumbWidth = config('admin.path.avatar.THUMB_WIDTH');
                    $thumbHeight = null;
                }
                if ($aspectByHeight) {
                    $thumbHeight = config('admin.path.avatar.THUMB_HEIGHT');
                    $thumbWidth = null;
                }
                if ($ext != 'pdf') {
                    $imageObj->resize($resizeWidth, $resizeHeight, function ($constraint) {
                        $constraint->aspectRatio();
                    })->save($basePath . $resizedPrefix . '_' . $fileName, 100);
                }
            }
            $originalUrl = url('/storage/app/public') . '/' . $uploadDir . '/' . $avatarDir . '/' . $user_id . '/' . $fileName;
        }

        //return response
        if ($uploaded) {
            $doc = Restaurant_menu::find($user_id);
            $doc->$f = $originalUrl;
            $doc->save();
            return true;
        }
        return false;
    }
}