<?php

namespace app\Model\Admin;

use Illuminate\Database\Eloquent\Model;
use app\Traits\FilePathUrl;
use File;
use Image;
use Kyslik\ColumnSortable\Sortable;
use Illuminate\Support\Facades\Storage;

class FoodCategory extends Model
{
    use FilePathUrl;
    use Sortable;
    protected $tables = 'food_categories';
    protected $fillable = ['name', 'status'];

    public static function rules($id = null)
    {
        return [
            'category' => 'bail|required|string|unique:food_categories,name,' . ($id ? $id : ''),

        ];
    }

    public static function messages()
    {
        return [
            'category.required' => __('admin/validation/category.category.required'),
            'category.string' => __('admin/validation/category.category.string'),
            'category.unique' => __('admin/validation/category.category.unique'),
        ];
    }
 

    function category($request)
    {

        $page = $request->query('page', 1);
        $limit = $request->query('limit', config('admin.constants.DEFAULT_PAGE_LIMIT'));
        $limit = $request->limit;
        $search = !empty($request->search) ? $request->search : '';
        $query = FoodCategory::select('id', 'name', 'image', 'status','isFav');
        if (!empty($search)) {

            $query->where(function ($q) use ($search) {
                $q->orWhere('name', 'LIKE', "%$search%");
            });
        }
        $query->orderBy('id', 'DESC');
        $users = $query->paginate($limit);
        return $users;
    }

    /** 
     * Author : jaidev
     * Email :  jaidev@nmgtechnologies.com
     * Method : upload_doc
     * Desc : save food category image
     * Create : 3-Aug-2021
     */

    public static function upload_doc($file = [], $user_id = null, $f = null)
    {

        if (empty($file) || empty($user_id)) {
            return false;
        }

        $uploaded = false;
        $uploadDir = config('admin.path.UPLOAD_DIR');
        $avatarDir = config('admin.path.avatar.FOOD_CATEGORY_DOC_DIR');
        //concat image name with prefixes
        $resizedPrefix = config('admin.path.avatar.RESIZE_PREFIX');
        $thumbPrefix = config('admin.path.avatar.THUMB_PREFIX');

        // getting image extension
        $ext = $file->getClientOriginalExtension();
        if (empty($ext)) {
            $mime = File::mimeType($file);
            if ($mime == 'image/jpeg') {
                $ext = 'jpeg';
            }
            if ($mime == 'image/png') {
                $ext = 'png';
            }
            if ($mime == 'image/gif') {
                $ext = 'gif';
            }
            if ($mime == 'image/svg+xml') {
                $ext = 'svg';
            }
            if ($mime == 'application/pdf') {
                $ext = 'pdf';
            }
        }
        //rename image
        $fileName = uniqid() . '.' . $ext;
        if ($ext == 'pdf') {
            $imageObj = $file->getRealPath();
            $aspectByWidth = true;
            $aspectByHeight = false;
        } else {
            $aspectByWidth = true;
            $aspectByHeight = false;
            $imageObj = Image::make($file->getRealPath());
            //identify greater dimension
            $initialWidth = $imageObj->width();
            $initialHeight = $imageObj->height();
            // die('sdf');
            if ($initialWidth < $initialHeight) {
                $aspectByHeight = true;
                $aspectByWidth = false;
            }
        }

        // dd($aspectByWidth);
        // check if image uploaded on aws s3
        if (env('S3_ENABLED') == true) {

            //unlink old image
            $fileDir = $uploadDir . '/' . $avatarDir . '/' . $user_id . '/';
            \app\Helpers\CommonHelper::unlinkOldImage('food_categories', $user_id, $fileDir);

            //Original image
            $originalFile = $uploadDir . '/' . $avatarDir . '/' . $user_id . '/' . $fileName;

            $uploaded = \Storage::disk('s3')->put(
                $originalFile,
                file_get_contents($file),
                'public'
            );

            $originalUrl = self::s3Uri($originalFile);
        } else {
            $fileDir = $uploadDir . '/' . $avatarDir . '/' . $user_id . '/';
            $basePath = storage_path('app/public') . '/' . $uploadDir . '/' . $avatarDir . '/' . $user_id . '/';

            //unlink old image
            File::deleteDirectory($basePath);

            // create directory
            if (!File::exists($basePath)) {
                File::makeDirectory($basePath, 0755, true);
            }
            //origional image
            $uploaded = $file->move($basePath, $fileName);
            //resized image
            $fileResize = config('admin.path.avatar.RESIZE');
            if ($fileResize && $uploaded) {
                if ($aspectByWidth) {
                    $resizeWidth = config('admin.path.avatar.RESIZE_WIDTH');
                    $resizeHeight = null;
                }
                if ($aspectByHeight) {
                    $resizeHeight = config('admin.path.avatar.RESIZE_HEIGHT');
                    $resizeWidth = null;
                }
                if ($ext != 'pdf') {
                    $imageObj->resize($resizeWidth, $resizeHeight, function ($constraint) {
                        $constraint->aspectRatio();
                    })->save($basePath . $resizedPrefix . '_' . $fileName, 100);
                }
            }
            //thumb image
            $isThumb = config('admin.path.avatar.THUMB');
            if ($isThumb && $uploaded) {
                if ($aspectByWidth) {
                    $thumbWidth = config('admin.path.avatar.THUMB_WIDTH');
                    $thumbHeight = null;
                }
                if ($aspectByHeight) {
                    $thumbHeight = config('admin.path.avatar.THUMB_HEIGHT');
                    $thumbWidth = null;
                }
                if ($ext != 'pdf') {
                    $imageObj->resize($resizeWidth, $resizeHeight, function ($constraint) {
                        $constraint->aspectRatio();
                    })->save($basePath . $resizedPrefix . '_' . $fileName, 100);
                }
            }
            $originalUrl = url('/storage/app/public') . '/' . $uploadDir . '/' . $avatarDir . '/' . $user_id . '/' . $fileName;
        }

        //return response
        if ($uploaded) {

            $food_category = FoodCategory::find($user_id);
            $food_category->$f = $originalUrl;
            $food_category->save();
            // dd($originalUrl);
            return true;
        }
        return false;
    }
}
