<?php

namespace app\Http\Controllers\Vendor;

use app\Http\Controllers\Admin\BaseController;
use Illuminate\Http\Request;
use app\Http\Controllers\Controller;
use app\Model\Admin\Role;
use app\Model\Admin\User;
use app\Model\Admin\UserAvatars;
use app\Model\Admin\GlobalSetting;
use app\Model\Vendor\Order;
use app\Model\Vendor\Restaurant_menu;
use DB;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Input;
use Validator;
use Auth;
use Session;
use app\Components\Api\V1\PushNotification;
use app\Model\Admin\Restaurant;

class VendorController extends BaseController
{
    private $url;
    private $title;
    private $breadCrumb;

    /**
     * @Method: __construct()
     * @Scope: public
     * @Params: null
     * @Description: Initialize variables like @avatarDir, @roleId, @roleName, @url within private scope of current class
     * @Created 28/June/2021
     * @Updated 28/June/2021
     */
    public function __construct()
    {
        try {

            $this->middleware('auth.restaurant')->except('vendorLogin');

            $this->url = config('admin.path.VENDOR_BASE_URL');
            $this->title = 'Restaurant';

            $this->const['url'] = $this->url;
            $this->const['title'] = $this->title;

            //used in adminForm layout to load validation messges in js variable
            $this->const['objValidationMsgs'] = 'user';
            $this->const['objValidationJs'] = 'user';

            $this->const = array_merge($this->adminConst(), $this->const);

            //default breadcrumb option (Dashboard) for every route
            $this->breadCrumb['faClass'] = 'fa-table';
            $this->breadCrumb['breadCrumData'] = [
                0 => [
                    'text' => 'Dashboard',
                    'url' => config('admin.path.VENDOR_BASE_URL'),
                    'breadFaClass' => 'fa-dashboard',
                ],

            ];
            //dd($this->const);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * @Method: login()
     * @Scope: public
     * @Input: @email, @password
     * @returns: redirect vendor to vendor dashboard after successful login
     * @Description: Vendor with role - Plateform Vendor Can login vendor panel
     *  using their credentials like- email, password.
     * @Created 28/June/2021
     * @Updated 28/June/2021
     */
    public function login(Request $request)
    {
        try {
            if (Auth::guard('restaurant')->check()) {
                return \Redirect::to($this->url . '/dashboard');
            }

            // method/action specific initializations
            $this->subTitle = 'Login';
            $formId = 'loginForm';

            $this->const['subTitle'] = $this->subTitle;
            $this->const['formId'] = $formId;

            $data = array_merge($this->const);

            if ($request->isMethod('post')) {
                $input = $request->only(['email', 'password', 'remember']);
                $validator = Validator::make($input, User::loginRules(), User::messages());
                if ($validator->fails()) {
                    return \Redirect::back()->withInput()->withErrors($validator->errors());
                } else {
                    $remember = (Input::has('remember')) ? true : false;
                    $userData = User::where('email', $request->email)->first();
                    if ($userData->status == '0' && isset($userData->verification_otp)) {
                        $plainText = $this->generatePIN(4);
                        $userData->status        = false;
                        $userData->verification_otp  = $plainText;
                        $userData->verification_otp_sent_at = Carbon::now();
                        $userData->save();

                        ########  Restaurant Registration Otp ###########
                        if (!empty($input['lang'])) {
                            \App::setLocale($input['lang']);
                        }
                        $params['replaceKeywords']['{USER_NAME}'] = $userData->user_name;
                        $params['replaceKeywords']['{LOGIN_ID}'] = $userData->email;
                        $params['replaceKeywords']['{OTP}'] = $plainText;
                        $params['toEmail'] = $userData->email;
                        $params['emailSlug'] = 'restaurant_registration_otp';
                        $this->customMailer($params);

                        $request->session()->put('restaurantEmail', $request->email);
                        $request->session()->put('restaurantPassword', $request->password);

                        \Session::flash('success', \Config::get('flash_msg.OTPSENT'));
                        return redirect('securerestaurant/verification-code');
                    } else if ($userData->status == '0') {
                        return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.accountBlocked'));
                    } else {
                        if (Auth::guard('restaurant')->attempt(['email' => $request->email, 'password' => $request->password, 'role_id' => [2]], true)) {
                            return \Redirect::to($this->url . '/dashboard');
                        } else {
                            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.InavlidLoginCredentials'));
                            //dd($request->all());
                        }
                    }
                }
            }
            return view('Vendor/vendor/login')->with($data);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * @Method: signUp()
     * @returns: 
     * @Description: Vendor sign up panel.
     * @Created 29/June/2021
     * @Updated 29/June/2021
     */
    public function signUp(Request $request)
    {

        try {
            if (Auth::guard('restaurant')->check()) {
                return \Redirect::to($this->url . '/dashboard');
            }
            // method/action specific initializations
            $this->subTitle = 'Sign up';
            $formId = 'signup';

            $this->const['subTitle'] = $this->subTitle;
            $this->const['formId'] = $formId;

            $data = array_merge($this->const);
            if ($request->isMethod('post')) {
                //echo "<pre>@@@"; print_r($input = request()->all()); exit;
                $input = $request->only(['name', 'bio', 'local', 'continental', 'groceries', 'email', 'phone_number', 'address1', 'address2', 'locality', 'country', 'postcode', 'password', 'latitude', 'longitude', 'status']);
                $validator = Validator::make($input, Restaurant::signupRules(), Restaurant::signupMessages());
                if ($validator->fails()) {
                    return \Redirect::back()->withInput()->withErrors($validator->errors());
                } else {
                    $rollBack = false;
                    \DB::beginTransaction();
                    $restTypes = [];
                    if (isset($input['local'])) {
                        $local = 1;
                        $restTypes[] = 'Local';
                    } else {
                        $local = 0;
                    }

                    if (isset($input['continental'])) {
                        $continental = 1;
                        $restTypes[] = 'Continental';
                    } else {
                        $continental = 0;
                    }

                    if (isset($input['groceries'])) {
                        $groceries = 1;
                        $restTypes[] = 'Groceries';
                    } else {
                        $groceries = 0;
                    }

                    $plainText = $this->generatePIN(4);

                    $userData = [
                        'role_id'    => config('admin.constants.RESTAURANT_OWNER_ROLE'),
                        'user_name'  => $input['name'],
                        'email'      => $input['email'],
                        'phone_number' => $input['phone_number'],
                        'password' => bcrypt($input['password']),
                        //'is_verified' => false,
                        'status'        => false,
                        'verification_otp'  => $plainText,
                        'verification_otp_sent_at' => Carbon::now(),
                    ];

                    $user = User::create($userData);

                    $restaurantData = [
                        'owner_id'    => $user->id,
                        'name'        => $input['name'],
                        'bio'         => $input['bio'],
                        'resturent_type_local' => $local,
                        'resturent_type_continental' => $continental,
                        'resturent_type_groceries' => $groceries,
                        'address1'    => $input['address1'],
                        'address2'    => $input['address2'],
                        'city'        => $input['locality'],
                        'region'      => $input['country'],
                        'zipcode'     => $input['postcode'],
                        'latitude'    => $input['latitude'],
                        'longitude'   => $input['longitude'],
                        'status'      => false,
                    ];
                    $result = Restaurant::create($restaurantData);
                    if ($result) {
                        ########  Restaurant Registration Otp ###########
                        if (!empty($input['lang'])) {
                            \App::setLocale($input['lang']);
                        }
                        $params['replaceKeywords']['{USER_NAME}'] = $user->user_name;
                        $params['replaceKeywords']['{LOGIN_ID}'] = $user->email;
                        $params['replaceKeywords']['{OTP}'] = $plainText;
                        $params['toEmail'] = $user->email;
                        $params['emailSlug'] = 'customer_restaurant_emailid_varification_otp';

                        if (!$this->customMailer($params)) {
                            $rollBack = true;
                        }


                        $params = [];
                        ########  Restaurant Registration Email to admin ###########
                        // get administator email id
                        $globalSetting = GlobalSetting::where('slug', 'administartor-mail')->first();
                        if (!empty($input['lang'])) {
                            \App::setLocale($input['lang']);
                        }
                        $params['replaceKeywords']['{USER_NAME}'] = 'Admin';
                        $params['replaceKeywords']['{RESTAURANT_NAME}'] = $input['name'];
                        $params['replaceKeywords']['{RESTAURANT_BIO}'] = $input['bio'];
                        $params['replaceKeywords']['{RESTAURANT_TYPES}'] = implode(',', $restTypes);
                        $params['replaceKeywords']['{EMAIL}'] = $input['email'];
                        $params['replaceKeywords']['{PHONE_NUMBER}'] = $input['phone_number'];
                        $params['replaceKeywords']['{ADDRESS}'] = $input['address1'];
                        $params['toEmail'] = $globalSetting->value;
                        $params['emailSlug'] = 'restaurant_signup_details_to_admin';
                        $this->customMailer($params);
                    }
                    if (!$rollBack) {
                        \DB::commit();
                        $request->session()->put('restaurantEmail', $user->email);
                        $request->session()->put('restaurantPassword', $input['password']);

                        // \Session::flash('success', \Config::get('flash_msg.OTPSENT'));
                        return redirect('securerestaurant/verification-code');
                    } else {
                        \DB::rollBack();
                        return \Redirect::back()->withInput()->with('error', __('admin/validation/restaurant.error.reset'));
                    }
                }
            }
            return view('Vendor/vendor/register')->with($data);
        } catch (\Exception $e) {
            dd($e);
            \DB::rollBack();
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.EmailWrong'));
        }
    }

    /**
     * @Method: verificationCode()
     * @returns: 
     * @Description: Verification code sent to your registered email Id.
     * @Created 05/July/2021
     * @Updated 05/July/2021
     */

    public function verificationCode(Request $request)
    {
        try {
            if (Auth::guard('restaurant')->check()) {
                return \Redirect::to($this->url . '/dashboard');
            }
            // method/action specific initializations
            $this->subTitle = 'Verification Code';
            $formId = 'verificationCode';

            $this->const['subTitle'] = $this->subTitle;
            $this->const['formId'] = $formId;
            $data = array_merge($this->const);

            if ($request->isMethod('post')) {
                $input     = $request->only('otp');
                $input['otp'] = implode('', $input['otp']);

                $user =   User::where('verification_otp', $input['otp'])->first();
                if (isset($user)) {
                    if (!empty($user->verification_otp == $input['otp'])) {
                        $user->status = true;
                        $user->verification_otp         = null;
                        $user->verification_otp_sent_at = null;
                        $user->save();

                        ########  	Restaurant Registration ###########
                        if (!empty($input['lang'])) {
                            \App::setLocale($input['lang']);
                        }
                        $params['replaceKeywords']['{USER_NAME}'] = $user->user_name;
                        $params['replaceKeywords']['{LOGIN_ID}'] = $user->email;
                        $params['toEmail'] = $user->email;
                        $params['emailSlug'] = 'restaurant_registration';
                        $this->customMailer($params);
                        ######## End Restaurant Registration ###########
                        \Session::flash('success', \Config::get('flash_msg.OTPSENT'));
                        \Session::flash('success-otp', \Config::get('flash_msg.OTPSENT'));
                    } else {
                        return \Redirect::back()->withInput()->with('error', __('admin/validation/restaurant.error.wrong_otp'));
                    }
                } else {
                    return \Redirect::back()->withInput()->with('error', __('admin/validation/restaurant.error.wrong_otp'));
                }
            }

            return view('Vendor/vendor/verification_code')->with($data);
        } catch (\Exception $e) {
            \DB::rollBack();
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.EmailWrong'));
        }
    }

    /**
     * @Method: sendOtp()
     * @returns: 
     * @Description: send otp code to your registered email Id.
     * @Created 09/July/2021
     * @Updated 09/July/2021
     */

    public function sendOtp(Request $request)
    {
        try {
            if (Auth::guard('restaurant')->check()) {
                return \Redirect::to($this->url . '/dashboard');
            }
            if (request()->ajax()) {

                if (request()->isMethod('get')) {

                    $user = User::where('email', $request->session()->get('restaurantEmail'))->first();
                    $otp = $this->generatePIN(4);
                    $user->verification_otp = (int)$otp;
                    $user->verification_otp_sent_at = Carbon::now();
                    $user->save();

                    // sending email to user
                    $full_name = $user->user_name;
                    $params['replaceKeywords']['{USER_NAME}'] = $full_name;
                    $params['replaceKeywords']['{OTP}'] = $otp;
                    $params['toEmail'] = $user->email;
                    $params['emailSlug'] = 'customer_restaurant_emailid_verification_resend_otp';
                    //echo "<pre>@@@"; print_r($params); exit;
                    $mail = $this->customMailer($params);
                    $status = '1';
                }
                return response()->json(['success' => $status]);
            }
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.EmailWrong'));
        }
    }
    /**
     * @Method: webRegister()
     * @returns: 
     * @Description: To check email and password to go to dashboard.
     * @Created 06/July/2021
     * @Updated 06/July/2021
     */

    public function webRegister(Request $request)
    {
        $emailId = $request->session()->get('restaurantEmail');
        $password = $request->session()->get('restaurantPassword');
        if (Auth::guard('restaurant')->attempt(['email' => $emailId, 'password' => $password, 'role_id' => [2]], true)) {
            $request->session()->forget('restaurantEmail');
            $request->session()->forget('restaurantPassword');
            return \Redirect::to($this->url . '/dashboard');
            $request->session()->forget('restaurantEmail');
            $request->session()->forget('restaurantPassword');
            return \Redirect::to($this->url . '/dashboard');
        }
        // echo "======" . $emailId . "@@" . $password;
        // exit;
        return \Redirect::to($this->url);
    }

    /**
     * @Method: forgotPassword()
     * @Scope: public
     * @Input: @email
     * @returns: send otp for reset password to registerd email inbox
     * @Description: Reset password otp has been sent to registered email inbox
     *  with an encrypted tokn expires few minutes set in vendor configuartion.
     * @Created 29/June/2021
     * @Updated 29/June/2021
     */

    public function forgotPassword(Request $request)
    {
        if (Auth::guard('restaurant')->check()) {
            return \Redirect::to($this->url . '/dashboard');
        }
        try {
            // method/action specific initializations
            $this->subTitle = 'Forgot Password';
            $formId = 'forgotPassword';

            $this->const['subTitle'] = $this->subTitle;
            $this->const['formId'] = $formId;

            $data = array_merge($this->const);
            if ($request->isMethod('post')) {
                $input = $request->only(['email']);
                $validator = Validator::make($input, User::forgotPasswordRules(), User::messages());
                if ($validator->fails()) {
                    return \Redirect::back()->withInput()->withErrors($validator->errors());
                } else {
                    // $user = User::where('email', $request->email)->where('role_id', '<>', config('admin.constants.PUBLIC_USERS_ROLE'))->select('id', 'email', 'first_name', 'last_name', 'status')->first();
                    $user = User::where('email', $request->email)->select('id', 'user_name', 'email', 'status')->first();

                    if (empty($user)) {
                        return \Redirect::back()->withInput()->withErrors(__('admin/flash_msg.error.emailNotExists'));
                    } else {

                        if ($user->status == 1) {
                            $rollBack = false;
                            \DB::beginTransaction();
                            $plainText = $this->randomPassword();
                            $user->reset_password_token = bcrypt($plainText);
                            $userId = $this->encryptDecryptId('encrypt', $user->id);
                            $email = $this->encryptDecryptId('encrypt', $user->email);

                            $plainText = $this->generatePIN(4);
                            $reset_password_token = $user->id . ':' . $user->email . ':' . Carbon::now()->timestamp;
                            $token = $this->encryptDecryptId('encrypt', $reset_password_token);
                            $user->reset_password_token = $token;
                            $user->reset_password_sent_at = Carbon::now();
                            $user->verification_otp = $plainText;
                            $user->verification_otp_sent_at = Carbon::now();

                            if ($user->save()) {
                                ###Reset Password Email#####
                                if (!empty($input['lang'])) {
                                    \App::setLocale($input['lang']);
                                }
                                $params['replaceKeywords']['{USER_NAME}'] = $user->user_name;
                                $params['replaceKeywords']['{LOGIN_ID}'] = $user->email;
                                $params['replaceKeywords']['{OTP}'] = $plainText;
                                $params['toEmail'] = $user->email;
                                // $params['emailSlug'] = 'reset_password';
                                $params['emailSlug'] = 'restaurant_forgot_reset_password';
                                if (!$this->customMailer($params)) {
                                    $rollBack = true;
                                }
                            }
                            if (!$rollBack) {
                                \DB::commit();
                                // return \Redirect::back()->withInput()->with('success', __('admin/validation/user.success.reset'));
                                // \Session::flash('success', \Config::get('flash_msg.OTPSENT'));
                                \Session::put('resetPasswordUserEmail', $user->email);
                                return redirect()->route('RestaurantResetPassword');
                            } else {
                                \DB::rollBack();
                                return \Redirect::back()->withInput()->with('error', __('admin/validation/user.error.reset'));
                            }
                        } else {
                            return \Redirect::back()->withInput()->with('error', __('admin/validation/user.error.blocked'));
                        }
                    }
                }
            }
            return view('Vendor/vendor/forgot_password')->with($data);
        } catch (\Exception $e) {
            // dd($e->getMessage());
            \DB::rollBack();
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.EmailWrong'));
        }
    }

    /**
     * @Method: restaurantResetPassword()
     * @Scope: public
     * @Input: @email, @otp, @new_password, @new_password_confirmation
     * @returns: redirect to login screen after successful operation
     * @Description: Reset password after-
     *   - validating token, check user email & user status
     * @Created 12/July/2021
     * @Updated 12/July/2021
     */
    public function restaurantResetPassword(Request $request, $token = null)
    {

        try {
            if (Auth::guard('restaurant')->check()) {
                return \Redirect::to($this->url . '/dashboard');
            }
            $this->subTitle = 'Reset Password';
            $formId = 'restaurantResetPassword';
            $this->const['subTitle'] = $this->subTitle;
            $this->const['formId'] = $formId;
            if (!empty($request->otp)) {
                $token = $request->otp;
                $user = User::where('email', $request->email)->first();
                if ($user) {
                    if (!empty($user->verification_otp == $request->otp)) {

                        $userId = $user->id;
                        $email = $user->email;
                        $otpCreatedTime = strtotime($user->verification_otp_sent_at);
                        $currentTime = strtotime(date('Y-m-d H:i:s'));
                        //Calculate the difference.
                        $difference = $currentTime - $otpCreatedTime;
                        $difference = floor($difference / (60));
                        if (empty($user)) {
                            return \Redirect::back()->withInput()->withErrors(__('admin/flash_msg.error.invalidResetToken'));
                        } elseif ($difference > \Config::get('admin.constants.RESET_TOKEN_EXPIRE')) {
                            return \Redirect::back()->withInput()->withErrors(__('admin/flash_msg.error.invalidResetToken'));
                        } else {
                            //$updateVal = array('reset_password_token' => '', 'password' => bcrypt($request->new_password));
                            $updateVal = [
                                'reset_password_sent_at' => null,
                                'reset_password_token' => null,
                                'verification_otp'     => null,
                                'verification_otp_sent_at' => null,
                                'email_verified_at' => null,
                                'password' => bcrypt($request->new_password),
                            ];
                            $pswdUpdate = \DB::table('users')->where('id', '=', $user->id)->update($updateVal);
                            if ($pswdUpdate) {
                                \Session::flash('success', __('admin/flash_msg.success.passwordReset'));
                                // return \Redirect::to(\Config::get('admin.path.VENDOR_BASE_URL'));
                                return redirect()->route('RestaurantResetPasswordSuccess');
                            } else {
                                return \Redirect::back()->withInput()->withErrors(__('admin/flash_msg.error.resetPassword'));
                            }
                        }
                    } else {

                        return \Redirect::back()->withInput()->withErrors(__('admin/flash_msg.error.wrongOTP'));
                    }
                } else {
                    return \Redirect::back()->withInput()->withErrors(__('admin/flash_msg.error.emailNotExists'));
                }
            } else {
                if (empty(session('resetPasswordUserEmail'))) {
                    return redirect()->route('forgotPassword');
                }
            }

            $data = array_merge($this->const, $this->breadCrumb);
            $data['userEmail'] =  session('resetPasswordUserEmail');
            return view('Vendor/vendor/reset_password', $data);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }



    /**
     * @Method: restaurantResetPasswordSuccess()
     * @Scope: public
     * @Input: 
     * @returns: redirect to reset password success page
     * @Description: Reset password success page
     * @Created 12/July/2021
     * @Updated 26/July/2021
     */
    public function restaurantResetPasswordSuccess(Request $request, $token = null)
    {
        if (Auth::guard('restaurant')->check()) {
            return \Redirect::to($this->url . '/dashboard');
        }
        try {
            $this->subTitle = 'Reset Password';
            $formId = 'restaurantResetPassword';
            $this->const['subTitle'] = $this->subTitle;
            $this->const['formId'] = $formId;

            $data = array_merge($this->const, $this->breadCrumb);
            $data['userEmail'] =  session('resetPasswordUserEmail');
            $data['showPopUp'] = true;
            return view('Vendor/vendor/reset_password', $data);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * @Method: dashboard()
     * @Scope: public
     * @Description: Display the recent statistics of CMS & serve all links as per user access right
     * @Created 29/June/2021
     * @Updated 29/June/2021
     */
    public function dashboard(Request $request)
    {
        try {
            // dd(Auth::guard('restaurant')->check());
            if (!Auth::guard('restaurant')->check()) {
                \Session::flash('error', __('admin/flash_msg.error.loginFirst'));
                return \Redirect::to($this->url);
            }
            //dd(Auth::guard('admin'));
            $this->title = 'Dashboard';
            $this->subTitle = 'Control panel';
            $this->const['formId'] = '';
            $this->const['title'] = $this->title;
            $this->const['subTitle'] = $this->subTitle;

            $role = Role::select('is_super_admin')->where('id', Auth::guard('restaurant')->user()->role_id)->first();
            $restaurantDetails = DB::table('restaurants')->where('owner_id', Auth::guard('restaurant')->user()->id)->first();
            $data = [];
            //echo "====".Carbon::today('Y-m-d'); exit;
            if ($role->is_super_admin) {
                $ordersCount = Order::where('restaurant_id', $restaurantDetails->id)->count();
                $menuItemsCount = Restaurant_menu::where('restaurant_id', $restaurantDetails->id)->count();


                $revenueCount = Order::select([DB::raw("SUM(sub_total - commission) as total_revenue")])->where(['restaurant_id' => $restaurantDetails->id, 'order_status_id' => config('admin.constants.ORDER_STATUS_COMPLETED')])->first();

               // $revenueCount = Order::select([DB::raw("SUM(sub_total + tax) as total_revenue")])->where(['restaurant_id' => $restaurantDetails->id, 'order_status_id' => config('admin.constants.ORDER_STATUS_COMPLETED')])->first();



                $orderDetails = Order::where(['restaurant_id' => $restaurantDetails->id, 'order_status_id' => config('admin.constants.ORDER_STATUS_COMPLETED')])->Where('updated_at', 'like', '%' . Carbon::today()->format('Y-m-d') . '%')->orderBy('id', 'DESC')->limit(6)->get();
            } else {
                $ordersCount = User::where('users.role_id', config('admin.constants.PUBLIC_USERS_ROLE'))->count();
                $menuItemsCount = User::where('users.role_id', config('admin.constants.DELIVERY_PERSON_ROLE'))->count();
                $revenueCount = User::where('users.role_id', config('admin.constants.RESTAURANT_OWNER_ROLE'))->count();
            }

            //facility logo path & base urls
            $this->const['defaultLogoUrl'] = config('admin.path.facilityLogo.DEFAULT_URL');
            $this->const['logoBaseUrl'] = config('admin.path.facilityLogo.BASE_URL');
            $this->const['logoStoragePath'] = config('admin.path.STORAGE_DIR') .
                config('admin.path.UPLOAD_DIR') . '/' .
                config('admin.path.facilityLogo.DIR');

            $data['ordersCount'] = $ordersCount;
            $data['menuItemsCount'] = $menuItemsCount;
            $data['revenueCount'] = $revenueCount;
            $data['orderDetails'] = $orderDetails;

            if ($role->is_super_admin) {
                $view = 'Vendor/vendor/dashboard';
            } else {
                $view = 'Admin/admin/facility-dashboard';
            }

            $data = array_merge($data, $this->const, $this->breadCrumb);
            //dd($data);
            return view($view)->with($data);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * @Method: logout()
     * @Scope: public
     * @returns: redirect login screen after successful logout
     * @Description: Ends the user authentication session
     * @Created 29/June/2021
     * @Updated 29/June/2021
     */
    public function logout()
    {
        try {
            //$x = 1 / 0;
            Auth::guard('restaurant')->logout();
            \Session::flash('success', __('admin/flash_msg.success.logout'));
            return \Redirect::to($this->url);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * @Method: rstOnOffLine()
     * @Scope: public
     * @Description: update offline and online food status delivery facilitated by restaurant.
     * @Created 25/Aug/2021
     * @Updated 25/Aug/2021
     */

    public function rstOnOffLine()
    {
        try {
            // //$ordersCount = User::where('users.role_id', config('admin.constants.PUBLIC_USERS_ROLE'))->count();
            // echo Auth::guard('restaurant')->user()->id."================".Auth::guard('restaurant')->user()->is_active;
            $restaurantStatus = Auth::guard('restaurant')->user()->is_active;
            $user = User::find(Auth::guard('restaurant')->user()->id);
            if ($restaurantStatus == 1) {
                $user->is_active = false;
                $modelValue = array('status' => '<label class="switch rstOnOffLine"> <input type="checkbox"> <span class="slider round"></span></label> Offline');
            } else {
                $user->is_active = true;
                $modelValue = array('status' => '<label class="switch rstOnOffLine"> <input type="checkbox" checked> <span class="slider round"></span></label> Online');
            }

            $title = "Restaurant Status";
            $body = "Restaurant status changed";
            $notification = array('title' => $title, 'body' => $body, 'sound' => 'default', 'badge' => '1', 'orderId' => null, 'actionId' => config('api.constants.push_notification.actionId.restaurant_online_offline'));

            PushNotification::broadcastNotification($notification);

            $user->save();
            return response()->json(['success' => $modelValue]);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }
}
