<?php

namespace app\Http\Controllers\Vendor;

use Illuminate\Http\Request;
use app\Http\Controllers\Vendor\BaseController;
use app\Model\Vendor\Order;
use app\Model\Admin\Role;


use Validator;
use Auth;
use app\Model\Admin\Restaurant as CurrentRestaurant;

class TransactionController extends BaseController
{
    //
    private $const;
    private $roleId;
    private $roleName;
    private $breadCrumb;
    private $url;
    private $title;
    private $subTitle;
    private $controllerUri;
    private $actionUri;

    /**
     * @Method: __construct()
     * @Scope: public
     * @Params: null
     * @Description: Initialize variables like  @roleId, @roleName, @url within private scope of current class
     * @Created 1/july/2021
     * @Updated 5/july/2021
     */

    public function __construct()
    {
        try {
            parent::__construct();
            $this->middleware('auth.restaurant')->except('vendorLogin');
            //fixed role for this module
            $this->roleId = config('admin.constants.RESTAURANT_OWNER_ROLE');

            $role = Role::select('name')->where('id', $this->roleId)->first();
            if ($role->name) {
                $this->roleName = $role->name;
            }

            $this->url = config('admin.path.VENDOR_BASE_URL') . '/transaction';
            $this->title = 'Manage Transaction';
            $this->controllerUri = 'transaction';

            $this->const['url'] = $this->url;
            $this->const['title'] = $this->title;
            $this->const['controllerUri'] = $this->controllerUri;
            $this->const['roleId'] = $this->roleId;
            $this->const['roleName'] = $this->roleName;
            $this->const['restaurantItemPath'] = config('admin.path.STORAGE_DIR') . config('admin.path.UPLOAD_DIR') . '/' . config('admin.path.avatar.RESTAURANT_ITEM_DIR');

            //default breadcrumb option (Dashboard) for every route
            $this->breadCrumb['faClass'] = 'fa-table';
            $this->breadCrumb['breadCrumData'] = [
                0 => [
                    'text' => 'Dashboard',
                    'url' => config('admin.path.ADMIN_BASE_URL'),
                    'breadFaClass' => 'fa-dashboard',
                ],

            ];
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }


    /**
     * Author: Jaidev
     * Email: jaidev@nmgtechnologies.com
     * @Method: index()
     * @Scope: public
     * @Params: @page,
     * @returns: A List of restaurant food menu list
     * @Description: This function fetches  Restaurant food menu from database, sets up
     *  breadcrumb & other constants.
     * @Created 5/july/2021
     * @Updated 25/April/2022
     */
    public function index(Request $request)
    {
        try {

            $this->breadCrumb['breadCrumData'][1] = [
                'text' => $this->const['title'],
            ];
            $restaurant_id = $this->getAuthRestaurantId(Auth::guard('restaurant')->user()->id);
            $request->request->add(['restaurant_id' => $restaurant_id]);


            $page = $request->query('page', 1);
            $this->subTitle = 'Manage Transactions';
            $limit = $request->query('limit', config('admin.constants.DEFAULT_PAGE_LIMIT'));
            $this->const['page'] = $page;
            $this->const['formId'] = '';
            $this->const['limit'] = $limit;
            $this->const['subTitle'] = $this->subTitle;
            $request->request->add(['status' => config('admin.constants.ORDER_STATUS_COMPLETED')]);
            $list = Order::list($request, 'punched');
            $paymentSummary = Order::getPaymentSummary($restaurant_id, $request->input('start_date'), $request->input('end_date'));
            $data['list'] = $list;
            $data['paymentSummary'] = $paymentSummary;
            $data = array_merge($data, $this->const, $this->breadCrumb);
            return view('Vendor.transaction.index')->with($data);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }



    /**
     * Author: Jaidev
     * Email: jaidev@nmgtechnologies.com
     * @Method: getAuthRestaurantId()
     * @Scope: public
     * @input: null
     * @Description: Return Logged user in restaurant  id
     * @Created 5/Jul/2021
     * @Updated 5/Jul/2021
     */
    public function getAuthRestaurantId()
    {
        $CurrentRestaurant = CurrentRestaurant::getRow(Auth::guard('restaurant')->user()->id);
        return $CurrentRestaurant->id;
    }
}
