<?php

namespace app\Http\Controllers\Vendor;

use Illuminate\Http\Request;
use app\Http\Controllers\Controller;
use app\Http\Controllers\Vendor\BaseController;
use app\Model\Admin\Restaurant_menu;
use app\Model\Vendor\Restaurant;
use app\Model\Admin\Role;
use app\Model\Vendor\FoodCategory;
use app\Model\Vendor\CartItem;
use app\Model\Vendor\OrderItem;


use Validator;
use Auth;
use app\Model\Admin\Restaurant as CurrentRestaurant;
use app\Model\Vendor\Restaurant_menu as VendorRestaurant_menu;

class RestaurantMenuController extends BaseController
{
    //
    private $const;
    private $roleId;
    private $roleName;
    private $breadCrumb;
    private $url;
    private $title;
    private $subTitle;
    private $controllerUri;
    private $actionUri;

    /**
     * @Method: __construct()
     * @Scope: public
     * @Params: null
     * @Description: Initialize variables like  @roleId, @roleName, @url within private scope of current class
     * @Created 1/july/2021
     * @Updated 5/july/2021
     */

    public function __construct(Restaurant $u)
    {
        try {
            parent::__construct();
            $this->middleware('auth.restaurant')->except('vendorLogin');
            //fixed role for this module
            $this->roleId = config('admin.constants.RESTAURANT_OWNER_ROLE');

            $role = Role::select('name')->where('id', $this->roleId)->first();
            if ($role->name) {
                $this->roleName = $role->name;
            }

            $this->url = config('admin.path.VENDOR_BASE_URL') . '/menu';
            $this->title = 'Manage Menu';
            $this->controllerUri = 'menu';

            $this->const['url'] = $this->url;
            $this->const['title'] = $this->title;
            $this->const['controllerUri'] = $this->controllerUri;
            $this->const['roleId'] = $this->roleId;
            $this->const['roleName'] = $this->roleName;
            $this->const['restaurantItemPath'] = config('admin.path.STORAGE_DIR') . config('admin.path.UPLOAD_DIR') . '/' . config('admin.path.avatar.RESTAURANT_ITEM_DIR');

            //used in Form layout to load validation messges in js variable
            $this->const['objValidationMsgs'] = 'delivery-person';
            $this->const['objValidationJs'] = 'delivery-person';



            //default breadcrumb option (Dashboard) for every route
            $this->breadCrumb['faClass'] = 'fa-table';
            $this->breadCrumb['breadCrumData'] = [
                0 => [
                    'text' => 'Dashboard',
                    'url' => config('admin.path.ADMIN_BASE_URL'),
                    'breadFaClass' => 'fa-dashboard',
                ],

            ];
            $this->u = $u;
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }


    /**
     * Author: Jaidev
     * Email: jaidev@nmgtechnologies.com
     * @Method: index()
     * @Scope: public
     * @Params: @page,
     * @returns: A List of restaurant food menu list 
     * @Description: This function fetches  Restaurant food menu from database, sets up
     *  breadcrumb & other constants. 
     * @Created 5/july/2021
     * @Updated 5/July/2021
     */
    public function index(Request $request)
    {
        try {
            $this->breadCrumb['breadCrumData'][1] = [
                'text' => $this->const['title'],
            ];
            $page = $request->query('page', 1);
            $this->subTitle = 'Menu Item List';
            $limit = $request->query('limit', config('admin.constants.DEFAULT_PAGE_LIMIT'));
            $this->const['page'] = $page;
            $this->const['formId'] = '';
            $this->const['limit'] = $limit;
            $this->const['subTitle'] = $this->subTitle;
            $list = FoodCategory::getDistinctCategory($request, Auth::guard('restaurant')->user()->id);
            $data['restaurent_id'] = $this->getAuthRestaurantId();
            $data['foodCatList'] = FoodCategory::where(['status' => 1])->get();
            $data['list'] = $list;
            $data = array_merge($data, $this->const, $this->breadCrumb);
            return \View('Vendor.menu.index')->with($data);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }



    /**
     * Author: Jaidev
     * Email: jaidev@nmgtechnologies.com
     * @Method: getAuthRestaurantId()
     * @Scope: public
     * @input: null
     * @Description: Return Logged user in restaurant  id
     * @Created 5/Jul/2021
     * @Updated 5/Jul/2021
     */
    public function getAuthRestaurantId()
    {
        $CurrentRestaurant = CurrentRestaurant::getRow(Auth::guard('restaurant')->user()->id);
        return $CurrentRestaurant->id;
    }


    /**
     * Author: Jaidev
     * Email: jaidev@nmgtechnologies.com
     * @Method: create()
     * @Scope: public
     * @input: menu items
     * @Description: Show input form
     * @Created 5/Jul/2021
     * @Updated 5/Jul/2021
     */
    public function create()
    {

        try {
            // method/action specific initializations
            $this->subTitle = 'Add';
            $this->actionUri = 'create';
            $formId = 'createRestaurantMenuItem';

            //pre-populated data
            $this->const['subTitle'] = $this->subTitle;
            $this->const['actionUri'] = $this->actionUri;
            $this->const['formId'] = $formId;
            $this->const['roles'] = [$this->roleId => $this->roleName];

            //push current controller then method in breadcrumb at position 1 & 2
            $this->breadCrumb['breadCrumData'][1] = [
                'text' => $this->title,
                'url' => $this->url,
                'breadFaClass' => '',
            ];
            $this->breadCrumb['breadCrumData'][2] = [
                'text' => $this->subTitle,
                'breadFaClass' => 'active',
            ];

            $data['foodCatList'] = FoodCategory::where(['status' => 1])->orderBy('name', 'asc')->get();

            $data = array_merge($this->const, $this->breadCrumb, $data);
            return \View('Vendor.menu.create')->with($data);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * Author: Jaidev
     * Email: jaidev@nmgtechnologies.com
     * @Method: store()
     * @Scope: public
     * @input: food item data
     * @returns: After successful submition of data redirect to listing page 
     * @Description: Create the specified resource in storage.
     * @Created 5/jul/2021
     * @Updated 5/jul/2021
     */
    public function store(Request $request)
    {
        try {
            if ($request->isMethod('post')) {
                $input = $request->only(['food_category_id', 'name', 'prep_time', 'price', 'allergy_advice', 'description']);
                $validator = Validator::make($input, Restaurant_menu::Rules(), Restaurant_menu::messages());

                if ($validator->fails()) {
                    return \Redirect::back()->withInput()->withErrors($validator->errors());
                } else {
                    $currentRestuarantData =  CurrentRestaurant::getRow(Auth::guard('restaurant')->user()->id);
                    $input['restaurant_id'] = $currentRestuarantData->id;
                    $input['status'] = 1;
                    $status = Restaurant_menu::create($input);
                    if ($status) {
                        if ($request->hasFile('image')) {
                            $field = 'image';
                            if (!Restaurant_menu::upload_doc($request->image, $status->id, $field)) {
                                return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.UnableToUploadMenuItemImage'));
                            }
                        }
                    } else {
                        return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.UnableToUpdateFacilityUser'));
                    }
                    \Session::flash('success', __('admin/flash_msg.success.RestaurentMenuCreated'));
                    return \Redirect::to($this->url);
                }
            }
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }



    /**
     * Author: Jaidev
     * Email: jaidev@nmgtechnologies.com
     * @Method: edit()
     * @Scope: public
     * @params: @id
     * @returns: Returns pre-populated form data with other constants used in UI
     * @Description: It only fetches restaurant menu detail as per given id & return change its status.
     * @Created 6/jul/2021
     * @Updated 6/jul/2021
     */
    public function edit($id)
    {
        try {
            if (request()->ajax()) {
                if (request()->isMethod('get')) {
                    if (request()->isJson()) {
                        $input = request()->json()->all();
                    } else {
                        $input = request()->all();
                    }
                    $id = \Crypt::decryptString($input['id']);
                    $model = 'restaurant_menus';
                    $modelStatus = \DB::table($model)->select('status')->find($id);
                    $modelData = array('status' => '1');
                    $editUrl = $this->url . '/' . $input['id'] . '/edit';
                    $statusTitle = 'ACTIVE';
                    $isChecked = 'checked';


                    if ($modelStatus->status == '1') {
                        $modelData = array('status' => '0');
                        $result = \DB::table($model)->where('id', $id)->update($modelData);

                        // setup variable values for inactive status
                        $statusTitle = 'INACTIVE';
                        $isChecked = '';
                    } else {
                        $result = \DB::table($model)->where('id', $id)->update($modelData);
                    }

                    $modelValue = array('status' => ' <label class="switch"> <input type="checkbox" data="' . $input['id'] . '" baseUrl="' . $editUrl . '" class="change-restaurant-menu-status" ' . $isChecked . '> <span class="slider round"></span></label>' . $statusTitle);

                    if ($result) {
                        $status = '1';
                    }
                }
                return response()->json(['success' => $status, 'changeStatus' => $modelValue, 'id' => $id]);
            }

            // method/action specific initializations
            $this->subTitle = 'Edit';
            $this->actionUri = 'edit';
            $formId = 'createRestaurantMenuItem';

            //pre-populated data
            $this->const['subTitle'] = $this->subTitle;
            $this->const['actionUri'] = $this->actionUri;
            $this->const['formId'] = $formId;
            $this->const['roles'] = [$this->roleId => $this->roleName];
            $id = \Crypt::decryptString($id);

            $data['foodCatList'] = FoodCategory::where(['status' => 1])->get();
            $data['menuData'] = VendorRestaurant_menu::getDetaById($id);

            $data = array_merge($data, $this->const);
            return \View('Vendor.menu.edit')->with($data);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }




    /**
     * Author: Jaidev
     * Email: jaidev@nmgtechnologies.com
     * @Method: show()
     * @Scope: public
     * @params: @id
     * @Description: show detail of records
     * @Created 6/Jul/2021
     * @Updated 6/Jul/2021
     */
    public function show($id)
    {
        try {
            // method/action specific initializations
            $this->subTitle = 'View';
            $this->actionUri = 'show';

            //pre-populated data
            $this->const['subTitle'] = $this->subTitle;
            $this->const['actionUri'] = $this->actionUri;
            $this->const['formId'] = '';

            $id = \Crypt::decryptString($id);
            $data['menuData'] = VendorRestaurant_menu::getDetaById($id);
            $data = array_merge($data, $this->const);
            return \View('Vendor.' . $this->controllerUri . '.show')->with($data);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }



    /**
     * Author: Jaidev
     * Email: jaidev@nmgtechnologies.com
     * @Method: destroy()
     * @Scope: public
     * @params: @id
     * @Description: Remove the specified resource from storage then redirect to listing page
     * @Created 6/Jul/2021
     * @Updated 6/Jul/2021
     */
    public function destroy($id)
    {
        try {
            $id = \Crypt::decryptString($id);
            $menu = Restaurant_menu::find($id);
            if (!empty($menu->image)) {
                $dirPath = $this->const['restaurantItemPath'] . '/' . $id;
                deleteDir($dirPath);
            }
            // check if it is associate with any cart item or order item
            $cartAssociate = CartItem::where(['item_id' => $id])->count();
            if ($cartAssociate > 0) {
                \Session::flash('error', __('admin/flash_msg.error.itemAssociateWithCart'));
                return \Redirect::to($this->url);
            }

            $orderAssociate = OrderItem::where(['item_id' => $id])->count();
            if ($orderAssociate > 0) {
                \Session::flash('error', __('admin/flash_msg.error.itemAssociateWithOrder'));
                return \Redirect::to($this->url);
            }
            $uid = $menu->id;
            $menu->delete();

            \Session::flash('success', __('admin/flash_msg.success.RestaurentMenuDeleted'));
            return \Redirect::to($this->url);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }


    /**
     * Author: Jaidev
     * Email: jaidev@nmgtechnologies.com
     * @Method: update()
     * @Scope: public
     * @params: @id
     * @returns: Returns pre-populated form data with other constants used in UI
     * @Description: Update the specified resource in storage.
     * @Created 6/jul/2021
     * @Updated 6/jul/2021
     */
    public function update(Request $request, $id)
    {
        try {
            $id = \Crypt::decryptString($id);
            if ($request->isMethod('put')) {
                $rollBack = false;
                \DB::beginTransaction();
                $input = $request->only(['food_category_id', 'name', 'prep_time', 'allergy_advice', 'description', 'price']);
                $validator = Validator::make($request->all(), Restaurant_menu::Rules($id), Restaurant_menu::messages());
                if ($validator->fails()) {
                    return \Redirect::back()->withInput()->withErrors($validator->errors());
                } else {
                    $menuItem = Restaurant_menu::find($id);
                    $menuItem->food_category_id = trim($input['food_category_id']);
                    $menuItem->name = trim($input['name']);
                    $menuItem->prep_time = trim($input['prep_time']);
                    $menuItem->allergy_advice = trim($input['allergy_advice']);
                    $menuItem->description = trim($input['description']);
                    $menuItem->price = trim($input['price']);

                    $updateStatus = $menuItem->save();

                    if ($updateStatus) {
                        if ($request->hasFile('image')) {
                            $field = 'image';
                            if (!Restaurant_menu::upload_doc($request->image, $id, $field)) {
                                return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.UnableToUploadMenuItemImage'));
                            }
                        }
                    } else {
                        return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.UnableToUpdateRestaurantMenu'));
                    }

                    if (!$rollBack) {
                        \DB::commit();
                    } else {
                        \DB::rollBack();
                    }
                }
                if (!$rollBack) {
                    \DB::commit();
                    \Session::flash('success', __('admin/flash_msg.success.RestaurentMenuUpdated'));
                    return \Redirect::to($this->url);
                }
            }
        } catch (\Exception $e) {
            dd($e->getMessage());
            \DB::rollBack();
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }
}
