<?php

namespace app\Http\Controllers\Vendor;

use Illuminate\Http\Request;
use app\Http\Controllers\Vendor\BaseController;
use app\Model\Admin\Restaurant;
use app\Model\Admin\Role;
use app\Model\Admin\User;
use Validator;
use Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Carbon;
use Twilio;
use Twilio\Rest\Client;
use DB;

class ProfileController extends BaseController
{
    //
    private $const;
    private $roleId;
    private $roleName;
    private $breadCrumb;
    private $url;
    private $title;
    private $subTitle;
    private $controllerUri;
    private $actionUri;

    /**
     * @Method: __construct()
     * @Scope: public
     * @Params: null
     * @Description: Initialize variables like  @roleId, @roleName, @url within private scope of current class
     * @Created 7/july/2021
     * @Updated 7/july/2021
     */

    public function __construct(Restaurant $u)
    {

        try {
            parent::__construct();
            $this->middleware('auth.restaurant')->except('vendorLogin');
            //fixed role for this module
            $this->roleId = config('admin.constants.RESTAURANT_OWNER_ROLE');

            $role = Role::select('name')->where('id', $this->roleId)->first();
            if ($role->name) {
                $this->roleName = $role->name;
            }

            $this->url = config('admin.path.VENDOR_BASE_URL') . '/profile';
            $this->title = 'Restaurant Profile';
            $this->controllerUri = 'profile';

            $this->const['url'] = $this->url;
            $this->const['title'] = $this->title;
            $this->const['controllerUri'] = $this->controllerUri;
            $this->const['roleId'] = $this->roleId;
            $this->const['roleName'] = $this->roleName;

            //used in Form layout to load validation messges in js variable
            // $this->const['objValidationMsgs'] = 'delivery-person';
            // $this->const['objValidationJs'] = 'delivery-person';

            //merge constant into single array with parent controller values
            $this->const = array_merge($this->const);

            //default breadcrumb option (Dashboard) for every route
            $this->breadCrumb['faClass'] = 'fa-table';
            $this->breadCrumb['breadCrumData'] = [
                0 => [
                    'text' => 'Dashboard',
                    'url' => config('admin.path.ADMIN_BASE_URL'),
                    'breadFaClass' => 'fa-dashboard',
                ],

            ];
            $this->u = $u;
        } catch (\Exception $e) {
            // dd($e->getMessage());
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }





    /**
     * Author: Jaidev
     * Email: jaidev@nmgtechnologies.com
     * @Method: index()
     * @Scope: public
     * @Params: @page,
     * @returns: Restaurant profile 
     * @Description: This function fetches  Restaurant profile from database 
     * @Created 7/july/2021
     * @Updated 7/July/2021
     */
    public function index(Request $request)
    {
        try {

            $page = $request->query('page', 1);
            $this->subTitle = 'Details';
            $limit = $request->query('limit', config('admin.constants.DEFAULT_PAGE_LIMIT'));
            $this->const['page'] = $page;
            $this->const['limit'] = $limit;
            $this->const['formId'] = 'index';
            $this->const['title'] = $this->title;
            $this->const['subTitle'] = $this->subTitle;
            $this->const['formId'] = '';
            $this->const['url'] = $this->url;
            $user_id = Auth::guard('restaurant')->user()->id;
            $data['profileData'] = Restaurant::getRestaurantData($user_id);
            $userAvatar = \app\Model\Admin\UserAvatars::where(['user_id' => $user_id])->latest();
            $data['userAvatar'] = (!empty($userAvatar->first()) ? $userAvatar->first()->avatar : '');
            $data = array_merge($data, $this->const);
            return \View('Vendor.profile.index')->with($data);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }





    /**
     * Author: Jaidev
     * Email: jaidev@nmgtechnologies.com
     * @Method: edit()
     * @Scope: public
     * @params: @id
     * @returns: Returns pre-populated form data with other constants used in UI
     * @Description: It fetch restaurant and its users profile detail as per given id & return change its status.
     * @Created 8/jul/2021
     * @Updated 8/jul/2021
     */
    public function edit($id)
    {
        try {
            if (request()->ajax()) {
                if (request()->isMethod('get')) {
                    if (request()->isJson()) {
                        $input = request()->json()->all();
                    } else {
                        $input = request()->all();
                    }
                    $id = \Crypt::decryptString($input['id']);
                    $model = 'restaurant_menus';
                    $modelStatus = \DB::table($model)->select('status')->find($id);
                    $modelData = array('status' => '1');
                    $editUrl = $this->url . '/' . $input['id'] . '/edit';
                    $statusTitle = 'ACTIVE';
                    $isChecked = 'checked';


                    if ($modelStatus->status == '1') {
                        $modelData = array('status' => '0');
                        $result = \DB::table($model)->where('id', $id)->update($modelData);

                        // setup variable values for inactive status
                        $statusTitle = 'INACTIVE';
                        $isChecked = '';
                    } else {
                        $result = \DB::table($model)->where('id', $id)->update($modelData);
                    }

                    $modelValue = array('status' => ' <label class="switch"> <input type="checkbox" data="' . $input['id'] . '" baseUrl="' . $editUrl . '" class="change-restaurant-menu-status" ' . $isChecked . '> <span class="slider round"></span></label>' . $statusTitle);

                    if ($result) {
                        $status = '1';
                    }
                }
                return response()->json(['success' => $status, 'changeStatus' => $modelValue, 'id' => $id]);
            }

            // method/action specific initializations
            $this->subTitle = 'Edit';
            $this->actionUri = 'edit';
            $formId = 'editRestaurant';

            //pre-populated data
            $this->const['subTitle'] = $this->subTitle;
            $this->const['actionUri'] = $this->actionUri;
            $this->const['formId'] = $formId;
            $this->const['roles'] = [$this->roleId => $this->roleName];
            $id = \Crypt::decryptString($id);
            $userId = Auth::guard('restaurant')->user()->id;
            $data['profileData'] = Restaurant::getRestaurantData($userId);
            $userAvatar = \app\Model\Admin\UserAvatars::where(['user_id' => $userId])->latest();
            $data['userAvatar'] = (!empty($userAvatar->first()) ? $userAvatar->first()->avatar : '');
            $data = array_merge($data, $this->const);
            return \View('Vendor.profile.edit')->with($data);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }




    /**
     * Author: Jaidev
     * Email: jaidev@nmgtechnologies.com
     * @Method: show()
     * @Scope: public
     * @params: @id
     * @Description: show detail of records
     * @Created 6/Jul/2021
     * @Updated 6/Jul/2021
     */
    public function show($id)
    {
        //
    }



    /**
     * Author: Jaidev
     * Email: jaidev@nmgtechnologies.com
     * @Method: destroy()
     * @Scope: public
     * @params: @id
     * @Description: Remove the specified resource from storage then redirect to listing page
     * @Created 6/Jul/2021
     * @Updated 6/Jul/2021
     */
    public function destroy($id)
    {
        //
    }


    /**
     * Author: Jaidev
     * Email: jaidev@nmgtechnologies.com
     * @Method: update()
     * @Scope: public
     * @params: @id
     * @returns: Returns pre-populated form data with other constants used in UI
     * @Description: Update the specified resource in storage.
     * @Created 8/jul/2021
     * @Updated 8/jul/2021
     */
    public function update(Request $request, $id)
    {
        try {
            $id = \Crypt::decryptString($id);
            $restData = Restaurant::getRestaurantData(Auth::guard('restaurant')->user()->id);
            if ($request->isMethod('put')) {
                $rollBack = false;
                \DB::beginTransaction();
                $input = $request->only(['name', 'restaurant_bio', 'address1', 'address2', 'locality', 'country', 'postcode', 'type', 'latitude', 'longitude', 'delivery_time', 'avatar']);
                $validator = Validator::make($request->all(), Restaurant::VendorPanelRules($restData->user->id), Restaurant::vendorPanelMessages());
                if ($validator->fails()) {
                    return \Redirect::back()->withInput()->withErrors($validator->errors());
                } else {
                    $restaurant = Restaurant::find($id);
                    $restaurant->name = trim($input['name']);
                    $restaurant->bio = trim($input['restaurant_bio']);
                    $restaurant->address1 = trim($input['address1']);
                    $restaurant->address2 = trim($input['address2']);
                    $restaurant->city = trim($input['locality']);
                    $restaurant->region = trim($input['country']);
                    $restaurant->zipcode = trim($input['postcode']);
                    $restaurant->delivery_time = trim($input['delivery_time']);
                    $restaurant->resturent_type_groceries = trim(!empty($input['type']['groceries']) ? 1 : 0);
                    $restaurant->resturent_type_local = trim(!empty($input['type']['local']) ? 1 : 0);
                    $restaurant->resturent_type_continental = trim(!empty($input['type']['continental']) ? 1 : 0);
                    $restaurant->latitude = trim($input['latitude']);
                    $restaurant->longitude = trim($input['longitude']);
                    $updateStatus = $restaurant->save();

                    if ($updateStatus) {
                        $user = User::find(Auth::guard('restaurant')->user()->id);
                        if ($request->hasFile('avatar')) {
                            if (!User::upload_avatar($request->avatar, $user->id)) {
                                return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.UnableToUploadAvatarImage'));
                            }
                        }
                    } else {
                        return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.UnableToUpdateRestaurantMenu'));
                    }

                    if (!$rollBack) {
                        \DB::commit();
                    } else {
                        \DB::rollBack();
                    }
                }
                if (!$rollBack) {
                    \DB::commit();
                    \Session::flash('success', __('admin/flash_msg.success.RestaurantUpdated'));
                    return \Redirect::to($this->url);
                }
            }
        } catch (\Exception $e) {
            \DB::rollBack();
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }



    public function validateEmail(Request $request)
    {
        // return $request->all();
        $status = 'fail';
        $msg = '';
        try {
            $user = User::find(Auth::guard('restaurant')->user()->id);
            if ($request->isMethod('post')) {
                $rollBack = false;
                $input = $request->only(['old_email', 'new_email']);
                $validator = Validator::make($input, User::changeEmailRules($user->id), User::changeEmailMessages());
                if ($validator->fails()) {
                    $msg = $validator->getMessageBag()->toArray();
                    // $msg =  $validator->errors();
                } else {
                    // validate restaurant old & new email  id
                    if ($user->email === $input['old_email']) {
                        $request->session()->put('restaurant_email_to_update', $input['new_email']);
                        $status = 'success';
                    } else {
                        $msg = [];
                        $msg['old_email'][0] = "Invalid old email";
                    }
                }
            }
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            $msg = 'Access Denied';
        }
        return ['status' => $status, 'msg' => $msg];
    }

    public function validateCurrentPassword(Request $request)
    {
        $status = 'fail';
        $msg = '';
        try {
            $user = User::find(Auth::guard('restaurant')->user()->id);
            if ($request->isMethod('post')) {
                $rollBack = false;
                $input = $request->only(['password']);
                $validator = Validator::make($input, User::currentPasswordRules($user->id), User::messages());
                if ($validator->fails()) {
                    $msg = $validator->getMessageBag()->toArray();
                    // $msg =  $validator->errors();
                } else {
                    if (Hash::check($input['password'], $user->password)) {

                        // generate and save otp
                        $otp = $this->generatePIN(4);
                        $user->verification_otp  = $otp;
                        $user->verification_otp_sent_at = Carbon::now();
                        $user->save();


                        ########  	Change Email OTP on previous email id ###########
                        $restaurantData = Restaurant::where(['owner_id' => $user->id])->first();
                        if (!empty($input['lang'])) {
                            \App::setLocale($input['lang']);
                        }
                        $params['replaceKeywords']['{USER_NAME}'] = $restaurantData->name;
                        $params['replaceKeywords']['{OTP}'] = $otp;
                        $params['toEmail'] = $user->email;
                        $params['emailSlug'] = 'emailid_varification_otp_by_mobile_or_restaurant';
                        $this->customMailer($params);
                        ######## End Change Email OTP on previous email id ###########
                        $status = 'success';
                    } else {
                        $msg = [];
                        $msg['password'][0] = "Invalid password";
                    }
                }
            }
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            $msg = $e->getMessage();
            // $msg = 'Access Denied';
        }
        return ['status' => $status, 'msg' => $msg];
    }



    public function verifyOtp(Request $request)
    {
        $status = 'fail';
        $msg = '';
        try {
            $user = User::find(Auth::guard('restaurant')->user()->id);
            if ($request->isMethod('post')) {
                $rollBack = false;
                $input = $request->only(['otp']);
                $validator = Validator::make($input, User::otpRules($user->id));
                if ($validator->fails()) {
                    $msg = $validator->getMessageBag()->toArray();
                } else {
                    $input['otp'] = implode('', $input['otp']);

                    if ($user->verification_otp === $input['otp']) {
                        $user->verification_otp  = null;
                        $user->verification_otp_sent_at = null;
                        $user->email = $request->session()->get('restaurant_email_to_update');
                        $user->save();
                        $status = 'success';
                    } else {
                        $msg = [];
                        $msg['otp'][0] = "Invalid otp";
                    }
                }
            }
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            $msg = $e->getMessage();
            // $msg = 'Access Denied';
        }
        return ['status' => $status, 'msg' => $msg];
    }


    /** 
     * author : jaidev
     * email : jaidev@nmgtechnologies.com
     * desc : validate user's phone number before change
     * param : old phone number & new phone number
     * return validation status success/fail  
     */
    public function validatePhoneNumber(Request $request)
    {
        $status = 'fail';
        $msg = '';
        try {
            $user = User::find(Auth::guard('restaurant')->user()->id);
            if ($request->isMethod('post')) {
                $rollBack = false;
                $input = $request->only(['old_number', 'new_number']);
                $validator = Validator::make($input, User::changePhoneNumberRules($user->id));
                if ($validator->fails()) {
                    $msg = $validator->getMessageBag()->toArray();
                    // $msg =  $validator->errors();
                } else {
                    // validate restaurant old & new email  id
                    $otp = $this->generatePIN(4);
                    if ($user->phone_number === $input['old_number']) {
                        // return $request->all();
                        //send sms to user
                        $twilio = new Client(env('TWILIO_ACCOUNT_SID'), env('TWILIO_SECRET_KEY'));
                        $sendSms = $twilio->messages->create(
                            config('admin.constants.COUNTRY_CODE') . $user->phone_number,
                            [
                                "body" => "You have requested for OTP, your otp is {$otp}",
                                "from" => env('TWILIO_TRIAL_NUMBER')
                            ]
                        );

                        //check if SMS goes to user then add sendEmail true or false
                        if ($twilio) {
                            $user->verification_otp  = $otp;
                            $user->verification_otp_sent_at = Carbon::now();
                            $user->save();
                        } else {
                            $msg = 'failed to send otp';
                        }
                        $request->session()->put('restaurant_phone_number_to_update', $input['new_number']);

                        $status = 'success';
                    } else {
                        $msg = [];
                        $msg['old_number'][0] = "Invalid old phone number";
                    }
                }
            }
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            $msg = 'Access Denied';
        }
        return ['status' => $status, 'msg' => $msg];
    }


    public function verifyOtpForNumber(Request $request)
    {
        $status = 'fail';
        $msg = '';
        try {
            $user = User::find(Auth::guard('restaurant')->user()->id);
            if ($request->isMethod('post')) {
                $rollBack = false;
                $input = $request->only(['otp']);
                $validator = Validator::make($input, User::otpRules($user->id));
                if ($validator->fails()) {
                    $msg = $validator->getMessageBag()->toArray();
                } else {
                    $input['otp'] = implode('', $input['otp']);

                    if ($user->verification_otp === $input['otp']) {
                        $user->verification_otp  = null;
                        $user->verification_otp_sent_at = null;
                        $user->phone_number = $request->session()->get('restaurant_phone_number_to_update');
                        $user->save();
                        $status = 'success';
                    } else {
                        $msg = [];
                        $msg['sms_otp'][0] = "Invalid otp";
                    }
                }
            }
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            $msg = $e->getMessage();
            // $msg = 'Access Denied';
        }
        return ['status' => $status, 'msg' => $msg];
    }



    function sendSmsOtp(Request $request)
    {
        $status = 'fail';
        $msg = '';
        try {
            $user = User::find(Auth::guard('restaurant')->user()->id);
            if ($request->isMethod('post')) {
                $rollBack = false;
                $otp = $this->generatePIN(4);
                if ($user->phone_number) {
                    //send sms to user
                    $twilio = new Client(env('TWILIO_ACCOUNT_SID'), env('TWILIO_SECRET_KEY'));
                    $sendSms = $twilio->messages->create(
                        config('admin.constants.COUNTRY_CODE') . $user->phone_number,
                        [
                            "body" => "You have requested for OTP, your otp is {$otp}",
                            "from" => env('TWILIO_TRIAL_NUMBER')
                        ]
                    );

                    //check if SMS goes to user then add sendEmail true or false
                    if ($twilio) {
                        $user->verification_otp  = $otp;
                        $user->verification_otp_sent_at = Carbon::now();
                        $user->save();
                        $status = 'success';
                    } else {
                        $msg = 'failed to send otp';
                    }
                } else {
                    $msg = "User phone number is not available";
                }
            }
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            $msg = $e->getMessage();
        }
        return ['status' => $status, 'msg' => $msg];
    }



    public function resendEmailChangeOtp(Request $request)
    {
        $status = 'fail';
        $msg = '';
        try {
            $user = User::find(Auth::guard('restaurant')->user()->id);
            if ($request->isMethod('post')) {
                // generate and save otp
                $otp = $this->generatePIN(4);
                $user->verification_otp  = $otp;
                $user->verification_otp_sent_at = Carbon::now();
                $user->save();
                ########  	Change Email OTP on previous email id ###########
                $restaurantData = Restaurant::where(['owner_id' => $user->id])->first();
                if (!empty($input['lang'])) {
                    \App::setLocale($input['lang']);
                }
                $params['replaceKeywords']['{USER_NAME}'] = $restaurantData->name;
                $params['replaceKeywords']['{OTP}'] = $otp;
                $params['toEmail'] = $user->email;
                $params['emailSlug'] = 'emailid_varification_otp_by_mobile_or_restaurant';
                $this->customMailer($params);
                ######## End Change Email OTP on previous email id ###########
                $status = 'success';
            }
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            $msg = $e->getMessage();
            // $msg = 'Access Denied';
        }
        return ['status' => $status, 'msg' => $msg];
    }


    /**
     * Author: Jaidev
     * Email: jaidev@nmgtechnologies.com
     * @Method: changePassword()
     * @Scope: public
     * @Params: @page,
     * @returns: Change Password Page 
     * @Description: This function fetches  Restaurant profile from database 
     * @Created 14/july/2021
     * @Updated 14/July/2021
     */
    public function changePassword(Request $request)
    {
        try {
            $this->subTitle = 'Change Password';
            $this->const['formId'] = 'changePassword';
            $this->const['title'] = $this->title;
            $this->const['subTitle'] = $this->subTitle;
            $this->const['url'] = $this->url;
            $data = $this->const;
            return \View('Vendor.profile.change-password')->with($data);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }


    /**
     * Author: Jaidev
     * Email: jaidev@nmgtechnologies.com
     * @Method: updatePassword()
     * @Scope: public
     * @Params: @page,
     * @returns: Password update status 
     * @Description: This function update current user password 
     * @Created 15/july/2021
     * @Updated 15/July/2021
     */
    public function updatePassword(Request $request)
    {
        $status = 'fail';
        $msg = '';
        try {
            $user = User::find(Auth::guard('restaurant')->user()->id);
            if ($request->isMethod('put')) {
                $input = $request->only(['password', 'new_password', 'new_password_confirmation']);
                $validator = Validator::make($input, User::changePasswordVendorRules($user->id), User::messages());
                if ($validator->fails()) {
                    $msg = $validator->getMessageBag()->toArray();
                } else {
                    if (Hash::check($input['password'], $user->password)) {
                        // change password and destroy logged in user session 
                        $user->password  = Hash::make($input['new_password']);
                        $user->save();


                        ########  	Email to user if password updated successfully ###########
                        $restaurantData = Restaurant::where(['owner_id' => $user->id])->first();
                        if (!empty($input['lang'])) {
                            \App::setLocale($input['lang']);
                        }
                        $params['replaceKeywords']['{USER_NAME}'] = $restaurantData->name;
                        $params['toEmail'] = $user->email;
                        // $params['toEmail'] = 'jaidev@nmgtechnologies.com';
                        $params['emailSlug'] = 'app_or_restaurant_changed_password_successfully';
                        $this->customMailer($params);
                        ######## End Change Email OTP on previous email id ###########


                        // logout current user 
                        Auth::guard('restaurant')->logout();
                        $status = 'success';
                    } else {
                        $msg = [];
                        $msg['password'][0] = 'Invalid password';
                    }
                }
            }
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            $msg = __('admin/flash_msg.error.passwordChanged');
        }
        return ['status' => $status, 'msg' => $msg];
    }
}
