<?php

namespace app\Http\Controllers\Vendor;

use Illuminate\Http\Request;
use app\Http\Controllers\Vendor\BaseController;
use app\Model\Vendor\Order;
use app\Model\Vendor\DeliveryPersonAssignment;
use app\Model\Vendor\OrderStatus;
use app\Model\Vendor\DeliveryPerson;
use app\Model\Vendor\OrderLog;
use app\Model\Vendor\Notification;


use app\Model\Admin\Role;
use app\Model\Admin\User;
use Validator;
use Auth;
use app\Model\Admin\Restaurant as CurrentRestaurant;
use app\Helpers\CommonHelper;
use Illuminate\Support\Facades\DB;
use app\Traits\DistanceCalculatorTrait;
use Twilio;
use Twilio\Rest\Client;
use app\Components\Api\V1\PushNotification;

class OrderController extends BaseController
{
    //
    private $const;
    private $roleId;
    private $roleName;
    private $breadCrumb;
    private $url;
    private $title;
    private $subTitle;
    private $controllerUri;
    private $actionUri;

    /**
     * @Method: __construct()
     * @Scope: public
     * @Params: null
     * @Description: Initialize variables like  @roleId, @roleName, @url within private scope of current class
     * @Created 1/july/2021
     * @Updated 5/july/2021
     */

    public function __construct(Order $order)
    {
        try {
            parent::__construct();
            $this->middleware('auth.restaurant')->except('vendorLogin');
            //fixed role for this module
            $this->roleId = config('admin.constants.RESTAURANT_OWNER_ROLE');

            $role = Role::select('name')->where('id', $this->roleId)->first();
            if ($role->name) {
                $this->roleName = $role->name;
            }

            $this->url = config('admin.path.VENDOR_BASE_URL') . '/order';
            $this->title = 'Manage Order';
            $this->controllerUri = 'order';

            $this->const['url'] = $this->url;
            $this->const['title'] = $this->title;
            $this->const['controllerUri'] = $this->controllerUri;
            $this->const['roleId'] = $this->roleId;
            $this->const['roleName'] = $this->roleName;

            //used in Form layout to load validation messges in js variable
            $this->const['objValidationMsgs'] = 'order';
            $this->const['objValidationJs'] = 'order';

            //default breadcrumb option (Dashboard) for every route
            $this->breadCrumb['faClass'] = 'fa-table';
            $this->breadCrumb['breadCrumData'] = [
                0 => [
                    'text' => 'Dashboard',
                    'url' => config('admin.path.ADMIN_BASE_URL'),
                    'breadFaClass' => 'fa-dashboard',
                ],
            ];
            $this->u = $order;
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }


    /**
     * Author: Jaidev
     * Email: jaidev@nmgtechnologies.com
     * @Method: index()
     * @Scope: public
     * @Params: @page,
     * @returns: List of restaurant orders
     * @Description: This function fetches  Restaurant orders from database, sets up
     *  breadcrumb & other constants.
     * @Created 10/aug/2021
     * @Updated 10/aug/2021
     */
    public function index(Request $request)
    {
        try {
            $this->breadCrumb['breadCrumData'][1] = [
                'text' => $this->const['title'],
            ];
            $restaurant_id = $this->getAuthRestaurantId(Auth::guard('restaurant')->user()->id);
            $request->request->add(['restaurant_id' => $restaurant_id]);
            $page = $request->query('page', 1);
            $this->subTitle = 'Order List';
            $limit = $request->query('limit', config('admin.constants.DEFAULT_PAGE_LIMIT'));
            $this->const['page'] = $page;
            $this->const['formId'] = '';
            $this->const['limit'] = $limit;
            $this->const['subTitle'] = $this->subTitle;
            if ($request->input('type') == 'pending') {
                $pendingPage = $request->input('page');
            } else {
                $pendingPage = 1;
            }

            if ($request->input('type') == 'punched') {
                $punchedPage = $request->input('page');
            } else {
                $punchedPage = 1;
            }
            // $input = $request->all();

            // dd($input);
            $request->request->add(['page' => $punchedPage]);
            $punchedList = Order::list($request, $type = 'punched');
            $request->request->add(['page' => $pendingPage]);
            $pendingList = Order::list($request, $type = 'pending');
            // dd($pendingList);
            $data['orderStatusList'] = OrderStatus::where(['status' => 1])->whereNotIn('id', [1, 2])->get();
            $data['pendingList'] = $pendingList;
            $data['punchedList'] = $punchedList;
            $data = array_merge($data, $this->const, $this->breadCrumb);
            return view('Vendor.order.index')->with($data);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }



    /**
     * Author: Jaidev
     * Email: jaidev@nmgtechnologies.com
     * @Method: getAuthRestaurantId()
     * @Scope: public
     * @input: null
     * @Description: Return Logged user in restaurant  id
     * @Created 5/Jul/2021
     * @Updated 5/Jul/2021
     */
    public function getAuthRestaurantId()
    {
        $CurrentRestaurant = CurrentRestaurant::getRow(Auth::guard('restaurant')->user()->id);
        return $CurrentRestaurant->id;
    }


    /**
     * Author: Jaidev
     * Email: jaidev@nmgtechnologies.com
     * @Method: create()
     * @Scope: public
     * @input: menu items
     * @Description: Show input form
     *
     */
    public function create()
    {
    }

    /**
     * Author: Jaidev
     * Email: jaidev@nmgtechnologies.com
     * @Method: store()
     * @Scope: public
     * @input: order data
     * @returns: After successful submition of data redirect to listing page
     * @Description: Create the specified resource in storage.
     * @Created 10/aug/2021
     * @Updated 10/aug/2021
     */
    public function store(Request $request)
    {
    }



    /**
     * Author: Jaidev
     * Email: jaidev@nmgtechnologies.com
     * @Method: edit()
     * @Scope: public
     * @params: @id
     * @returns: Returns pre-populated form data with other constants used in UI
     * @Description: It only fetches restaurant menu detail as per given id & return change its status.
     * @Created 6/jul/2021
     * @Updated 6/jul/2021
     */
    public function edit($id)
    {
        //
    }




    /**
     * Author: Jaidev
     * Email: jaidev@nmgtechnologies.com
     * @Method: show()
     * @Scope: public
     * @params: @id
     * @Description: show detail of records
     * @Created 10/aug/2021
     * @Updated 10/aug/2021
     */
    public function show($id)
    {
        try {
            // method/action specific initializations
            $this->subTitle = 'View';
            $this->actionUri = 'show';

            //pre-populated data
            $this->const['subTitle'] = $this->subTitle;
            $this->const['actionUri'] = $this->actionUri;
            $this->const['formId'] = '';
            $id = \Crypt::decryptString($id);
            $data['orderData'] = Order::details($id);
            // dd($data['orderData']);
            if ($data['orderData']->delivery_person_id > 0) {
                $data['assignmentData'] = DeliveryPersonAssignment::where(['order_id' => $id, 'delivery_person_id' => $data['orderData']->delivery_person_id])->latest()->first();
                $data['deliveryPersonData'] = DeliveryPerson::where(['id' => $data['orderData']->delivery_person_id])->with('user')->first();
            }

            $data = array_merge($data, $this->const);
            return \View('Vendor.' . $this->controllerUri . '.show')->with($data);
        } catch (\Exception $e) {

            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }



    /**
     * Author: Jaidev
     * Email: jaidev@nmgtechnologies.com
     * @Method: approve()
     * @Scope: public
     * @params: @id
     * @Description: approve order by restaurant and assign nearby delivery boy
     * @Created 11/aug/2021
     * @Updated 11/aug/2021
     */
    public function approve($id)
    {
        try {
            $id = \Crypt::decryptString($id);
            $order = Order::find($id);
            // dd($order);
            if (!empty($order)) {
                DB::beginTransaction();
                if ($order->order_status_id == config('admin.constants.ORDER_STATUS_PENDING')) {
                    $order->order_status_id = config('admin.constants.ORDER_STATUS_IN_PREP');
                    if ($order->save()) {
                        OrderLog::saveLog($order->id, $order->order_status_id);
                        // get nearby delivery person
                        $availableRider =  DeliveryPerson::getClosestAvailableRider($order->latitude, $order->longitude, $order->id);
                        if (!empty($availableRider) && $availableRider->distance <= config('admin.constants.DELIVERY_PERSON_AVAILABILITY_RANGE')) {
                            $order->delivery_person_id = $availableRider->id;
                            $order->save();
                            //
                            $dpData['order_id'] = $order->id;
                            $dpData['latitude'] = $availableRider->latitude;
                            $dpData['longitude'] = $availableRider->longitude;
                            $dpData['delivery_person_id'] = $availableRider->id;
                            $dpData['status'] = 1;
                            DeliveryPersonAssignment::create($dpData);

                            ### send push notification to customer ###
                            $title = "Order accepted by restaurant";
                            $body = "Your order has been accepted by restaurant";
                            $notification = array('title' => $title, 'body' => $body, 'sound' => 'default', 'badge' => '1', 'actionId' => config('api.constants.push_notification.actionId.customer_order_details'), 'orderId' => $order->id);
                            PushNotification::sendNotification($order->user_id, $notification, $order->id);
                            ### send push notification to customer ###



                            #### send notification to rider ###
                            $title = "New order request";
                            $body = "You have a new order request";
                            $notification = array('title' => $title, 'body' => $body, 'sound' => 'default', 'badge' => '1', 'actionId' => config('api.constants.push_notification.actionId.rider_order_details'), 'orderId' => $order->id);

                            PushNotification::sendNotification($availableRider->user_id, $notification, $order->id);
                            #### end sending notification to rider ###


                            \Session::flash('success', __('admin/flash_msg.success.orderAccepted'));
                        } else {
                            // send order in queue for rider assignment after 10 mins
                            $order->delivery_person_id = null;
                            $order->rider_assignment_at = date("Y-m-d H:i:s");
                            $order->save();

                            ### send push notification to customer ###
                            $title = config('api.constants.push_notification.title.rider_in_queue');
                            $body = config('api.constants.push_notification.description.rider_in_queue');
                            $notification = array('title' => $title, 'body' => $body, 'sound' => 'default', 'badge' => '1', 'actionId' => config('api.constants.push_notification.actionId.customer_order_details'), 'orderId' => $order->id);
                            \app\Components\Api\V1\PushNotification::sendNotification($order->user_id, $notification, $order->id);
                            ### send push notification to customer ###

                            \Session::flash('success', __('admin/flash_msg.success.orderAccepted'));
                        }
                        DB::commit();
                        return \Redirect::back();
                    } else {
                        return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.unableToChangeOrderStatus'));
                    }
                } else {
                    //
                    return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.orderNotAcceptable'));
                }
            } else {
                //
                return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.orderNotFound'));
            }
        } catch (\Exception $e) {
            DB::rollBack();
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * Author: Jaidev
     * Email: jaidev@nmgtechnologies.com
     * @Method: reject()
     * @Scope: public
     * @params: @id
     * @Description: reject order by restaurant
     * @Created 11/aug/2021
     * @Updated 11/aug/2021
     */
    public function reject($id)
    {
        try {
            $id = \Crypt::decryptString($id);
            $order = Order::find($id);
            // dd($order);
            if (!empty($order)) {
                //
                if ($order->order_status_id == config('admin.constants.ORDER_STATUS_PENDING')) {
                    $order->order_status_id = config('admin.constants.ORDER_STATUS_REJECTED');
                    if ($order->save()) {
                        // order reject mail to customer
                        $userData = User::find($order->user_id);
                        if (!empty($userData->email)) {
                            if (!empty($input['lang'])) {
                                \App::setLocale($input['lang']);
                            }
                            $params['replaceKeywords']['{USER_NAME}'] = $userData->first_name . ' ' . $userData->last_name;
                            $params['toEmail'] = $userData->email;
                            $params['emailSlug'] = 'customer_order_rejected_restaurant';
                            $this->customMailer($params);
                        }

                        ### send push notification to customer ###
                        $title = "Order rejected by restaurant";
                        $body = "Your order has been rejected by restaurant";
                        $notification = array('title' => $title, 'body' => $body, 'sound' => 'default', 'badge' => '1', 'actionId' => config('api.constants.push_notification.actionId.customer_order_details'), 'orderId' => $order->id);
                        PushNotification::sendNotification($order->user_id, $notification, $order->id);
                        ### send push notification to customer ###


                        \Session::flash('success', __('admin/flash_msg.success.orderRejected'));
                        return \Redirect::back();
                    } else {
                        return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.unableToChangeOrderStatus'));
                    }
                } else {
                    //
                    return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.orderNotAcceptable'));
                }
            } else {
                //
                return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.orderNotFound'));
            }
        } catch (\Exception $e) {
            // dd($e);
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }


    /**
     * Author: Jaidev
     * Email: jaidev@nmgtechnologies.com
     * @Method: outForDelivery()
     * @Scope: public
     * @params: @id
     * @returns: Returns pre-populated form data with other constants used in UI
     * @Description: Update the specified resource in storage.
     * @Created 12/aug/2021
     * @Updated 12/aug/2021
     */
    public function outForDelivery($id)
    {
        //
        try {
            $id = \Crypt::decryptString($id);
            $order = Order::find($id);
            // dd($order);
            if (!empty($order)) {
                //
                DB::beginTransaction();
                if ($order->order_status_id == config('admin.constants.ORDER_STATUS_IN_PREP')) {
                    $order->order_status_id = config('admin.constants.ORDER_STATUS_ON_THE_WAY');
                    if ($order->save()) {
                        OrderLog::saveLog($order->id, $order->order_status_id);
                        $notification['title'] = 'Order handover';
                        $notification['description'] = 'Order no #' . $order->id . ' has been handover to delivery person';
                        $notification['order_id'] = $order->id;
                        $notification['user_id'] = Auth::guard('restaurant')->user()->id;
                        Notification::create($notification);


                        ### send handover notification to customer ###
                        $title = "Order handover to delivery person";
                        $body = "Order no #" . $order->id . " has been handover to delivery person";
                        $notification = array('title' => $title, 'body' => $body, 'sound' => 'default', 'badge' => '1', 'orderId' => $order->id, 'actionId' => config('api.constants.push_notification.actionId.customer_order_details'));
                        PushNotification::sendNotification($order->user_id, $notification, $order->id);
                        ### end sending handover notification to customer ###



                        ### send notification to delivery person ##
                        $body = "Order no #" . $order->id . " has been handover to you";
                        $riderNotification = array('title' => $title, 'body' => $body, 'sound' => 'default', 'badge' => '1', 'orderId' => $order->id, 'actionId' => config('api.constants.push_notification.actionId.rider_order_details'));

                        $deliveryPerson = DeliveryPerson::find($order->delivery_person_id);
                        PushNotification::sendNotification($deliveryPerson->user_id, $riderNotification, $order->id);


                        ### end sending notification to delivery person ##


                        $user = User::find($order->user_id);
                        $twilio = new Client(env('TWILIO_ACCOUNT_SID'), env('TWILIO_SECRET_KEY'));
                        $sendSms = $twilio->messages->create(
                            config('admin.constants.COUNTRY_CODE') . $user->phone_number,
                            [
                                "body" => "Dear " . $order->firstname . ",\n" . $order->delivery_code . " is your OTP for completing your order. Please share it with the delivery person if you received your order.\nRegards,\nTeam BMF",
                                "from" => env('TWILIO_TRIAL_NUMBER')
                            ]
                        );

                        //check if SMS goes to user then add sendEmail true or false
                        if ($twilio) {
                            DB::commit();
                            \Session::flash('success', __('admin/flash_msg.success.orderHandover'));
                            return \Redirect::back();
                        } else {
                            DB::rollBack();
                            \Session::flash('success', __('admin/flash_msg.error.unableToSendSms'));
                            return \Redirect::back();
                        }
                    } else {
                        return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.unableToChangeOrderStatus'));
                    }
                } else {
                    //
                    DB::rollBack();
                    return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.orderNotHandover'));
                }
            } else {
                //
                return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.orderNotFound'));
            }
        } catch (\Exception $e) {
            DB::rollBack();
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }



    /**
     * Author: Jaidev
     * Email: jaidev@nmgtechnologies.com
     * @Method: reAssignRider()
     * @Scope: public
     * @params: @id
     * @Description: re-assign order to nearby delivery boy
     * @Created 31/March/2022
     * @Updated 31/March/2022
     */
    public function reAssignRider($id)
    {
        try {
            $id = \Crypt::decryptString($id);
            $order = Order::find($id);
            // dd($order);
            if (!empty($order)) {
                DB::beginTransaction();
                if ($order->order_status_id == config('admin.constants.ORDER_STATUS_IN_PREP') || $order->order_status_id == config('admin.constants.ORDER_STATUS_ON_THE_WAY')) {
                    // mark old delivery person's status rejected
                    $currentRiderAssignment = DeliveryPersonAssignment::where(['order_id' => $order->id])->latest()->first();

                    $currentRiderAssignment->status = 3;
                    $currentRiderAssignment->reason = "Rider Re-Assigned by Restaurant";
                    $currentRiderAssignment->save();
                    if ($currentRiderAssignment->save()) {

                        $currentDP = \app\Model\Admin\DeliveryPersons::where(['id' => $currentRiderAssignment->delivery_person_id])->first();
                        $currentDP->work_status = 0;
                        $currentDP->save();

                        OrderLog::saveLog($order->id, $order->order_status_id);
                        // get nearby delivery person
                        $availableRider =  DeliveryPerson::getClosestAvailableRider($order->latitude, $order->longitude, $order->id);
                        if (!empty($availableRider) && $availableRider->distance <= config('admin.constants.DELIVERY_PERSON_AVAILABILITY_RANGE')) {
                            $order->delivery_person_id = $availableRider->id;
                            $order->save();
                            //
                            $dpData['order_id'] = $order->id;
                            $dpData['latitude'] = $availableRider->latitude;
                            $dpData['longitude'] = $availableRider->longitude;
                            $dpData['delivery_person_id'] = $availableRider->id;
                            $dpData['status'] = 1;
                            DeliveryPersonAssignment::create($dpData);


                            ### send push notification to customer ###
                            $title = config('api.constants.push_notification.title.rider_updated');
                            $body = config('api.constants.push_notification.description.rider_updated');
                            $notification = array('title' => $title, 'body' => $body, 'sound' => 'default', 'badge' => '1', 'actionId' => config('api.constants.push_notification.actionId.customer_order_details'), 'orderId' => $order->id);
                            \app\Components\Api\V1\PushNotification::sendNotification($order->user_id, $notification, $order->id);
                            ### send push notification to customer ###

                            #### send notification to rider ###
                            $title = config('api.constants.push_notification.title.rider_new_order');
                            $body =  config('api.constants.push_notification.description.rider_new_order');
                            $notification = array('title' => $title, 'body' => $body, 'sound' => 'default', 'badge' => '1', 'actionId' => config('api.constants.push_notification.actionId.rider_order_details'), 'orderId' => $order->id);
                            PushNotification::sendNotification($availableRider->user_id, $notification, $order->id);
                            #### end sending notification to rider ###
                            \Session::flash('success', __('admin/flash_msg.success.orderDeliveryPersonAssigned'));
                        } else {
                            // send order in queue for rider assignment after 10 mins
                            $order->delivery_person_id = null;
                            $order->rider_assignment_at = date("Y-m-d H:i:s");
                            $order->save();

                            ### send push notification to customer ###
                            $title = config('api.constants.push_notification.title.rider_in_queue');
                            $body = config('api.constants.push_notification.description.rider_in_queue');
                            $notification = array('title' => $title, 'body' => $body, 'sound' => 'default', 'badge' => '1', 'actionId' => config('api.constants.push_notification.actionId.customer_order_details'), 'orderId' => $order->id);
                            \app\Components\Api\V1\PushNotification::sendNotification($order->user_id, $notification, $order->id);
                            ### send push notification to customer ###


                            // end send order in queue for rider assignment after 10 mins
                            \Session::flash('success', __('admin/flash_msg.success.orderInQueue'));
                        }
                        DB::commit();
                        return \Redirect::back();
                    } else {
                        return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.UnableToAssignRider'));
                    }
                } else {
                    //
                    return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.UnableToAssignRider'));
                }
            } else {
                //
                return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.orderNotFound'));
            }
        } catch (\Exception $e) {
            DB::rollBack();
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }





    /**
     * Author: Jaidev
     * Email: jaidev@nmgtechnologies.com
     * @Method: cancel()
     * @Scope: public
     * @params: @id
     * @Description: cancel order by restaurant
     * @Created 04/apr/2022
     * @Updated 04/apr/2022
     */
    public function cancel($id)
    {
        try {
            $id = \Crypt::decryptString($id);
            $order = Order::find($id);
            // dd($order);
            if (!empty($order)) {
                //
                if ($order->order_status_id != config('admin.constants.ORDER_STATUS_COMPLETED') && $order->order_status_id != config('admin.constants.ORDER_STATUS_CANCELED')) {
                    $order->order_status_id = config('admin.constants.ORDER_STATUS_CANCELED');
                    if ($order->save()) {
                        // order reject mail to customer
                        $userData = User::find($order->user_id);
                        if (!empty($userData->email)) {
                            if (!empty($input['lang'])) {
                                \App::setLocale($input['lang']);
                            }
                            $params['replaceKeywords']['{USER_NAME}'] = $userData->first_name . ' ' . $userData->last_name;
                            $params['toEmail'] = $userData->email;
                            $params['emailSlug'] = 'customer_order_rejected_restaurant';
                            $this->customMailer($params);
                        }

                        ### send push notification to customer ###
                        $title = "Order cancelled by restaurant";
                        $body = "Your order has been cancelled by restaurant";
                        $notification = array('title' => $title, 'body' => $body, 'sound' => 'default', 'badge' => '1', 'actionId' => config('api.constants.push_notification.actionId.customer_order_details'), 'orderId' => $order->id);
                        PushNotification::sendNotification($order->user_id, $notification, $order->id);
                        ### send push notification to customer ###




                        OrderLog::saveLog($order->id, $order->order_status_id);

                        $notificationDB['title'] = 'Order cancelled';
                        $notificationDB['description'] = 'Order no #' . $order->id . ' has been cancelled by restaurant';
                        $notificationDB['order_id'] = $order->id;
                        $notificationDB['user_id'] = Auth::guard('restaurant')->user()->id;
                        Notification::create($notificationDB);


                        //
                        if (!empty($order->delivery_person_id)) {

                            $currentDP = \app\Model\Admin\DeliveryPersons::where(['id' => $order->delivery_person_id])->first();
                            $currentDP->work_status = 0;
                            $currentDP->save();


                            $currentRiderAssignment = \app\Model\Vendor\DeliveryPersonAssignment::where(['order_id' => $order->id])->latest()->first();

                            #### send push notification to rider if assignment status is only accepted ##
                            if ($currentRiderAssignment->status == 2) {
                                $title = "Order cancelled by restaurant";
                                $body = "Order has been cancelled by restaurant";
                                $notification = array('title' => $title, 'body' => $body, 'sound' => 'default', 'badge' => '1', 'actionId' => config('api.constants.push_notification.actionId.rider_order_details'), 'orderId' => $order->id);
                                PushNotification::sendNotification($currentDP->user_id, $notification, $order->id);
                            }
                            ### send push notification to rider ###
                        }




                        \Session::flash('success', __('admin/flash_msg.success.orderCanceled'));
                        return \Redirect::back();
                    } else {
                        return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.unableToChangeOrderStatus'));
                    }
                } else {
                    //
                    return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.orderNotCancellable'));
                }
            } else {
                //
                return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.orderNotFound'));
            }
        } catch (\Exception $e) {
            dd($e);
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }
}
