<?php

namespace app\Http\Controllers;

use app\Traits\FilePathUrl;
use Jenssegers\Agent\Agent;
use Illuminate\Http\Request;
use app\Model\Admin\ErrorLog;
use app\Model\Api\V1\EmailTemplate;
use app\Mail\Api\V1\SendEmailTemplate;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Illuminate\Foundation\Bus\DispatchesJobs;
use Illuminate\Routing\Controller as BaseController;
use Illuminate\Foundation\Validation\ValidatesRequests;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;

class Controller extends BaseController
{

    use AuthorizesRequests, DispatchesJobs, ValidatesRequests, FilePathUrl;

    /**
     * @Method: saveErrorLog()
     * @Scope: public
     * @Params: exception object
     * @returns: true / false
     * @Description: This method is used track the error/exception and store them in database `error_logs` table.
     * @Created 17/July/2021
     * @Updated 17/July/2021
     */
    public function saveErrorLog($e)
    {
        $request = new Request();
        $agent = new Agent();
        $data = [
            'message' => $e->getMessage(),
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'browser' => $agent->browser(),
            'os' => $agent->platform(),
            'referer' => $agent->device(),
            'created' => date('Y-m-d H:i:s'),
        ];
        Log::error($data);
        if (!ErrorLog::create($data)) {
            return false;
        }
        return true;
    }

    /**
     * Author: NMG
     * Function: customMailer
     * Description: Used for sending emails as per different slugs provided as input
     * Input: ['emailSlug', 'toEmail', 'fromEmail', 'fromName', 'replaceKeywords']
     * Output: return true if email sent successfully
     * Dated: 17/July/2021
     */
    public function customMailer($params = [])
    {
        
        info($params);
        if (empty($params['emailSlug'])) {
            return false;
        }
        if (!empty($params['content']) && !empty($params['subject'])) {
            $template['content'] = $params['content'];
            $template['subject'] = $params['subject'];
            $template = (object) $template;
        } else {
            $template = EmailTemplate::fetchEmailTemplateBySlug($params['emailSlug']);
            
        }
        if (!empty($template)) {
            $content = $template->content;
            $content = str_replace('../../public/storage/uploads/', config('admin.path.BASE_URL') . 'public/storage/uploads/', $content);
            $params['emailTemplate'] = $content;
            $params['subject'] = $template->subject;

            $params['to'] = !empty($params['toEmail']) ? $params['toEmail'] : env('SITE_ADMINISTRATOR_EMAIL');
            $params['from'] = (!empty($this->params['fromEmail'])) ? $this->params['fromEmail'] : env('MAIL_FROM_ADDRESS');
            $params['fromName'] = (!empty($this->params['fromName'])) ? $this->params['fromName'] : env('MAIL_FROM_NAME');

            if (!empty($params['noReply'])) {
                $params['from'] = env('MAIL_FROM_ADDRESS');
            }
            // dd($template);

            $body = str_replace('{SITE_NAME}', env('APP_NAME'), $params['emailTemplate']);

            if (!empty($params['replaceKeywords'])) {
                foreach ($params['replaceKeywords'] as $keyword => $replaceValue) {
                    $body = str_replace($keyword, $replaceValue, $body);
                }
            }
            $params['emailTemplate'] = $body;
            //print_r($params);//die;
            $sendEmailTemplate = new SendEmailTemplate($params);
            $send = Mail::to($params['to'])->send($sendEmailTemplate);
            // check for failures
            if (Mail::failures()) {
                return false;
            }
            return true;
        }
        return false;
    }

    /**
     * Author: NMG
     * Function: encryptDecryptId
     * Description:encrypt/decrypt any string as per the action provided
     * Input: ['action(encrypt/decrypt)', 'string']
     * Output: return encrypted/decrypted string back
     * Dated: 17/July/2021
     */
    public static function encryptDecryptId($action, $string)
    {
        $output = false;
        $encryptMethod = "AES-256-CBC";
        $secretKey = config('api.constants.httpCodes.ENCRYPT_SECRET_KEY');
        $secret_iv = config('api.constants.httpCodes.ENCRYPT_SECRET_KEY_iv');

        //hash
        $key = hash('sha256', $secretKey);

        //iv - encrypt method AES-256-CBC expects 16 bytes - else you will get a warning
        $iv = substr(hash('sha256', $secret_iv), 0, 16);

        if ($action == 'encrypt') {
            $output = openssl_encrypt($string, $encryptMethod, $key, 0, $iv);
            $output = base64_encode($output);
        } else if ($action == 'decrypt') {
            $output = openssl_decrypt(base64_decode($string), $encryptMethod, $key, 0, $iv);
        }

        return $output;
    }

    /**
     * Author: NMG
     * Function: apiEncryption
     * Description:used for api request/response encrypt/decrypt
     * Input: ['action(encrypt/decrypt)', 'string']
     * Output: return encrypted/decrypted string back
     * Dated: 17/July/2021
     */
    public static function apiEncryption($action, $string = null)
    {
        try {
            $output = false;
            $encryptMethod = "AES-256-CBC";
            $secretKey = env('API_ENCRYPTION_KEY');

            if ($action == 'encrypt') {
                $secret_iv = hash('sha256', $secretKey);
                //iv - encrypt method AES-256-CBC expects 16 bytes - else you will get a warning
                $iv = substr($secret_iv, 0, 16);
                $output = @openssl_encrypt($string, $encryptMethod, $secretKey, 1, $iv);
                $output = base64_encode($output);
                $output = $iv . $output;

            } else if ($action == 'decrypt') {
                $search = ["%2F"];
                $replace = ["/"];

                $string = str_replace($search, $replace, $string);
                $iv = substr($string, 0, 16);
                $string = substr($string, 16);
                $string = base64_decode($string);
                $output = openssl_decrypt($string, $encryptMethod, $secretKey, 1, $iv);
            }

            return $output;
        } catch (\Exception $e) {
            throw new \Exception(__('api/common.EXCEPTION.ENCRYPTION.message'), (int) __('api/common.EXCEPTION.ENCRYPTION.code'));
        }
    }

    /**
     * @Method: rrmdir()
     * @Scope: public
     * @Params: directory path
     * @Description: Use to delete a directory & all it's files recursively
     * @Created 17/July/2021
     * @Updated 17/July/2021
     */
    public function rrmdir($dir)
    {
        if (is_dir($dir)) {
            $objects = scandir($dir);
            foreach ($objects as $object) {
                if ($object != "." && $object != "..") {
                    if (is_dir($dir . "/" . $object)) {
                        $this->rrmdir($dir . "/" . $object);
                    } else {
                        if (!unlink($dir . "/" . $object)) {
                            return false;
                        }
                    }
                }
            }
            if (!rmdir($dir)) {
                return false;
            }
        }
        return true;
    }

    /**
     * @Method: randomPassword()
     * @Scope: public
     * @Params: random string length
     * @Description: Use to generate unique password
     * @Created 17/July/2021
     * @Updated 17/July/2021
     */
    public function randomPassword($len = null)
    {
        //generates a random password of length minimum 8
        //contains at least one lower case letter, one upper case letter,
        // one number and one special character,
        //not including ambiguous characters like iIl|1 0oO
        //enforce min length 8
        if (empty($len) || $len < config('api.constants.user.password.minLength')) {
            $len = config('api.constants.user.password.minLength');
        }

        //define character libraries - remove ambiguous characters like iIl|1 0oO
        $sets = array();
        $sets[] = 'ABCDEFGHJKLMNPQRSTUVWXYZ';
        $sets[] = 'abcdefghjkmnpqrstuvwxyz';
        $sets[] = '23456789';
        $sets[] = '!@#$%&*(){}[]/?';

        $password = '';

        //append a character from each set - gets first 4 characters
        foreach ($sets as $set) {
            $password .= $set[array_rand(str_split($set))];
        }

        //use all characters to fill up to $len
        while (strlen($password) < $len) {
            //get a random set
            $randomSet = $sets[array_rand($sets)];

            //add a random char from the random set
            $password .= $randomSet[array_rand(str_split($randomSet))];
        }

        //shuffle the password string before returning!
        return str_shuffle($password);
    }

    /**
     * @Method: execCommand()
     * @Scope: public
     * @Params: bake command to execute in background
     * @Description: this function will execute the console command in application background
     * @Created 17/July/2021
     * @Updated 17/July/2021
     */
    public function execCommand($bakeCommand = null)
    {
        if (empty($bakeCommand)) {
            return false;
        }
        $root = base_path() . '/';
        //$bakeCommand = "php artisan email:send 1 --queue";
        $cmd = "cd " . $root . " && " . $bakeCommand;

        $output = exec($cmd . " > /dev/null &"); // do not wait
        //$output = exec($cmd . " 2>&1", $output);//wait for the response
    }
    
    /**
     * Author: NMG
     * Function: generatePIN
     * Description:generate random n number digit long pin
     * Input: ['number)', 'Int']
     * Output: return random n length number
     * Dated: 23/June/2021
     */
    public static function generatePIN($digits) {

        $pin = rand(pow(10, $digits-1), pow(10, $digits)-1);
        return (string) $pin;
    }
}
