<?php

namespace app\Http\Controllers\Api\V1;

use app\Http\Controllers\Api\V1\BaseController;
use app\Model\Api\V1\Messages;
use app\Model\Api\V1\Rider;
use app\Model\Api\V1\User;
use app\Rules\Api\ValidatePassword;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Twilio\Rest\Client;

class UserController extends BaseController
{
    /**
     * Author: NMG
     * Function: forgotPassword
     * Description: User can retrive it's password using otp  and mobile
     * Input: email
     * Output: Otp has been sent on user's registered mobile
     * Dated:25/Apr/2021
     * UPDATED :19/JULY/2021
     */
    public function forgotPassword(Request $request)
    {

        try {
            if ($request->isMethod('post')) {
                if ($request->isJson()) {
                    $input = $request->json()->all();
                } else {
                    $input = $request->all();
                }
                if (!empty($input['request'])) {
                    //encrypted input request
                    $encrypted = $input['request'];
                    $decrypted = $this->apiEncryption('decrypt', $encrypted);
                    $input     = (array) json_decode($decrypted);
                } else {
                    $input = $request->only(['phone']);
                }

                // server side validations
                $validator = Validator::make($input, User::forgotPasswordRules(), User::messages());
                if ($validator->fails()) {
                    $this->code    = config('api.constants.httpCodes.ERROR');
                    $this->message = $validator->errors()->first();
                } else {
                    $user = User::where([['phone_number', '=', $input['phone']], ['role_id', '=', config('admin.constants.PUBLIC_USERS_ROLE')]])
                        ->select('id', 'email', 'phone_number', 'first_name', 'last_name', 'status')->first();
                    if (empty($user)) {
                        $this->code    = config('api.constants.httpCodes.ERROR');
                        $this->message = __('api/validation/user.phone.exists');
                    } else {
                        if ($user->status == 1 || $user->phone_number_verified == 0) {
                            $rollBack = false;
                            $otp      = $this->generatePIN(4);
                            \DB::beginTransaction();
                            $user->verification_otp = $otp;
                            if ($user->save()) {
                                //send sms to user
                                $twilio  = new Client(env('TWILIO_ACCOUNT_SID'), env('TWILIO_SECRET_KEY'));
                                $sendSms = $twilio->messages->create(
                                    config('admin.constants.COUNTRY_CODE') . $user->phone_number,
                                    [
                                        "body" => "Dear User,\n {$otp} is your OTP for resetting your password. Please do not share it with anyone.\nRegards,\nTeam BMF",
                                        "from" => env('TWILIO_TRIAL_NUMBER'),
                                    ]
                                );
                                // dd($user);
                                //check if SMS goes to user then add sendEmail true or false
                                if ($twilio) {
                                    $this->dataList['sendSms'] = true;
                                } else {
                                    $this->dataList['sendSms'] = false;
                                }
                                ###Reset Password Email#####
                            }
                            if (!$rollBack) {
                                \DB::commit();
                                $this->code    = config('api.constants.httpCodes.SUCCESS');
                                $this->message = __('api/validation/user.success.otpSendMobile');
                            } else {
                                \DB::rollBack();
                                $this->code    = config('api.constants.httpCodes.ERROR');
                                $this->message = __('api/validation/user.error.reset');
                            }
                        } else {
                            $this->code    = config('api.constants.httpCodes.ERROR');
                            $this->message = __('api/validation/user.error.blocked');
                        }
                    }
                }
                return $this->sendResponse();
            }
        } catch (\Exception $exception) {
            $this->saveErrorLog($exception);
            return $this->sendError($exception);
        }
    }
    /**
     * Author: NMG
     * Function: resetPassword
     * Description: User can reset password following the link sent to there email
     * Input: 'token', 'new_password', 'new_password_confirmation'
     * Output: jsone response
     * Dated: 05/September/2019
     */
    public function resetPassword(Request $request)
    {
        try {
            if ($request->isMethod('post')) {
                if ($request->isJson()) {
                    $input = $request->json()->all();
                } else {
                    $input = $request->all();
                }
                $input      = $request->only(['otp', 'phone', 'password', 'password_confirmation']);
                $validation = Validator::make($input, User::resetPasswordRules(), User::messages());
                if ($validation->fails()) {
                    $this->code    = config('api.constants.httpCodes.ERROR');
                    $this->message = $validation->errors()->first();
                } else {
                    //Registerd user
                    $user = DB::table('users')->where('phone_number', $request->phone)->first();
                    if (!empty($user->id)) {
                        $otp = $user->verification_otp;
                        if ($otp == $request->otp) {
                            $password['password'] = bcrypt($input['password']);
                            User::where('id', $user->id)->update($password);

                            // send change password email to user
                            if (!empty($user->email)) {
                                ########      change password successfully mail to user   ###########
                                if (!empty($input['lang'])) {
                                    \App::setLocale($input['lang']);
                                }
                                $params['replaceKeywords']['{USER_NAME}'] = $user->first_name . ' ' . $user->last_name;
                                $params['toEmail']                        = $user->email;
                                $params['emailSlug']                      = 'app_or_restaurant_changed_password_successfully';
                                $this->customMailer($params);
                                ######## End Change Email OTP on previous email id ###########
                            }
                            $this->httpCode  = config('api.constants.httpCodes.SUCCESS_WITH_RESPONSE_BODY');
                            $this->apiStatus = config('api.constants.httpCodes.SUCCESS');
                            $this->message   = __('api/validation/user.success.changePassword');
                        } else {
                            $this->httpCode  = config('api.constants.httpCodes.SUCCESS_WITH_RESPONSE_BODY');
                            $this->apiStatus = config('api.constants.httpCodes.ERROR');
                            $this->code      = __('api/validation/user.otp.code');
                            $this->message   = __('api/validation/user.error.wrong_otp');
                        }
                    } else {
                        // invalid user
                        $this->code    = config('api.constants.httpCodes.ERROR');
                        $this->message = __('api/validation/user.email.exists');
                    }
                }
                return $this->sendResponse();
            }
        } catch (\Exception $exception) {
            $this->saveErrorLog($exception);
            $this->httpCode = config('api.constants.httpCodes.SERVER_ERROR');
            $this->code     = config('api.constants.httpCodes.SERVER_ERROR');
            $this->message  = __('api/common.SERVER_ERROR');
            return $this->sendError();
        }
    }
    /**
     * Author: NMG
     * Function: change mobile
     * Description: User can change mobile using otp
     * Input:  mobile_no
     * Output: Otp has been sent on user's registered mobile
     * Dated:25/Apr/2021
     */
    public function changeMobile(Request $request)
    {
        try {
            if ($request->isMethod('post')) {
                $input = $request->only(['mobile_no']);
                $validator = Validator::make($input, User::changemobileRules(), User::messages());
                if ($validator->fails()) {
                    $this->code    = config('api.constants.httpCodes.ERROR');
                    $this->message = $validator->errors()->first();
                    return $this->sendResponse();
                } else {
                    $user = $user = Auth::user();
                    if (empty($user)) {
                        $this->code    = config('api.constants.httpCodes.ERROR');
                        $this->message = __('api/validation/user.mobile_no.exists');
                    } else {
                        if ($user->status == 1) {
                            $rollBack = false;
                            $otp      = rand(1000, 9999);
                            \DB::beginTransaction();
                            $user->verification_otp = $otp;
                            $user->verification_otp_sent_at = now();
                            $user->tmp_phone_number = $input['mobile_no'];
                            if ($user->save()) {

                                //send sms to user
                                $twilio = new Client(env('TWILIO_ACCOUNT_SID'), env('TWILIO_SECRET_KEY'));
                                $sendSms = $twilio->messages->create(
                                    '+' . $user->country_code . $user->tmp_phone_number,
                                    [
                                        "body" => "Dear User,\n{$otp} is your OTP for updating your mobile number. Please do not share it with anyone.\nRegards,\nTeam BMF",
                                        "from" => env('TWILIO_TRIAL_NUMBER')
                                    ]
                                );
                                //check if SMS goes to user then add sendEmail true or false
                                if ($twilio) {
                                    $this->dataList['sendSms'] = true;
                                } else {
                                    $this->dataList['sendSms'] = false;
                                }
                            }
                            if (!$rollBack) {
                                \DB::commit();
                                $this->code    = config('api.constants.httpCodes.SUCCESS');
                                $this->message = __('api/validation/user.success.otpSend');
                            } else {
                                \DB::rollBack();
                                $this->code    = config('api.constants.httpCodes.ERROR');
                                $this->message = __('api/validation/user.error.reset');
                            }
                        } else {
                            $this->code    = config('api.constants.httpCodes.FORBIDDEN');
                            $this->message = __('api/validation/user.error.blocked');
                        }
                    }
                }
                return $this->sendResponse();
            }
        } catch (\Exception $exception) {
            $this->saveErrorLog($exception);
            return $this->sendError($exception);
        }
    }
    /**
     * Author: NMG
     * Function: updateMobile
     * Description: User can mobile using otp
     * Input: 'otp', 'mobile_no'
     * Output: jsone response
     * Dated: 26/Apr/2021
     */
    public function updateMobile(Request $request)
    {
        try {
            if ($request->isMethod('post')) {
                $input      = $request->only(['otp', 'mobile_no']);
                $validation = Validator::make($input, User::verifyMobile('tmp_phone_number'), User::messages());
                if ($validation->fails()) {
                    $errorCode      = key(current($validation->errors()));
                    $this->httpCode = config('api.constants.httpCodes.SUCCESS_WITH_RESPONSE_BODY');
                    $this->code     = config('api.constants.httpCodes.ERROR');
                    $this->message  = $validation->errors()->first();
                    return $this->sendResponse();
                }
            }
            //Registerd user
            $user = Auth::user();
            if ($user->status == 0) {
                $this->code = config('api.constants.httpCodes.FORBIDDEN');
                $this->message = __('api/validation/user.error.blocked');
                return $this->sendResponse();
            }
            if ($user->verification_otp == $request->otp && $request->mobile_no == $user->tmp_phone_number) {
                $mobile['phone_number'] = $request->mobile_no;
                $mobile['phone_number_verified'] = 1;
                User::where('id', $user->id)->update($mobile);

                User::where('tmp_phone_number', $request->mobile_no)->update(['tmp_phone_number' => null]);


                $this->httpCode  = config('api.constants.httpCodes.SUCCESS_WITH_RESPONSE_BODY');
                $this->apiStatus = config('api.constants.httpCodes.SUCCESS');
                $this->message   = __('api/validation/user.success.mobile_updated');
            } else {
                $this->httpCode  = config('api.constants.httpCodes.ERROR');
                $this->code    = config('api.constants.httpCodes.ERROR');
                $this->message   = __('api/validation/user.error.email_otp_error');
            }
            return $this->sendResponse();
        } catch (\Exception $exception) {
            $this->saveErrorLog($exception);
            $this->httpCode = config('api.constants.httpCodes.SERVER_ERROR');
            $this->code     = config('api.constants.httpCodes.SERVER_ERROR');
            $this->message  = __('api/common.SERVER_ERROR');
            return $this->sendError();
        }
    }
    /**
     * Author: NMG
     * Function: show
     * Description: it will return back auth/user details
     * Input: headers[Authorization, Accept, Content-type] with access_token & application/json
     * Output: json response of user details
     * Dated: 12/Dec/2019
     * Updated: 13/Dec/2019
     */
    public function show(Request $request, $id = null)
    {
        try {
            if ($request->isMethod('get')) {

                if ($request->user()->status == 0) {
                    $this->code = config('api.constants.httpCodes.FORBIDDEN');
                    $this->message = __('api/validation/user.error.blocked');
                    return $this->sendResponse();
                }

                $user_id = $id ?? $request->user()->id;
                if (!empty($user_id)) {
                    $profileData = User::userProfileInfo($user_id);
                    if (!empty($profileData)) {
                        $this->dataList = $profileData;
                        $this->code     = config('api.constants.httpCodes.SUCCESS');
                        $this->message  = __('api/validation/user.success.profile');
                    } else {
                        $this->code    = config('api.constants.httpCodes.ERROR');
                        $this->message = __('api/validation/user.error.profile');
                    }
                } else {
                    $this->code    = config('api.constants.httpCodes.ERROR');
                    $this->message = __('api/validation/user.error.profile');
                }
                return $this->sendResponse();
            }
        } catch (\Exception $exception) {
            return $this->processError($exception);
        }
    }

    /**
     * Author: NMG
     * Function: editProfile
     * Description: it will return back auth/user details
     * Input: update user details ['user_name', 'first_name', 'last_name', 'department', 'title']
     * Output: json response of updated user details
     * Dated: 17/Dec/2019
     * Updated: 17/Dec/2019
     */
    public function editProfile(Request $request)
    {
        try {
            if ($request->isMethod('post')) {

                $input = $request->only('fullName', 'userImage');
                $user  = Auth::user();
                if ($user->status == 0) {
                    $this->code = config('api.constants.httpCodes.FORBIDDEN');
                    $this->message = __('api/validation/user.error.blocked');
                    return $this->sendResponse();
                }
                \DB::beginTransaction();
                if (!empty($user->id)) {
                    $updateData = [];
                    $rollBack   = false;

                    if (!empty($input['fullName'])) {
                        // explode fullName
                        $fullName = explode(' ', $input['fullName']);
                        if (count($fullName) > 0) {
                            $updateData['first_name'] = $fullName[0];
                            unset($fullName[0]);
                            $updateData['last_name'] = implode(' ', $fullName);
                        } else {
                            $updateData['first_name'] = $input['fullName'];
                        }
                    }


                    if (!empty($input['userImage'])) {
                        $updateData['profile_image'] = $input['userImage'];
                    }
                    if (User::where('id', $user->id)->update($updateData)) {
                        $this->httpCode = config('api.constants.httpCodes.SUCCESS_WITH_RESPONSE_BODY');
                        $this->code     = config('api.constants.httpCodes.SUCCESS');
                        $this->message  = __('api/validation/user.success.profileUpdated');
                    } else {
                        $rollBack       = true;
                        $this->httpCode = config('api.constants.httpCodes.SUCCESS_WITH_RESPONSE_BODY');
                        $this->code     = config('api.constants.httpCodes.ERROR');
                        $this->message  = __('api/validation/user.error.profileUpdated');
                    }
                    if (!$rollBack) {
                        \DB::commit();
                        $profileData = User::userProfileInfo($user->id);

                        $this->dataList = $profileData;
                    }
                } else {
                    \DB::rollBack();
                    $this->code     = config('api.constants.httpCodes.ERROR');
                    $this->message  = __('api/validation/user.success.profileUpdated');
                }
                return $this->sendResponse();
            }
        } catch (\Exception $exception) {
            return $this->processError($exception);
        }
    }

    /**
     * Author: NMG
     * Function: changeAvatar
     * Description: it will return back auth/user details
     * Input: update user vatar image
     * Output: json response of updated user details
     * Dated: 17/Dec/2019
     * Updated: 17/Dec/2019
     */
    public function changeAvatar(Request $request)
    {
        try {
            if ($request->isMethod('post')) {
                if ($request->isJson()) {
                    $input = $request->json()->all();
                } else {
                    //$input = $request->getContent();
                    $input = $request->all();
                }
                $input = $request->only('avatar');

                //server side validations
                $validation = Validator::make($input, User::apiAvatarRules(), User::messages());
                if ($validation->fails()) {
                    $errorCode     = key(current($validation->errors()));
                    $this->code    = config('api.constants.httpCodes.ERROR');
                    $this->message = $validation->errors()->first();
                } else {
                    $user = Auth::user();
                    if (!empty($user->id)) {
                        if ($request->hasFile('avatar')) {
                            if (\app\Model\Admin\User::upload_avatar($request->avatar, $user->id)) {
                                $this->dataList = User::userProfileInfo($user->id);
                                $this->code     = config('api.constants.httpCodes.SUCCESS');
                                $this->message  = __('api/validation/user.success.avatar');
                            } else {
                                $this->code    = config('api.constants.httpCodes.ERROR');
                                $this->message = __('api/validation/user.error.avatar');
                            }
                        }
                    }
                }

                return $this->sendResponse();
            }
        } catch (\Exception $exception) {
            $this->saveErrorLog($exception);
            $this->httpCode = config('api.constants.httpCodes.SERVER_ERROR');
            $this->code     = config('api.constants.httpCodes.SERVER_ERROR');
            $this->message  = __('api/common.SERVER_ERROR');
            return $this->sendError();
        }
    }

    /**
     * Author: NMG
     * Function: changePassword
     * Description: Loggedin user can change it's current password
     * Input: ['password', 'new_password']
     * Output: json response of password change
     * Dated: 17/Dec/2019
     */
    public function changePassword(Request $request)
    {
        try {
            if ($request->isMethod('post')) {
                if ($request->isJson()) {
                    $input = $request->json()->all();
                } else {
                    //$input = $request->getContent();
                    $input = $request->all();
                }
                $user = Auth::user();
                if ($user->status == 0) {
                    $this->code = config('api.constants.httpCodes.FORBIDDEN');
                    $this->message = __('api/validation/user.error.blocked');
                    return $this->sendResponse();
                }
                if (!empty($input['request'])) {
                    //encrypted input request
                    $encrypted = $input['request'];
                    $decrypted = $this->apiEncryption('decrypt', $encrypted);
                    $input     = (array) json_decode($decrypted);
                } else {
                    $input = $request->only(['password', 'newPassword']);
                }

                //server side validations
                $validation = Validator::make($input, User::changePasswordRules($user), User::messages());
                if ($validation->fails()) {
                    $this->httpCode = config('api.constants.httpCodes.SUCCESS');
                    $this->code     = config('api.constants.httpCodes.ERROR');
                    $this->message  = $validation->errors()->first();
                } else {
                    if (User::where('id', $user->id)->update(['password' => Hash::make($input['newPassword'])])) {
                        ###Change Password Mail#####
                        if (!empty($user->email)) {

                            if (!empty($input['lang'])) {
                                \App::setLocale($input['lang']);
                            }
                            $params['replaceKeywords']['{USER_NAME}'] = trim($user->first_name . ' ' . $user->last_name);
                            $params['toEmail']                        = $user->email;
                            $params['emailSlug']                      = 'change_password';
                            $this->customMailer($params);
                            #######################

                        }
                        $this->code    = config('api.constants.httpCodes.SUCCESS');
                        $this->message = __('api/validation/user.success.changePassword');
                    } else {
                        $this->code    = config('api.constants.httpCodes.ERROR');
                        $this->message = __('api/validation/user.error.changePassword');
                    }
                }

                return $this->sendResponse();
            }
        } catch (\Exception $exception) {
            // dd($exception);
            $this->saveErrorLog($exception);
            $this->httpCode = config('api.constants.httpCodes.SERVER_ERROR');
            $this->code     = config('api.constants.httpCodes.SERVER_ERROR');
            $this->message  = __('api/common.SERVER_ERROR');
            return $this->sendError();
        }
    }

    /**
     * Author: NMG
     * Function: send otp to user
     * Description: User can retrive it's otp
     * Input: email
     * Output: New Otp has been sent on user's registered email/phone
     * Dated: 02/June/2020
     */

    public function sendOtp(Request $request)
    {
        try {
            if ($request->isMethod('POST')) {
                if ($request->isJson()) {
                    $input = $request->json()->all();
                } else {
                    $input = $request->all();
                }
                $input       = $request->only('email');
                $userDetails = DB::table('users')->select('id', 'mobile_no', 'email', 'country_id')->where('email', $request->email)->first();
                if (!empty($userDetails->id)) {
                    if ($userDetails->email == $request->email) {
                        $user = User::find($userDetails->id);
                        if ($user->status != 0) {
                            $otp                           = $this->generatePIN(4);
                            $user->verification_otp        = (int) $otp;
                            $user->verification_otp_set_at = Carbon::now();
                            $user->save();
                            $this->httpCode = config('api.constants.httpCodes.SUCCESS_WITH_RESPONSE_BODY');
                            $this->code     = config('api.constants.httpCodes.SUCCESS');
                            $this->message  = __('api/validation/user.success.otpSend');
                        } else {
                            $this->httpCode = config('api.constants.httpCodes.SUCCESS_WITH_RESPONSE_BODY');
                            $this->code     = config('api.constants.httpCodes.FORBIDDEN');
                            $this->message  = __('api/validation/user.error.blocked');
                        }
                    } else {
                        $this->httpCode = config('api.constants.httpCodes.SUCCESS_WITH_RESPONSE_BODY');
                        $this->code     = config('api.constants.httpCodes.ERROR');
                        $this->message  = __('api/validation/user.error.userNotFound');
                    }

                    //send sms to user
                    $twilio  = new Client(env('TWILIO_ACCOUNT_SID'), env('TWILIO_SECRET_KEY'));
                    $sendSms = $twilio->messages->create(
                        '+' . $userDetails->country_id . $userDetails->mobile_no,
                        [
                            "body" => "You have requested for OTP, you otp is {$otp}",
                            "from" => env('TWILIO_TRIAL_NUMBER'),
                        ]
                    );
                    //check if SMS goes to user then add sendEmail true or false
                    if ($twilio) {
                        $this->dataList['sendSms'] = true;
                    } else {
                        $this->dataList['sendSms'] = false;
                    }
                    // sending email to user
                    $full_name                                = User::fullName($user->id)['user_name'];
                    $params['replaceKeywords']['{USER_NAME}'] = $full_name;
                    $params['replaceKeywords']['{OTP}']       = $user->verification_otp;
                    $params['toEmail']                        = $user->email;
                    $params['emailSlug']                      = 'send_otp';
                    $mail                                     = $this->customMailer($params);

                    //check if mail goes then add sendEmail true or false
                    if ($mail) {
                        $this->dataList['sendEmail'] = true;
                    } else {
                        $this->dataList['sendEmail'] = false;
                    }
                } else {
                    $this->httpCode = config('api.constants.httpCodes.SUCCESS_WITH_RESPONSE_BODY');
                    $this->code     = config('api.constants.httpCodes.ERROR');
                    $this->message  = __('api/validation/user.error.userNotFound');
                }
                return $this->sendResponse();
            }
        } catch (\Exception $exception) {
            //dd($exception);
            $this->saveErrorLog($exception);
            $this->httpCode = config('api.constants.httpCodes.SERVER_ERROR');
            $this->code     = config('api.constants.httpCodes.SERVER_ERROR');
            $this->message  = __('api/common.SERVER_ERROR');
            return $this->sendError();
        }
    }

    /**
     * Author: NMG
     * Function: VerifyOTP
     * Description: User can verify his otp through it
     * Input: otp, email
     * Output: OTP verified
     * Dated: 02/June/2020
     */

    public function verifyOtp(Request $request)
    {
        try {
            if ($request->isMethod('post')) {
                if ($request->isJson()) {
                    $input = $request->json()->all();
                } else {
                    $input = $request->all();
                }
                $input     = $request->only('otp', 'email');
                $validator = Validator::make($input, User::verifyOtp());
                if ($validator->fails()) {
                    $errorCode       = key(current($validator->errors()));
                    $this->apiStatus = __("api.constants.httpCodes.ERROR");
                    $this->message   = $validator->errors()->first();
                    return $this->sendResponse();
                }
                $user = DB::table('users')->where('verification_otp', $request->otp)->first();
                if (!empty($user->id)) {
                    $otp = $user->verification_otp;
                    if ($user->email == $request->email) {
                        //code if necessary
                    } else {
                        $this->httpCode  = config('api.constants.httpCodes.SUCCESS_WITH_RESPONSE_BODY');
                        $this->apiStatus = config('api.constants.httpCodes.SUCCESS');
                        $this->message   = __('api/validation/user.error.mwList');
                    }
                    if ($otp == $request->otp) {
                        $verification['is_verified'] = true;
                        User::where('id', $user->id)->update($verification);
                        $this->httpCode  = config('api.constants.httpCodes.SUCCESS_WITH_RESPONSE_BODY');
                        $this->apiStatus = config('api.constants.httpCodes.SUCCESS');
                        $this->message   = __('api/validation/user.success.otp_verified');
                    } else {
                        $this->httpCode  = config('api.constants.httpCodes.SUCCESS_WITH_RESPONSE_BODY');
                        $this->apiStatus = config('api.constants.httpCodes.ERROR');
                        $this->message   = __('api/validation/user.error.mwList');
                    }
                } else {
                    $this->httpCode  = config('api.constants.httpCodes.SUCCESS_WITH_RESPONSE_BODY');
                    $this->apiStatus = config('api.constants.httpCodes.ERROR');
                    $this->message   = __('api/validation/user.error.mwList');
                }
            }
            return $this->sendResponse();
        } catch (\Exception $exception) {
            $this->saveErrorLog($exception);
            $this->httpCode  = config('api.constants.httpCodes.SUCCESS_WITH_RESPONSE_BODY');
            $this->apiStatus = config('api.constants.httpCodes.ERROR');
            $this->message   = __('api/common.SERVER_ERROR');
            return $this->sendError();
        }
    }

    /**
     * Author: NMG
     * Function: Upload audio clip
     * Description: Upload recorded audio clip to s3 or local storage
     * Input: auth token, toUserId, audio/video file
     * Output: return with uploaded url
     * Dated: 05/June/2020
     */

    public function uploadVideo(Request $request)
    {
        try {
            if ($request->isMethod('post')) {
                if ($request->isJson()) {
                    $input = $request->json()->all();
                } else {
                    $input = $request->all();
                }
            }
            $input     = $request->only('mediaFile', 'id');
            $validator = Validator::make($input, User::videoRules());
            if ($validator->fails()) {
                $errorCode       = key(current($validator->errors()));
                $this->apiStatus = __("api.constants.httpCodes.ERROR");
                $this->message   = $validator->errors()->first();
                return $this->sendResponse();
            }
            $user = Auth::user();
            if (!empty($user->id)) {
                if ($request->hasFile('mediaFile')) {
                    $videoUpload = User::uploadMedia($request->mediaFile, $user->id);
                }
            }
            if (!empty($videoUpload)) {
                $insert = Messages::create([
                    'type'        => $videoUpload['type'],
                    'message'     => $videoUpload['media'],
                    'created_by'  => $user['id'],
                    'created_for' => $request->id,
                ]);
            } else {
                $this->httpCode  = config('api.constants.httpCodes.SUCCESS_WITH_RESPONSE_BODY');
                $this->apiStatus = config('api.constants.httpCodes.SUCCESS');
                $this->message   = __('api/validation/user.error.video_error');
            }
            if ($insert) {
                $media = User::userMedia($user->id);
                if (env('S3_ENABLED') == true) {
                    $this->dataList['media'] = $media;
                } else {
                    $this->dataList['media'] = $media['message'];
                }
                $this->dataList['type'] = $videoUpload['type'];

                $this->httpCode  = config('api.constants.httpCodes.SUCCESS_WITH_RESPONSE_BODY');
                $this->apiStatus = config('api.constants.httpCodes.SUCCESS');
                $this->message   = __('api/validation/user.success.video_uploaded');
            } else {
                $this->httpCode  = config('api.constants.httpCodes.SUCCESS_WITH_RESPONSE_BODY');
                $this->apiStatus = config('api.constants.httpCodes.SUCCESS');
                $this->message   = __('api/validation/user.error.video_error');
            }
            return $this->sendResponse();
        } catch (\Exception $exception) {
            dd($exception);
            $this->saveErrorLog($exception);
            $this->httpCode  = config('api.constants.httpCodes.SUCCESS_WITH_RESPONSE_BODY');
            $this->apiStatus = config('api.constants.httpCodes.ERROR');
            $this->message   = __('api/common.SERVER_ERROR');
            return $this->sendError();
        }
    }
    /**
     * toggle enable/disable notification setting of user.
     *
     * @param Request $request
     * @return void
     */
    public function toggleNotification(Request $request)
    {
        try {
            if ($request->isMethod('put')) {
                $user                      = User::find($request->user()->id);
                if ($user->status == 0) {
                    $this->code = config('api.constants.httpCodes.FORBIDDEN');
                    $this->message = __('api/validation/user.error.blocked');
                    return $this->sendResponse();
                }
                $msg                       = $user->notification_status == 1 ? 'notification-disabled' : 'notification-enabled';
                $user->notification_status = !$user->notification_status;
                if ($user->save()) {
                    $this->code    = config('api.constants.httpCodes.SUCCESS');
                    $this->message = __('api/validation/user.success.' . $msg);
                } else {
                    $this->code    = config('api.constants.httpCodes.ERROR');
                    $this->message = 'Error while toggle notification settings';
                }
                return $this->sendResponse();
            }
        } catch (\Throwable $th) {
            return $this->processError($th);
        }
    }
    /**
     * verify password
     *
     * @param Request $request
     * @return void
     */
    public function verifyPassword(Request $request)
    {
        try {
            if ($request->isMethod('post')) {
                if ($request->user()->status == 1) {

                    $user       = User::find($request->user()->id);
                    $validation = Validator::make($request->all(), ['password' => ['bail', 'required', new ValidatePassword($user)]], User::messages());
                    if ($validation->fails()) {
                        $this->code     = config('api.constants.httpCodes.ERROR');
                        $this->message  = $validation->errors()->first();
                        $this->dataList['status'] = false;
                        return $this->sendResponse();
                    }
                    $this->dataList['status'] = true;
                    $this->code    = config('api.constants.httpCodes.SUCCESS');
                    $this->message = 'success';
                } else {
                    $this->code = config('api.constants.httpCodes.FORBIDDEN');
                    $this->message = __('api/validation/user.error.blocked');
                }
                return $this->sendResponse();
            }
        } catch (\Throwable $th) {
            return $this->processError($th);
        }
    }
    /**
     * change email
     *
     * @param Request $request
     * @return void
     */
    public function changeEmail(Request $request)
    {
        try {
            if ($request->isMethod('post')) {
                $user = User::find($request->user()->id);

                $validation = Validator::make($request->all(), User::changeEmailRules($user), User::messages());
                if ($validation->fails()) {
                    $this->httpCode = config('api.constants.httpCodes.SUCCESS');
                    $this->code     = config('api.constants.httpCodes.ERROR');
                    $this->message  = $validation->errors()->first();
                    return $this->sendResponse();
                }
                if ($user->status == 0) {
                    $this->code = config('api.constants.httpCodes.FORBIDDEN');
                    $this->message = __('api/validation/user.error.blocked');
                    return $this->sendResponse();
                }
                $otp                    = rand(1001, 9999);
                $user->tmp_email        = $request->email;
                $user->verification_otp = $otp;
                $user->verification_otp_sent_at = now();
                // DB::beginTransaction();
                if ($user->save()) {
                    // send otp email
                    $params['replaceKeywords']['{USER_NAME}'] = trim($user->first_name . ' ' . $user->last_name);
                    $params['replaceKeywords']['{OTP}']       = $user->verification_otp;
                    $params['toEmail']                        = $user->tmp_email;
                    $params['emailSlug']                      = 'emailid_varification_otp_by_mobile_or_restaurant';
                    $mail                                     = $this->customMailer($params);

                    // check if mail goes then add sendEmail true or false
                    if ($mail) {
                        $this->dataList['sendEmail'] = true;
                        DB::commit();
                    } else {
                        $this->dataList['sendEmail'] = false;
                        DB::rollback();
                    }
                    $this->code    = config('api.constants.httpCodes.SUCCESS');
                    $this->message = __('api/validation/user.success.otpSend');
                } else {
                    $this->code    = config('api.constants.httpCodes.ERROR');
                    $this->message = 'Error while toggle notification settings';
                }
                return $this->sendResponse();
            }
        } catch (\Throwable $th) {
            return $this->processError($th);
        }
    }
    /**
     * validate otp send to tmp_email and assign them to user
     *
     * @param Request $request
     * @return void
     */
    public function validateEmail(Request $request)
    {
        try {
            if ($request->isMethod('post')) {
                $user = User::find($request->user()->id);
                if ($user->status == 0) {
                    $this->code = config('api.constants.httpCodes.FORBIDDEN');
                    $this->message = __('api/validation/user.error.blocked');
                    return $this->sendResponse();
                }
                $validation = Validator::make($request->all(), User::verifyEmail('tmp_email'), User::messages());
                if ($validation->fails()) {
                    $this->httpCode = config('api.constants.httpCodes.SUCCESS');
                    $this->code     = config('api.constants.httpCodes.ERROR');
                    $this->message  = $validation->errors()->first();
                    return $this->sendResponse();
                }

                if ($user->verification_otp != $request->otp) {
                    $this->code    = config('api.constants.httpCodes.ERROR');
                    $this->message = __('api/validation/user.error.otp_invalid');
                    return $this->sendResponse();
                }
                if (now()->diffInMinutes($user->verification_otp_sent_at) > config('api.constants.EXPIR.MINUTES.EMAIL_OTP')) {
                    $this->code    = config('api.constants.httpCodes.ERROR');
                    $this->message = __('api/validation/user.error.otp_expired');
                    return $this->sendResponse();
                }
                $user->email = $user->tmp_email;
                $user->verification_otp = null;
                $user->verification_otp_sent_at = null;
                $user->email_verified_at = now();
                DB::beginTransaction();
                User::where('tmp_email', $request->email)->update(['tmp_email' => null]);
                if ($user->save()) {
                    DB::commit();
                    $this->code    = config('api.constants.httpCodes.SUCCESS');
                    $this->message = __('api/validation/user.success.email_updated');
                } else {
                    DB::rollback();
                    $this->code    = config('api.constants.httpCodes.ERROR');
                    $this->message = __('api/validation/user.error.email_updated');
                }
                return $this->sendResponse();
            }
        } catch (\Throwable $th) {
            return $this->processError($th);
        }
    }




    /**
     * Author: Jaidev
     * Function: disable profile
     * Description: User can disable his profile
     * Input:  auth token
     * Output: status of account deactivation
     * Dated:12/July/2022
     */
    public function deactivate(Request $request)
    {
        try {
            if ($request->isMethod('put')) {
                $user = $user = Auth::user();
                if (empty($user)) {
                    $this->code    = config('api.constants.httpCodes.ERROR');
                    $this->message = __('api/validation/user.mobile_no.exists');
                } else {
                    if ($user->status == 1) {
                        $user->status = 0;
                        if ($user->save()) {
                            $this->code    = config('api.constants.httpCodes.SUCCESS');
                            $this->message = __('api/validation/user.success.profile_deactivated');
                        } else {
                            $this->code    = config('api.constants.httpCodes.ERROR');
                            $this->message = __('api/validation/user.error.reset');
                        }
                    } else {
                        $this->code    = config('api.constants.httpCodes.FORBIDDEN');
                        $this->message = __('api/validation/user.error.blocked');
                    }
                }
                return $this->sendResponse();
            }
        } catch (\Exception $exception) {
            $this->saveErrorLog($exception);
            return $this->sendError($exception);
        }
    }
}
