<?php

namespace app\Http\Controllers\Api\V1;

use app\Model\Api\V1\User;
use app\Model\Api\V1\Order;
use app\Model\Api\V1\Rider;
use Illuminate\Http\Request;
use app\Helpers\CommonHelper;
use app\Model\Vendor\OrderLog;
use app\Model\Api\V1\Restaurant;
use app\Model\Api\V1\Notification;
use app\Model\Api\V1\RiderEarning;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\App;
use app\Model\Api\V1\DeliveryPerson;
use app\Transformers\OrderTransformer;
use app\Transformers\RiderTransformer;
use Illuminate\Support\Facades\Validator;
use app\Components\Api\V1\PushNotification;
use app\Model\Api\V1\DeliveryPersonAssignment;
use app\Http\Controllers\Api\V1\BaseController;
use app\Model\Vendor\DeliveryPerson as DeliveryPersonVendor;


class RiderController extends BaseController
{
    private $rider, $user;
    /**
     * constructor of class.
     *
     * @param Rider $rider
     */
    public function __construct(DeliveryPerson $rider, User $user)
    {
        $this->rider = $rider;
        $this->user = $user;
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        // dd($request->all());
        return $this->rider->updateOrCreate(
            [
                'user_id' => $request->user()->id,
            ],
            [
                'age'             => $request->age,
                'id_proof'        => $request->id_proof,
                'vehicle_info'    => $request->vehicle_information,
                'driving_license' => $request->driving_licence,
                'bank_name'       => $request->bank,
                'account_number'  => $request->account_number,
                'status'  => 1,
            ]
        );
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * apply to become a rider.
     *
     * @param Request $request
     * @param Rider $rider
     * @return void
     */
    public function applyRider(Request $request)
    {
        try {
            if ($request->isMethod('post')) {
                $user = User::find($request->user()->id);
                if ($user->status == 0) {
                    $this->code = config('api.constants.httpCodes.FORBIDDEN');
                    $this->message = __('api/validation/user.error.blocked');
                    return $this->sendResponse();
                }
                $request->request->add(['is_rider' => $user->is_delivery_person]);
                $validation = Validator::make($request->all(), User::riderProfileRules(), User::messages());
                if ($validation->fails()) {
                    $this->httpCode = config('api.constants.httpCodes.SUCCESS');
                    $this->code     = config('api.constants.httpCodes.ERROR');
                    $this->message  = $validation->errors()->first();
                    return $this->sendResponse();
                }
                $result = $this->store($request);
                if ($result->id) {
                    $this->dataList['rider'] = (new RiderTransformer)->transform($result);
                    $this->code              = config('api.constants.httpCodes.SUCCESS');
                    $this->message           = __('api/validation/user.success.default');
                } else {
                    $this->code    = config('api.constants.httpCodes.ERROR');
                    $this->message = 'Error while updating applying to be a rider';
                }
                return $this->sendResponse();
            }
        } catch (\Throwable $th) {
            return $th;
            return $this->processError($th);
        }
    }
    /**
     * toggle rider status
     *
     * @param Request $request
     * @return void
     */
    public function toggleRiderStatus(Request $request)
    {
        try {
            if ($request->isMethod('put')) {
                $user = $this->user->find($request->user()->id);
                if ($user->status == 0) {
                    $this->code = config('api.constants.httpCodes.FORBIDDEN');
                    $this->message = __('api/validation/user.error.blocked');
                    return $this->sendResponse();
                }
                if ($user->is_delivery_person != 1) {
                    $this->code    = config('api.constants.httpCodes.ERROR');
                    $this->message = 'You are not allowed to this operation';
                    return $this->sendError();
                }
                $user->is_active = (string) !$user->is_active;
                if ($user->save()) {
                    // dd($user);
                    $this->dataList['rider'] = (new RiderTransformer)->transform($user->rider);
                    $this->code              = config('api.constants.httpCodes.SUCCESS');
                    $this->message           = __('api/validation/user.success.default');
                } else {
                    $this->code    = config('api.constants.httpCodes.ERROR');
                    $this->message = 'Error while updating your work status';
                }
                return $this->sendResponse();
            }
        } catch (\Throwable $th) {
            return $this->processError($th);
        }
    }





    /**
     * Author : jaidev
     * Email : jaidev@nmgtechnologies.com
     * Method : validateRiderAccount
     * Desc : check approved status and account status of delivery person
     * Create : 02-sep-2021
     *
     */
    public function validateRiderAccount($userId)
    {
        $riderDetails =  $this->rider->where(['user_id' => $userId])->first();
        // dd($riderDetails);
        if (!empty($riderDetails)) {
            if ($riderDetails->status == 0) {
                // rider account blocked by admin
                $data['httpCode'] = config('api.constants.httpCodes.SUCCESS');
                $data['code']    = config('api.constants.httpCodes.ERROR');
                $data['message'] = 'Your profile is blocked by Admin. Please contact admin to Unblock';
                return $data;
            } else if ($riderDetails->is_approved != 1) {
                // account disapproved
                $data['httpCode']    = config('api.constants.httpCodes.SUCCESS');
                $data['code']    = config('api.constants.httpCodes.ERROR');
                $data['message'] = 'Your rider request is not approved by Admin. Please contact admin';
                return $data;
            } else {
                //
                return true;
            }
        } else {
            $data['httpCode']    = config('api.constants.httpCodes.SUCCESS');
            $data['code']    = config('api.constants.httpCodes.ERROR');
            $data['message'] = 'You are not allowed to perform this operation';
            return $data;
        }
    }


    /**
     * Author : jaidev
     * Email : jaidev@nmgtechnologies.com
     * Method : updateCurrentLocation
     * Desc : update current location of rider
     * Create : 17-Aug-2021
     *
     */
    public function updateCurrentLocation(Request $request)
    {
        try {
            if ($request->isMethod('put')) {
                $user = $this->user->find($request->user()->id);
                $validateResponse =  $this->validateRiderAccount($request->user()->id);
                if ($validateResponse !== true) {
                    $this->code = $validateResponse['code'];
                    $this->message = $validateResponse['message'];
                    $this->httpCode = $validateResponse['httpCode'];
                    return $this->sendResponse();
                }
                if ($user->status == 0) {
                    $this->code = config('api.constants.httpCodes.FORBIDDEN');
                    $this->message = __('api/validation/user.error.blocked');
                    return $this->sendResponse();
                }
                if ($user->is_delivery_person != 1) {
                    $this->httpCode    = config('api.constants.httpCodes.SUCCESS');
                    $this->code    = config('api.constants.httpCodes.ERROR');
                    $this->message = 'You are not allowed to perform this operation';
                    return $this->sendResponse();
                }
                $input = $request->all();
                $validation = Validator::make($request->all(), DeliveryPerson::currentLocationRules(), DeliveryPerson::messages());
                if ($validation->fails()) {
                    $this->httpCode = config('api.constants.httpCodes.SUCCESS');
                    $this->code     = config('api.constants.httpCodes.ERROR');
                    $this->message  = $validation->errors()->first();
                    return $this->sendResponse();
                }
                $deliveryPerson = DeliveryPerson::where(['user_id' => $user->id])->first();
                $deliveryPerson->latitude = $input['latitude'];
                $deliveryPerson->longitude = $input['longitude'];
                if ($deliveryPerson->save()) {
                    $this->dataList = [];
                    $this->code     = config('api.constants.httpCodes.SUCCESS');
                    $this->message  = __('api/validation/delivery-person.success.location');
                } else {
                    $this->code    = config('api.constants.httpCodes.ERROR');
                    $this->message = __('api/validation/delivery-person.error.location');
                }
                return $this->sendResponse();
            }
        } catch (\Throwable $th) {
            return $this->processError($th);
        }
    }



    /**
     * Author : jaidev
     * Email : jaidev@nmgtechnologies.com
     * Method : updateCurrentLocation
     * Desc : update current location of rider
     * Create : 17-Aug-2021
     *
     */
    public function orders(Request $request)
    {
        try {
            if ($request->isMethod('get')) {
                $user = $this->user->find($request->user()->id);

                if ($user->status == 0) {
                    $this->code = config('api.constants.httpCodes.FORBIDDEN');
                    $this->message = __('api/validation/user.error.blocked');
                    return $this->sendResponse();
                }
                if ($user->is_delivery_person != 1) {
                    $this->httpCode    = config('api.constants.httpCodes.SUCCESS');
                    $this->code    = config('api.constants.httpCodes.ERROR');
                    $this->message = 'You are not allowed to perform this operation';
                    return $this->sendResponse();
                }
                $validateResponse =  $this->validateRiderAccount($request->user()->id);
                if ($validateResponse !== true) {
                    $this->code = $validateResponse['code'];
                    $this->message = $validateResponse['message'];
                    $this->httpCode = $validateResponse['httpCode'];
                    return $this->sendResponse();
                }
                $input = $request->all();
                $validation = Validator::make($request->all(), DeliveryPerson::getOrderRules(), DeliveryPerson::messages());
                if ($validation->fails()) {
                    $this->httpCode = config('api.constants.httpCodes.SUCCESS');
                    $this->code     = config('api.constants.httpCodes.ERROR');
                    $this->message  = $validation->errors()->first();
                    return $this->sendResponse();
                }
                $deliveryPerson = DeliveryPerson::where(['user_id' => $user->id])->first();
                $request->request->add(['delivery_person_id' => $deliveryPerson->id]);
                $orderList = DeliveryPerson::orderList($request);

                // dd($orderList);
                if (!empty($orderList)) {
                    $transformData = (new OrderTransformer)->collection($orderList, false);
                    $this->dataList['orderCount'] = $orderList->count();
                    $this->dataList['orders'] = $transformData->data;
                    $this->dataList['pageData']['total'] = (int) $orderList->total();
                    $this->dataList['pageData']['per_page'] = (int) $orderList->perPage();
                    $this->dataList['pageData']['current_page'] = (int) $orderList->currentPage();
                    $this->dataList['pageData']['last_page'] = (int) $orderList->lastPage();
                    $this->code    = config('api.constants.httpCodes.SUCCESS');
                    $this->message = __('api/validation/order.success.fetch');
                } else {
                    $this->dataList = [];
                    $this->code    = config('api.constants.httpCodes.SUCCESS');
                    $this->message = __('api/validation/order.error.empty');
                }

                return $this->sendResponse();
            }
        } catch (\Throwable $th) {
            return $this->processError($th);
        }
    }

    /**
     * Author : jaidev
     * Email : jaidev@nmgtechnologies.com
     * Method : orderDetails
     * Create : 18/Aug/2021
     * Desc : return order details assigned to delivery person
     */
    public function orderDetails(Request $request)
    {
        try {
            if ($request->isMethod('get')) {
                $user = $this->user->find($request->user()->id);
                if ($user->status == 0) {
                    $this->code = config('api.constants.httpCodes.FORBIDDEN');
                    $this->message = __('api/validation/user.error.blocked');
                    return $this->sendResponse();
                }
                if ($user->is_delivery_person != 1) {
                    $this->httpCode    = config('api.constants.httpCodes.SUCCESS');
                    $this->code    = config('api.constants.httpCodes.ERROR');
                    $this->message = 'You are not allowed to perform this operation';
                    return $this->sendResponse();
                }
                $validateResponse =  $this->validateRiderAccount($request->user()->id);
                if ($validateResponse !== true) {
                    $this->code = $validateResponse['code'];
                    $this->message = $validateResponse['message'];
                    $this->httpCode = $validateResponse['httpCode'];
                    return $this->sendResponse();
                }
                $input = $request->all();
                $validation = Validator::make($request->all(), DeliveryPerson::orderDetailRules(), DeliveryPerson::messages());
                if ($validation->fails()) {
                    $this->httpCode = config('api.constants.httpCodes.SUCCESS');
                    $this->code     = config('api.constants.httpCodes.ERROR');
                    $this->message  = $validation->errors()->first();
                    return $this->sendResponse();
                }
                $deliveryPerson = DeliveryPerson::where(['user_id' => $user->id])->first();
                $request->request->add(['delivery_person_id' => $deliveryPerson->id]);
                $orderData =  DeliveryPerson::orderDetails($request);
                if (!empty($orderData)) {
                    if ($orderData->delivery_person_id == $deliveryPerson->id) {
                        $transformData = (new OrderTransformer)->transform($orderData, false);
                        $this->dataList = $transformData;
                        $this->code    = config('api.constants.httpCodes.SUCCESS');
                        $this->message = __('api/validation/order.success.fetch');
                    } else {
                        $this->dataList = [];
                        $this->code    = config('api.constants.httpCodes.ERROR');
                        $this->message = __('api/validation/delivery-person.error.notAuthorized');
                    }
                } else {
                    $this->dataList = [];
                    $this->code    = config('api.constants.httpCodes.ERROR');
                    $this->message = __('api/validation/order.error.empty');
                }

                return $this->sendResponse();
            }
        } catch (\Throwable $th) {
            return $this->processError($th);
        }
    }


    /**
     * Author : jaidev
     * Email : jaidev@nmgtechnologies.com
     * Method : acceptOrderAssignment
     * Create : 18/Aug/2021
     * Desc : accept order assignment
     */

    public function acceptOrderAssignment(Request $request)
    {
        try {
            if ($request->isMethod('post')) {
                $user = $this->user->find($request->user()->id);
                if ($user->status == 0) {
                    $this->code = config('api.constants.httpCodes.FORBIDDEN');
                    $this->message = __('api/validation/user.error.blocked');
                    return $this->sendResponse();
                }
                if ($user->is_delivery_person != 1) {
                    $this->httpCode    = config('api.constants.httpCodes.SUCCESS');
                    $this->code    = config('api.constants.httpCodes.ERROR');
                    $this->message = 'You are not allowed to perform this operation';
                    return $this->sendResponse();
                }
                $validateResponse =  $this->validateRiderAccount($request->user()->id);
                if ($validateResponse !== true) {
                    $this->code = $validateResponse['code'];
                    $this->message = $validateResponse['message'];
                    $this->httpCode = $validateResponse['httpCode'];
                    return $this->sendResponse();
                }
                $input = $request->all();
                $validation = Validator::make($request->all(), DeliveryPerson::acceptOrderAssignmentRules(), DeliveryPerson::messages());
                if ($validation->fails()) {
                    $this->httpCode = config('api.constants.httpCodes.SUCCESS');
                    $this->code     = config('api.constants.httpCodes.ERROR');
                    $this->message  = $validation->errors()->first();
                    return $this->sendResponse();
                }

                DB::beginTransaction();

                $deliveryPerson = DeliveryPerson::where(['user_id' => $user->id])->first();
                $request->request->add(['delivery_person_id' => $deliveryPerson->id]);
                $orderData =  Order::find($input['orderId']);
                if (!empty($orderData)) {
                    if ($orderData->delivery_person_id == $deliveryPerson->id) {
                        // get order assignment
                        $delAssignData = DeliveryPersonAssignment::where(['order_id' => $orderData->id, 'delivery_person_id' => $orderData->delivery_person_id])->first();
                        if ($delAssignData->status == 1) {
                            $delAssignData->status = 2;
                            $delAssignData->save();
                            $restaurant = Restaurant::find($orderData->restaurant_id);
                            $notification['title'] = 'Order accepted';
                            $notification['description'] = 'Order no #' . $orderData->id . ' has been accepted by delivery person';
                            $notification['order_id'] = $orderData->id;
                            $notification['user_id'] = $restaurant->owner_id;
                            Notification::create($notification);

                            ## send notification to customer ##
                            $title = "Order accepted";
                            $body = "Your order has been accepted by delivery person";
                            $notification = array('title' => $title, 'body' => $body, 'sound' => 'default', 'badge' => '1', 'orderId' => $orderData->id, 'actionId' => config('api.constants.push_notification.actionId.customer_order_details'));

                            PushNotification::sendNotification($orderData->user_id, $notification, $orderData->id);
                            ## end sending notification to customer ##


                            // mark delivery person work status as "on-ride"
                            $deliveryPerson->work_status = 1;
                            $deliveryPerson->save();


                            $this->dataList = [];
                            $this->code    = config('api.constants.httpCodes.SUCCESS');
                            $this->message = __('api/validation/delivery-person.success.orderAccepted');
                        } else {
                            $this->dataList = [];
                            $this->code    = config('api.constants.httpCodes.ERROR');
                            $this->message = ($delAssignData->status == 2 ? __('api/validation/delivery-person.error.alreadyAccepted') : __('api/validation/delivery-person.error.alreadyRejected'));
                        }
                    } else {
                        $this->dataList = [];
                        $this->code    = config('api.constants.httpCodes.ERROR');
                        $this->message = __('api/validation/delivery-person.error.notAuthorized');
                    }
                } else {
                    $this->dataList = [];
                    $this->code    = config('api.constants.httpCodes.ERROR');
                    $this->message = __('api/validation/order.error.empty');
                }

                return $this->sendResponse();
            }
        } catch (\Throwable $th) {
            DB::rollBack();
            // return $th;
            return $this->processError($th);
        }
    }

    /**
     * Author : jaidev
     * Email : jaidev@nmgtechnologies.com
     * Method : rejectOrderAssignment
     * Create : 18/Aug/2021
     * Desc : reject order assignment
     */

    public function rejectOrderAssignment(Request $request)
    {
        try {
            if ($request->isMethod('post')) {
                $user = $this->user->find($request->user()->id);
                if ($user->status == 0) {
                    $this->code = config('api.constants.httpCodes.FORBIDDEN');
                    $this->message = __('api/validation/user.error.blocked');
                    return $this->sendResponse();
                }
                if ($user->is_delivery_person != 1) {
                    $this->httpCode    = config('api.constants.httpCodes.SUCCESS');
                    $this->code    = config('api.constants.httpCodes.ERROR');
                    $this->message = 'You are not allowed to perform this operation';
                    return $this->sendResponse();
                }
                $validateResponse =  $this->validateRiderAccount($request->user()->id);
                if ($validateResponse !== true) {
                    $this->code = $validateResponse['code'];
                    $this->message = $validateResponse['message'];
                    $this->httpCode = $validateResponse['httpCode'];
                    return $this->sendResponse();
                }
                $input = $request->all();
                $validation = Validator::make($request->all(), DeliveryPerson::rejectOrderAssignmentRules(), DeliveryPerson::messages());
                if ($validation->fails()) {
                    $this->httpCode = config('api.constants.httpCodes.SUCCESS');
                    $this->code     = config('api.constants.httpCodes.ERROR');
                    $this->message  = $validation->errors()->first();
                    return $this->sendResponse();
                }
                $deliveryPerson = DeliveryPerson::where(['user_id' => $user->id])->first();
                $request->request->add(['delivery_person_id' => $deliveryPerson->id]);
                $orderData =  Order::find($input['orderId']);
                if (!empty($orderData)) {
                    DB::beginTransaction();

                    $deliveryPerson->work_status = 0;
                    $deliveryPerson->save();

                    if ($orderData->delivery_person_id == $deliveryPerson->id) {
                        // get order assignment
                        $delAssignData = DeliveryPersonAssignment::where(['order_id' => $orderData->id, 'delivery_person_id' => $orderData->delivery_person_id])->first();
                        if ($delAssignData->status == 1) {
                            $delAssignData->status = 3; // reject status
                            $delAssignData->reason = $input['reason'];
                            $delAssignData->save();

                            // find nearest rider if order status is in-prep
                            if ($orderData->order_status_id == config('admin.constants.ORDER_STATUS_IN_PREP')) {
                                // assign new delivery person or cancel or if no rider available
                                $availableRider =  DeliveryPersonVendor::getClosestAvailableRider($orderData->latitude, $orderData->longitude, $orderData->id);
                                if (!empty($availableRider)) {

                                    $orderData->delivery_person_id = $availableRider->id;
                                    $orderData->save();
                                    //
                                    $dpData['order_id'] = $orderData->id;
                                    $dpData['latitude'] = $availableRider->latitude;
                                    $dpData['longitude'] = $availableRider->longitude;
                                    $dpData['delivery_person_id'] = $availableRider->id;
                                    $dpData['status'] = 1;
                                    DeliveryPersonAssignment::create($dpData);

                                    // send notification to new delivery person
                                    #### send notification to rider ###
                                    $title = "New order request";
                                    $body = "You have a new order request";
                                    $notification = array('title' => $title, 'body' => $body, 'sound' => 'default', 'badge' => '1', 'actionId' => config('api.constants.push_notification.actionId.rider_order_details'), 'orderId' => $orderData->id);

                                    PushNotification::sendNotification($availableRider->user_id, $notification, $orderData->id);
                                    #### end sending notification to rider ###


                                } else {
                                    // send order in queue to find another rider after 10 mins
                                    $orderData->delivery_person_id = null;
                                    $orderData->rider_assignment_at = date("Y-m-d H:i:s");
                                    $orderData->save();


                                    ### send push notification to customer ###
                                    $title = config('api.constants.push_notification.title.rider_in_queue');
                                    $body = config('api.constants.push_notification.description.rider_in_queue');
                                    $notification = array('title' => $title, 'body' => $body, 'sound' => 'default', 'badge' => '1', 'actionId' => config('api.constants.push_notification.actionId.customer_order_details'), 'orderId' => $orderData->id);
                                    \app\Components\Api\V1\PushNotification::sendNotification($orderData->user_id, $notification, $orderData->id);
                                    ### send push notification to customer ###
                                }
                            }



                            DB::commit();
                            $this->dataList = [];
                            $this->code    = config('api.constants.httpCodes.SUCCESS');
                            $this->message = __('api/validation/delivery-person.success.orderRejected');
                        } else if ($delAssignData->status == 2) {
                            // if reject order after acceptance then send notification to restaurant and admin
                            $delAssignData->status = 3; // reject status
                            $delAssignData->reason = $input['reason'];
                            $delAssignData->save();


                            // send notification to restaurant and admin
                            // Rider (ID / Name) has cancel the Order id (ID number) picked due to (reason selected)
                            $restaurant = Restaurant::find($orderData->restaurant_id);
                            $notification['title'] = 'Order Cancellation';
                            $notification['description'] = 'Rider ' . \Auth::user()->first_name . ' has cancel the Order ID #' . $orderData->id . '  due to ' . $input['reason'];
                            $notification['order_id'] = $orderData->id;
                            $notification['user_id'] = $restaurant->owner_id;
                            Notification::create($notification);




                            // find nearest rider if order status is in-prep or on the way
                            if ($orderData->order_status_id == config('admin.constants.ORDER_STATUS_IN_PREP')) {
                                // assign new delivery person or cancel or if no rider available

                                $availableRider =  DeliveryPersonVendor::getClosestAvailableRider($orderData->latitude, $orderData->longitude, $orderData->id);

                                if (!empty($availableRider)) {

                                    $orderData->delivery_person_id = $availableRider->id;
                                    $orderData->save();
                                    //
                                    $dpData['order_id'] = $orderData->id;
                                    $dpData['latitude'] = $availableRider->latitude;
                                    $dpData['longitude'] = $availableRider->longitude;
                                    $dpData['delivery_person_id'] = $availableRider->id;
                                    $dpData['status'] = 1;
                                    DeliveryPersonAssignment::create($dpData);

                                    // send notification to new delivery person
                                    #### send notification to rider ###
                                    $title = "New order request";
                                    $body = "You have a new order request";
                                    $notification = array('title' => $title, 'body' => $body, 'sound' => 'default', 'badge' => '1', 'actionId' => config('api.constants.push_notification.actionId.rider_order_details'), 'orderId' => $orderData->id);

                                    PushNotification::sendNotification($availableRider->user_id, $notification, $orderData->id);
                                    #### end sending notification to rider ###

                                } else {


                                    // send order in queue to find another rider after 10 mins
                                    $orderData->delivery_person_id = null;
                                    $orderData->rider_assignment_at = date("Y-m-d H:i:s");
                                    $orderData->save();

                                    ### send push notification to customer ###
                                    $title = config('api.constants.push_notification.title.rider_in_queue');
                                    $body = config('api.constants.push_notification.description.rider_in_queue');
                                    $notification = array('title' => $title, 'body' => $body, 'sound' => 'default', 'badge' => '1', 'actionId' => config('api.constants.push_notification.actionId.customer_order_details'), 'orderId' => $orderData->id);
                                    \app\Components\Api\V1\PushNotification::sendNotification($orderData->user_id, $notification, $orderData->id);
                                    ### send push notification to customer ###
                                }
                            }



                            DB::commit();
                            $this->dataList = [];
                            $this->code    = config('api.constants.httpCodes.SUCCESS');
                            $this->message = __('api/validation/delivery-person.success.orderRejected');
                        } else {
                            $this->dataList = [];
                            $this->code    = config('api.constants.httpCodes.ERROR');
                            $this->message = ($delAssignData->status == 2 ? __('api/validation/delivery-person.error.alreadyAccepted') : __('api/validation/delivery-person.error.alreadyRejected'));
                        }
                    } else {
                        $this->dataList = [];
                        $this->code    = config('api.constants.httpCodes.ERROR');
                        $this->message = __('api/validation/delivery-person.error.notAuthorized');
                    }
                } else {
                    $this->dataList = [];
                    $this->code    = config('api.constants.httpCodes.ERROR');
                    $this->message = __('api/validation/order.error.empty');
                }

                return $this->sendResponse();
            }
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->processError($th);
        }
    }



    /**
     * Author : jaidev
     * Email : jaidev@nmgtechnologies.com
     * Method : completeOrder
     * Create : 19/Aug/2021
     * Desc : complete order
     */

    public function completeOrder(Request $request)
    {
        try {
            if ($request->isMethod('post')) {
                $user = $this->user->find($request->user()->id);
                if ($user->status == 0) {
                    $this->code = config('api.constants.httpCodes.FORBIDDEN');
                    $this->message = __('api/validation/user.error.blocked');
                    return $this->sendResponse();
                }
                if ($user->is_delivery_person != 1) {
                    $this->httpCode    = config('api.constants.httpCodes.SUCCESS');
                    $this->code    = config('api.constants.httpCodes.ERROR');
                    $this->message = 'You are not allowed to perform this operation';
                    return $this->sendResponse();
                }
                $validateResponse =  $this->validateRiderAccount($request->user()->id);
                if ($validateResponse !== true) {
                    $this->code = $validateResponse['code'];
                    $this->message = $validateResponse['message'];
                    $this->httpCode = $validateResponse['httpCode'];
                    return $this->sendResponse();
                }
                $input = $request->all();
                $validation = Validator::make($request->all(), DeliveryPerson::orderCompleteRules(), DeliveryPerson::messages());
                if ($validation->fails()) {
                    $this->httpCode = config('api.constants.httpCodes.SUCCESS');
                    $this->code     = config('api.constants.httpCodes.ERROR');
                    $this->message  = $validation->errors()->first();
                    return $this->sendResponse();
                }
                $deliveryPerson = DeliveryPerson::where(['user_id' => $user->id])->first();

                $request->request->add(['delivery_person_id' => $deliveryPerson->id]);
                $orderData =  Order::find($input['orderId']);





                if (!empty($orderData)) {
                    DB::beginTransaction();
                    if ($orderData->delivery_person_id == $deliveryPerson->id) {
                        // get order assignment
                        $delAssignData = DeliveryPersonAssignment::where(['order_id' => $orderData->id, 'delivery_person_id' => $orderData->delivery_person_id])->first();
                        if ($delAssignData->status != 2) {
                            $this->code    = config('api.constants.httpCodes.ERROR');
                            $this->message = 'You did not accept this order';
                            return $this->sendResponse();
                        }
                        if ($orderData->delivery_code != $input['verificationCode']) {
                            $this->code    = config('api.constants.httpCodes.ERROR');
                            $this->message =  __('api/validation/delivery-person.error.invalidDeliveryCode');
                            return $this->sendResponse();
                        }

                        if ($orderData->order_status_id == config('admin.constants.ORDER_STATUS_ON_THE_WAY')) {
                            $orderData->order_status_id = config('admin.constants.ORDER_STATUS_COMPLETED');
                            $orderData->save();
                            // save notification on behalf of restaurant
                            $restaurant = Restaurant::find($orderData->restaurant_id);
                            $notification['title'] = 'Order completed';
                            $notification['description'] = 'Order no #' . $orderData->id . ' has been completed';
                            $notification['order_id'] = $orderData->id;
                            $notification['user_id'] = $restaurant->owner_id;
                            Notification::create($notification);


                            #### send order completion notification to customer  ###
                            $title = "Order completed";
                            $body = "Order #" . $orderData->id . " has been delivered";
                            $notification = array('title' => $title, 'body' => $body, 'sound' => 'default', 'badge' => '1', 'orderId' => $orderData->id, 'actionId' => config('api.constants.push_notification.actionId.customer_order_details'));
                            PushNotification::sendNotification($orderData->user_id, $notification, $orderData->id);
                            ## end sending notification to customer ##

                            // send notification to delivery person ##
                            $deliveryPerson = DeliveryPerson::find($orderData->delivery_person_id);
                            $riderNotification = array('title' => $title, 'body' => $body, 'sound' => 'default', 'badge' => '1', 'orderId' => $orderData->id, 'actionId' => config('api.constants.push_notification.actionId.rider_order_details'));
                            PushNotification::sendNotification($deliveryPerson->user_id, $riderNotification,  $orderData->id);
                            ## end sending notification to rider ##



                            $orderUser = User::find($orderData->user_id);

                            if (!empty($orderUser->email)) {

                                ### Email Sending #####
                                if (!empty($input['lang'])) {
                                    \App::setLocale($input['lang']);
                                }
                                $params['replaceKeywords']['{USER_NAME}'] = $orderData->firstname . ' ' . $orderData->lastname;
                                $params['replaceKeywords']['{ORDER_ID}'] = $orderData->id;
                                $params['replaceKeywords']['{AMOUNT}'] = $orderData->total_amount;
                                $params['replaceKeywords']['{DELIVER_TO}'] = $orderData->firstname . ' ' . $orderData->lastname;
                                $params['replaceKeywords']['{ADDRESS}'] = $orderData->address1;
                                $params['replaceKeywords']['{ADDRESS_TYPE}'] = CommonHelper::getAddressType($orderData->address_type);
                                $params['toEmail'] = $orderUser->email;
                                // $params['toEmail'] = 'jaidev@nmgtechnologies.com';
                                $params['emailSlug'] = 'customer_order_completed';
                                $this->customMailer($params);
                            }



                            #### send mail to delivery person####
                            if (!empty($user->email)) {

                                ### Email Sending #####
                                if (!empty($input['lang'])) {
                                    \App::setLocale($input['lang']);
                                }
                                $params['replaceKeywords']['{USER_NAME}'] = $orderData->firstname . ' ' . $orderData->lastname;
                                $params['replaceKeywords']['{ORDER_ID}'] = $orderData->id;
                                $params['replaceKeywords']['{AMOUNT}'] = $orderData->total_amount;
                                $params['replaceKeywords']['{DELIVER_TO}'] = $orderData->firstname . ' ' . $orderData->lastname;
                                $params['replaceKeywords']['{ADDRESS}'] = $orderData->address1;
                                $params['replaceKeywords']['{ADDRESS_TYPE}'] = CommonHelper::getAddressType($orderData->address_type);
                                $params['toEmail'] = $user->email;
                                // $params['toEmail'] = 'jaidev@nmgtechnologies.com';
                                $params['emailSlug'] = 'delivery_person_order_completed';
                                $this->customMailer($params);
                            }




                            ##### send mail to restaurant ###########

                            $restaurant = Restaurant::get($orderData->restaurant_id);
                            if (!empty($restaurant->user->email)) {
                                ### Email Sending #####
                                if (!empty($input['lang'])) {
                                    \App::setLocale($input['lang']);
                                }
                                $params['replaceKeywords']['{USER_NAME}'] = $orderData->firstname . ' ' . $orderData->lastname;
                                $params['replaceKeywords']['{ORDER_ID}'] = $orderData->id;
                                $params['replaceKeywords']['{AMOUNT}'] = $orderData->total_amount;
                                $params['replaceKeywords']['{DELIVER_TO}'] = $orderData->firstname . ' ' . $orderData->lastname;
                                $params['replaceKeywords']['{ADDRESS}'] = $orderData->address1;
                                $params['replaceKeywords']['{ADDRESS_TYPE}'] = CommonHelper::getAddressType($orderData->address_type);
                                $params['toEmail'] = $restaurant->user->email;
                                // $params['toEmail'] = 'jaidev@nmgtechnologies.com';
                                $params['emailSlug'] = 'restaurant_order_completed';
                                $this->customMailer($params);
                            }


                            // change delivery person work status to "available"
                            $deliveryPerson->work_status = 0;
                            $deliveryPerson->save();
                            // DB::rollback();
                            // dd($orderData->delivery_charge);
                            RiderEarning::updateOrCreate(['order_id' => $orderData->id, 'rider_id' => $deliveryPerson->id], ['amount' => $orderData->delivery_person_revenue]);

                            OrderLog::saveLog($orderData->id, $orderData->order_status_id);
                            DB::commit();
                            $this->dataList = [];
                            $this->code    = config('api.constants.httpCodes.SUCCESS');
                            $this->message = __('api/validation/delivery-person.success.orderCompleted');
                        } else {
                            $this->dataList = [];
                            $this->code    = config('api.constants.httpCodes.ERROR');
                            $this->message = __('api/validation/delivery-person.error.onTheWayStatusRequired');
                        }
                    } else {
                        $this->dataList = [];
                        $this->code    = config('api.constants.httpCodes.ERROR');
                        $this->message = __('api/validation/delivery-person.error.notAuthorized');
                    }
                } else {
                    $this->dataList = [];
                    $this->code    = config('api.constants.httpCodes.ERROR');
                    $this->message = __('api/validation/order.error.empty');
                }

                return $this->sendResponse();
            }
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->processError($th);
        }
    }



    public function getEarningSummary(Request $request)
    {
        try {
            if ($request->isMethod('get')) {
                $user = $this->user->find($request->user()->id);
                if ($user->status == 0) {
                    $this->code = config('api.constants.httpCodes.FORBIDDEN');
                    $this->message = __('api/validation/user.error.blocked');
                    return $this->sendResponse();
                }
                if ($user->is_delivery_person != 1) {
                    $this->httpCode    = config('api.constants.httpCodes.SUCCESS');
                    $this->code    = config('api.constants.httpCodes.ERROR');
                    $this->message = 'You are not allowed to perform this operation';
                    return $this->sendResponse();
                }

                $input = $request->all();

                $deliveryPerson = DeliveryPerson::where(['user_id' => $user->id])->first();
                $startDate = \Carbon\Carbon::now()->startOfWeek()->format('Y-m-d H:i:s');
                $endDate = \Carbon\Carbon::now()->endOfWeek()->format('Y-m-d H:i:s');
                $todayStartDate = \Carbon\Carbon::now()->format('Y-m-d 00:00:00');
                $todayEndDate = \Carbon\Carbon::now()->format('Y-m-d 23:59:59');
                $weeklyEarningData =  RiderEarning::getAggregate($startDate, $endDate, $deliveryPerson->id);
                $todayEarningData =  RiderEarning::getAggregate($todayStartDate, $todayEndDate, $deliveryPerson->id);
                if (!empty($weeklyEarningData) || !empty($todayEarningData)) {
                    $data['weeklyOrders'] = (!empty($weeklyEarningData->totalOrders) ? $weeklyEarningData->totalOrders : 0);
                    $data['weeklyEarnings'] = (!empty($weeklyEarningData->totalAmount) ? $weeklyEarningData->totalAmount : 0);
                    $data['todayOrders'] = (!empty($todayEarningData->totalOrders) ? $todayEarningData->totalOrders : 0);
                    $data['todayEarnings'] = (!empty($todayEarningData->totalAmount) ? $todayEarningData->totalAmount : 0);
                    $this->dataList = $data;
                    $this->code    = config('api.constants.httpCodes.SUCCESS');
                    $this->message = __('api/validation/delivery-person.success.riderEarning');
                } else {
                    $this->dataList = [];
                    $this->code    = config('api.constants.httpCodes.ERROR');
                    $this->message = __('api/validation/delivery-person.error.riderEarningEmpty');
                }

                return $this->sendResponse();
            }
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->processError($th);
        }
    }
}
