<?php

namespace app\Http\Controllers\Api\V1;

use app\Model\Api\V1\Cart;
use app\Model\Api\V1\Order;

use Illuminate\Http\Request;
use app\Helpers\CommonHelper;
use app\Model\Api\V1\User;
use app\Model\Api\V1\Address;
use app\Model\Api\V1\CartItem;
use app\Model\Api\V1\OrderItem;
use app\Model\Api\V1\Transaction;
use app\Model\Api\V1\Notification;
use Illuminate\Support\Facades\DB;
use app\Model\Admin\GeneralSetting;
use app\Model\Api\V1\RestaurantMenu;
use app\Traits\DistanceCalculatorTrait;
use Illuminate\Support\Facades\Validator;
use app\Http\Controllers\Api\V1\BaseController;
use app\Components\Api\V1\PushNotification;
use app\Http\Middleware\Restaurant;

class CartController extends BaseController
{
    /**
     * private variable address
     *
     * @var [object]
     */
    private $cart;

    /**
     * constructor of class.
     *
     * @param Cart $cart
     */
    public function __construct(Cart $cart)
    {
        $this->cart = $cart;
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        try {
            if ($request->user()->status == 1) {
                $request->request->add(['userId' => $request->user()->id]);
                $cartData = Cart::where(['user_id' => $request->user()->id])->first();
                if (!empty($cartData)) {
                    $cartData =  $this->cart->list($request);
                    // dd($cartData);
                    $data = $this->formatResponse($cartData);
                    $this->dataList = $data;
                    $this->code    = config('api.constants.httpCodes.SUCCESS');
                    $this->message = __('api/validation/cart.success.fetch');
                } else {
                    $this->dataList = [];
                    $this->code    = config('api.constants.httpCodes.SUCCESS');
                    $this->message = __('api/validation/cart.error.emptyCart');
                }
            } else {
                $this->code = config('api.constants.httpCodes.FORBIDDEN');
                $this->message = __('api/validation/user.error.blocked');
            }
            return $this->sendResponse();
        } catch (\Throwable $th) {
            return $this->processError($th);
        }
    }

    /**
     * Author : jaidev
     * Email : jaidev@nmgtechnologies.common
     * Method : formatResponse
     * Desc : format get cart details api response
     * return : api formatted response
     * Create : 02-Aug-2021
     */

    public function formatResponse($cart)
    {
        $imageHeight = request()->header('height');
        $imageWidth = request()->header('width');

        $data['cartId'] = $cart->id;
        $data['restaurant']['restaurantId'] = $cart->restaurant->id;
        $data['restaurant']['restaurantName'] = $cart->restaurant->name;
        $data['restaurant']['restaurantAddress'] = $cart->restaurant->address1;
        $data['restaurant']['deliveryTime'] = (!empty($cart->restaurant->delivery_time) ? $cart->restaurant->delivery_time . ' min' : '15-30 min');
        // get total items price
        $i = 0;
        $orderCost = 0;
        foreach ($cart->cartItems as $item) {
            $data['item'][$i]['itemId'] = $item->foodItem->id;
            $data['item'][$i]['itemName'] = $item->foodItem->name;

            if (!empty($imageHeight) || !empty($imageWidth)) { // Get lambda images.
                $foodItemImage = (!empty($item->foodItem->image) ? \app\Helpers\ResizeImagesHelper::resizeImage($item->foodItem->image) : '');
            } else {
                $foodItemImage = (!empty($item->foodItem->image) ? $item->foodItem->image : '');
            }

            $data['item'][$i]['itemImage'] = $foodItemImage;
            $data['item'][$i]['itemCount'] = $item->item_qty;
            $data['item'][$i]['itemCost'] = $item->foodItem->price;
            $data['item'][$i]['isActive'] = (bool) $item->foodItem->status;
            $orderCost = $orderCost + ($item->item_qty * $item->foodItem->price);
            $i++;
        }
        $serviceFee = $this->serviceFeeCalc($orderCost, $cart->restaurant->id);
        $deliveryFee = CommonHelper::calculateDeliveryFees($cart->distance);
        $data['price']['orderCost'] = number_format($orderCost, 2, '.', '');
        $data['price']['serviceFee'] = number_format($serviceFee, 2, '.', '');
        $data['price']['deliveryFee'] = number_format($deliveryFee, 2, '.', '');
        $data['price']['totalBill'] = number_format(($orderCost + $serviceFee + $deliveryFee), 2, '.', '');

        $data['address']['latitude'] = $cart->latitude;
        $data['address']['longitude'] = $cart->longitude;
        $data['address']['distance'] = $cart->distance;

        $checkWithAdd = $this->cart->checkAddressWithCoordinates($cart->user_id, $cart->latitude, $cart->longitude);
        if (!empty($checkWithAdd)) {
            $data['address']['id'] = $checkWithAdd->id;
            $data['address']['address1'] = $checkWithAdd->address1;
            $data['address']['addressType'] = $checkWithAdd->address_type;
        } else {
            $data['address']['id'] = null;
            $data['address']['address1'] = null;
            $data['address']['addressType'] = null;
        }
        // $data['address']['addressLine'] = $cart->address->address1;
        // $data['address']['addressType'] = CommonHelper::getAddressType($cart->address->address_type);
        return $data;
    }

    /**
     * Author : jaidev
     * Email : jaidev@nmgtechnologies.common
     * Method : serviceFeeCalc
     * Desc : calculate service of of given amount
     * return : service fees
     * Create : 02-Aug-2021
     * Update : 16-March-2022
     */
    public function serviceFeeCalc($amount, $restaurantId)
    {
        $restaurant = \app\Model\Api\V1\Restaurant::find($restaurantId);
        // $getTaxPerc = GeneralSetting::getTaxFee();
        $getTaxAmt = GeneralSetting::getTaxableAmount($amount);
        // if ($restaurant->is_include_commission == 1) {
        //     $getCommissionPerc = GeneralSetting::getCommissionPerc();
        // } else {
        //     $getCommissionPerc = 0;
        // }
        $fee = 0;
        // if (!empty($getCommissionPerc)) {
        //     $fee = $fee + (($amount * $getCommissionPerc) / 100);
        // }
        if (!empty($getTaxAmt)) {
            $fee = $fee + $getTaxAmt;
        }
        return $fee;
    }




    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        try {
            if ($request->isMethod('post')) {
                $user = $request->user();
                if ($request->user()->status == 1) {
                    $input = $request->all();
                    $validation = Validator::make($input, Cart::rules(), Cart::messages());
                    if ($validation->fails()) {
                        $this->httpCode = config('api.constants.httpCodes.SUCCESS');
                        $this->code     = config('api.constants.httpCodes.ERROR');
                        $this->message  = $validation->errors()->first();
                        return $this->sendResponse();
                    }
                    // validate restaurant
                    $getCart = Cart::where(['user_id' => $user->id])->first();
                    $getItemData = RestaurantMenu::where(['id' => $input['itemId']])->with('restaurant')->first();
                    // $getAddress = Address::find($input['addressId']);
                    $restLat = $getItemData->restaurant->latitude;
                    $restLong = $getItemData->restaurant->longitude;
                    $distance  = DistanceCalculatorTrait::distanceOneToOneApi($input['latitude'], $input['longitude'], $restLat, $restLong);
                    $allowedRange = (float) GeneralSetting::getDeliveryRange();
                    if ($input['itemQty'] == 0) {
                        // delete cart item if qty is "0"
                        $cart = Cart::where(['user_id' => $user->id])->first();
                        if (!empty($cart)) {
                            $deleteStatus = CartItem::where(['cart_id' => $cart->id, 'item_id' => $input['itemId']])->delete();
                            $cartItem = CartItem::where(['cart_id' => $cart->id])->get();
                            $cartId = $cart->id;
                            if ($cartItem->count() == 0) {
                                $cart->delete();
                            }
                            $this->dataList['summary'] = Cart::summary($cartId);
                            $this->code    = config('api.constants.httpCodes.SUCCESS');
                            $this->message = __('api/validation/cart.success.deleted');
                        } else {
                            $this->dataList = [];
                            $this->code    = config('api.constants.httpCodes.SUCCESS');
                            $this->message = __('api/validation/cart.error.emptyCart');
                        }
                    } else if ($getItemData->restaurant->user->is_active == 0) {
                        // check if restaurant is offline
                        $this->httpCode = config('api.constants.httpCodes.SUCCESS');
                        $this->code     = config('api.constants.httpCodes.ERROR');
                        $this->message  = __('api/validation/restaurant.error.offline');
                    } else if ($distance > $allowedRange) {
                        $this->httpCode = config('api.constants.httpCodes.SUCCESS');
                        $this->code     = config('api.constants.httpCodes.ERROR');
                        $this->message  = __('api/validation/restaurant.error.exceedAllowedRange');
                    } else if (!empty($getCart) && ($getCart->restaurant_id != $getItemData->restaurant_id)) {
                        // check if restaurant id is not matched with requested restaurant id
                        $this->httpCode = config('api.constants.httpCodes.SUCCESS');
                        $this->code     = config('api.constants.httpCodes.ERROR');
                        $this->message  = __('api/validation/cart.error.singleRestaurantAllowed');
                    } else if ($getItemData->status == 0) {
                        $this->httpCode = config('api.constants.httpCodes.SUCCESS');
                        $this->code     = config('api.constants.httpCodes.ERROR');
                        $this->message  = __('api/validation/restaurant.error.itemInactive');
                    } else {

                        $cartStatus = Cart::updateOrCreate(['user_id' => $user->id], [
                            'restaurant_id' => $getItemData->restaurant_id,
                            'latitude' => $input['latitude'],
                            'longitude' => $input['longitude'],
                            'distance' => $distance
                        ]);

                        if ($cartStatus) {
                            CartItem::UpdateOrCreate(['cart_id' => $cartStatus->id, 'item_id' => $input['itemId']], ['item_qty' => $input['itemQty']]);
                            $this->dataList['summary'] = Cart::summary($cartStatus->id);
                            $this->code    = config('api.constants.httpCodes.SUCCESS');
                            $this->message = __('api/validation/cart.success.added');
                        } else {
                            $this->code    = config('api.constants.httpCodes.SUCCESS');
                            $this->message = __('api/validation/cart.error.added');
                        }
                    }
                } else {
                    $this->code = config('api.constants.httpCodes.FORBIDDEN');
                    $this->message = __('api/validation/user.error.blocked');
                }
                return $this->sendResponse();
            }
        } catch (\Throwable $th) {
            return $this->processError($th);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        //
    }



    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request, $id)
    {
        //
    }


    public function updateAddress(Request $request)
    {
        // return $request->all();
        try {
            if ($request->isMethod('post')) {
                if ($request->user()->status == 1) {
                    $user = $request->user();
                    $request->request->add(['userId' => $request->user()->id]);
                    $input = $request->all();
                    $validation = Validator::make($input, Cart::updateAddressRules(), Cart::messages());
                    if ($validation->fails()) {
                        $this->httpCode = config('api.constants.httpCodes.SUCCESS');
                        $this->code     = config('api.constants.httpCodes.ERROR');
                        $this->message  = $validation->errors()->first();
                        return $this->sendResponse();
                    }
                    // validate restaurant
                    $getCart = Cart::where(['user_id' => $user->id])->with('restaurant')->first();
                    if (!empty($getCart)) {
                        // $getAddress = Address::find($input['addressId']);
                        $restLat = $getCart->restaurant->latitude;
                        $restLong = $getCart->restaurant->longitude;
                        $distance  = DistanceCalculatorTrait::distanceOneToOne($input['latitude'], $input['longitude'], $restLat, $restLong);
                        $allowedRange = (float) GeneralSetting::getDeliveryRange();
                        if ($distance <= $allowedRange) {
                            $actualDistance = DistanceCalculatorTrait::distanceOneToOneApi($input['latitude'], $input['longitude'], $restLat, $restLong);
                            $getCart->latitude = $input['latitude'];
                            $getCart->longitude = $input['longitude'];
                            $getCart->distance = (float) $actualDistance;
                            $getCart->save();

                            $cartData =  $this->cart->list($request);
                            $data = $this->formatResponse($cartData);
                            $this->dataList = $data;
                            $this->httpCode = config('api.constants.httpCodes.SUCCESS');
                            $this->code     = config('api.constants.httpCodes.SUCCESS');
                            $this->message  = __('api/validation/cart.success.addressUpdate');
                        } else {
                            $this->httpCode = config('api.constants.httpCodes.SUCCESS');
                            $this->code     = config('api.constants.httpCodes.ERROR');
                            $this->message  = __('api/validation/restaurant.error.exceedAllowedRange');
                        }
                    } else {
                        $this->dataList = [];
                        $this->code    = config('api.constants.httpCodes.SUCCESS');
                        $this->message = __('api/validation/cart.error.emptyCart');
                    }
                } else {
                    $this->code = config('api.constants.httpCodes.FORBIDDEN');
                    $this->message = __('api/validation/user.error.blocked');
                }
                return $this->sendResponse();
            }
        } catch (\Throwable $th) {
            return $this->processError($th);
        }
    }



    public function checkout(Request $request)
    {
        try {
            if ($request->isMethod('post')) {
                if ($request->user()->status == 1) {

                    $user = $request->user();
                    $request->request->add(['userId' => $request->user()->id]);
                    $input = $request->all();
                    // validate restaurant
                    $validator = Validator::make($input, Order::rules(), Order::messages());
                    if ($validator->fails()) {
                        $this->code = config('api.constants.httpCodes.ERROR');
                        $this->message = $validator->errors()->first();
                        return $this->sendResponse();
                    }


                    $getCart = $this->cart->list($request);
                    if (!empty($getCart)) {
                        DB::beginTransaction();
                        $restLat = $getCart->restaurant->latitude;
                        $restLong = $getCart->restaurant->longitude;
                        $address = Address::where(['id' => $input['addressId']])->first();
                        if ($address->user_id != $request->user()->id) {
                            $this->code = config('api.constants.httpCodes.ERROR');
                            $this->message = "Invalid address";
                            return $this->sendResponse();
                        }

                        if ($getCart->restaurant->status != 1) {
                            $this->code = config('api.constants.httpCodes.ERROR');
                            $this->message = "Selected restaurant is not active";
                            return $this->sendResponse();
                        }

                        $restaurantUser = User::find($getCart->restaurant->owner_id);
                        if ($restaurantUser->status != 1) {
                            $this->code = config('api.constants.httpCodes.ERROR');
                            $this->message = "Selected restaurant is not active";
                            return $this->sendResponse();
                        }


                        if ($address->latitude != $getCart->latitude || $address->longitude != $getCart->longitude) {
                            $this->code = config('api.constants.httpCodes.ERROR');
                            $this->message = "Selected address is not matched with cart address";
                            return $this->sendResponse();
                        }

                        if ($input['paymentStatus'] == 0) {
                            $this->code = config('api.constants.httpCodes.ERROR');
                            $this->message = "Transaction has been failed";


                            $title = "Payment failed";
                            $body = "Your payment for order has been failed";
                            $notification = array('title' => $title, 'body' => $body, 'sound' => 'default', 'badge' => '1', 'actionId' => config('api.constants.push_notification.actionId.cart_details'), 'orderId' => null);
                            PushNotification::sendNotification($getCart->user->id, $notification, null);
                            DB::commit();
                            return $this->sendResponse();
                        }


                        // order processing
                        $orderData['restaurant_id'] = $getCart->restaurant_id;
                        $orderData['user_id'] = $getCart->user_id;
                        $orderData['order_status_id'] = 1; // default id of "placed" as order status
                        $orderData['firstname'] = $getCart->user->first_name;
                        $orderData['lastname'] = $getCart->user->last_name;
                        $orderData['phone_number'] = $getCart->user->phone_number;
                        $orderData['email'] = $getCart->user->email;
                        $orderData['address1'] = $address->address1;
                        $orderData['address2'] = $address->address2;
                        $orderData['city'] = $address->city;
                        $orderData['state'] = $address->state;
                        $orderData['zipcode'] = $address->zipcode;
                        $orderData['address_type'] = $address->address_type;
                        $orderData['latitude'] = $address->latitude;
                        $orderData['longitude'] = $address->longitude;
                        $orderData['country'] = $address->country;
                        $orderData['delivery_note'] = (!empty($input['deliveryNote']) ? $input['deliveryNote'] : null);



                        $orderData['distance'] = $getCart->distance;
                        $orderData['updated_at'] = date("Y-m-d H:i:s");


                        // $getTaxPerc = GeneralSetting::getTaxFee();
                        $getCommissionPerc = GeneralSetting::getCommissionPerc();

                        // getting sub_total and setting up items data
                        $i = 0;
                        $subTotal = 0;
                        foreach ($getCart->cartItems as $item) {
                            $itemData[$i]['item_id'] = (string) $item->foodItem->id;
                            $itemData[$i]['item_qty'] = (string) $item->item_qty;
                            $itemData[$i]['item_price'] = (string) $item->foodItem->price;
                            $subTotal = $subTotal + ($item->item_qty * $item->foodItem->price);
                            $i++;
                        }

                        $orderData['sub_total'] = (string) $subTotal;

                        // tax calculation
                        $getTaxData = GeneralSetting::where(['slug' => 'tax_fee', 'status' => 1])->first();
                        if ($getTaxData) {
                            if ($getTaxData->symbal == 'fixed') {
                                $taxAmt =  $getTaxData->value;
                            } else {
                                $taxAmt =  (($subTotal * $getTaxData->value) / 100);
                            }
                            $orderData['tax_value'] = (string) $getTaxData->value;
                            $orderData['tax_type'] = (string) $getTaxData->symbal;
                        } else {
                            $taxAmt = 0;
                            $orderData['tax_value'] = null;
                            $orderData['tax_type'] = null;
                        }
                        $orderData['tax'] = (string) $taxAmt;
                        // end tax calculation

                        // check if commission is enabled on restaurant or not
                        if ($getCart->restaurant->is_include_commission == 1) {
                            $orderData['commission'] = (string) ($subTotal * $getCommissionPerc / 100);
                            $orderData['commission_percentage'] = (string) $getCommissionPerc;
                        } else {
                            $orderData['commission'] = (string) 0;
                            $orderData['commission_percentage'] = (string) 0;
                        }


                        $orderData['delivery_charge'] = (string) CommonHelper::calculateDeliveryFees($getCart->distance);
                        // $orderData['total_amount'] = (string) ($subTotal + $orderData['tax'] + $orderData['commission'] + $orderData['delivery_charge']);
                        $orderData['total_amount'] = (string) ($subTotal + $orderData['tax']  + $orderData['delivery_charge']);
                        $orderData['delivery_code'] = $this->generatePIN(4);


                        $orderData['delivery_person_revenue'] = (string) CommonHelper::calcDeliveryPersonRevenue($orderData['distance']);
                        $orderData['revenue_on_delivery_charge'] = (string) ($orderData['delivery_charge'] -  $orderData['delivery_person_revenue']);
                        $transactionData['txn_no'] = $input['txnNo'];
                        $transactionData['mode'] = $input['paymentType'];
                        $transactionData['status'] = $input['paymentStatus'];
                        $transactionData['amount'] = (string) $orderData['total_amount'];

                        $order = Order::create($orderData);

                        // dd($order);

                        if ($order) {
                            foreach ($itemData as $key => $item) {
                                $itemData[$key] = $item;
                                $itemData[$key]['order_id'] = $order->id;
                            }

                            $transactionData['order_id'] = $order->id;

                            if (OrderItem::insert($itemData)) {
                                DB::commit();
                                $transaction = Transaction::create($transactionData);
                                // update transaction id in order
                                $order->transaction_id = $transaction->id;
                                $order->save();

                                if (!empty($request->user()->email)) {

                                    ### Email Sending #####
                                    if (!empty($input['lang'])) {
                                        \App::setLocale($input['lang']);
                                    }
                                    $params['replaceKeywords']['{USER_NAME}'] = $orderData['firstname'] . ' ' . $orderData['lastname'];
                                    $params['replaceKeywords']['{ORDER_ID}'] = $order->id;
                                    $params['replaceKeywords']['{TXN_ID}'] = $input['txnNo'];
                                    $params['replaceKeywords']['{AMOUNT}'] = $orderData['total_amount'];
                                    $params['replaceKeywords']['{PAYMENT_TYPE}'] = ($input['paymentType'] == 1 ? 'Online Payment' : 'COD');
                                    $params['replaceKeywords']['{PAYMENT_STATUS}'] = ($input['paymentStatus'] == 0 ? 'Failed' : 'Success');
                                    $params['toEmail'] = $request->user()->email;


                                    if ($transaction->status == 0) {
                                        DB::rollBack();
                                        // payment failed mail
                                        $params['emailSlug'] = 'customer_order_payment_failed';
                                    } else {
                                        // payment success mail
                                        $params['emailSlug'] = 'customer_order_payment_completed_successfully';
                                    }

                                    if (!$this->customMailer($params)) {
                                        DB::rollBack();
                                        return $this->processError('failed to send mail');
                                    }
                                    #### End Email Sending #######
                                }



                                ### sending push notification to customer  ###
                                $title = "Payment completed";
                                $body = "Your payment for order has been completed";

                                $notification = array('title' => $title, 'body' => $body, 'sound' => 'default', 'badge' => '1', 'actionId' => config('api.constants.push_notification.actionId.customer_order_details'), 'orderId' => $order->id);
                                PushNotification::sendNotification($getCart->user->id, $notification, $order->id);
                                ### end sending push notification customer ###

                                // save notification
                                Notification::create([
                                    'order_id' => $order->id,
                                    'user_id' => $getCart->restaurant->owner_id,
                                    'title' => 'New order',
                                    'description' => 'Order no. #' . $order->id . ', you received an order'
                                ]);


                                // delete cart items
                                CartItem::where(['cart_id' => $getCart->id])->delete();
                                Cart::where(['id' => $getCart->id])->delete();
                                // end delete cart items



                                $this->dataList = $order;
                                $this->httpCode = config('api.constants.httpCodes.SUCCESS');
                                $this->code     = config('api.constants.httpCodes.SUCCESS');
                                $this->message  = __('api/validation/order.success.placed');
                            } else {
                                DB::rollBack();
                                $this->httpCode = config('api.constants.httpCodes.SUCCESS');
                                $this->code     = config('api.constants.httpCodes.ERROR');
                                $this->message  = __('api/validation/order.error.placed');
                            }
                        } else {
                            // failed to place order
                            DB::rollback();
                            $this->httpCode = config('api.constants.httpCodes.SUCCESS');
                            $this->code     = config('api.constants.httpCodes.ERROR');
                            $this->message  = __('api/validation/order.error.placed');
                        }
                    } else {
                        $this->dataList = [];
                        $this->code    = config('api.constants.httpCodes.SUCCESS');
                        $this->message = __('api/validation/cart.error.emptyCart');
                    }
                } else {
                    $this->code = config('api.constants.httpCodes.FORBIDDEN');
                    $this->message = __('api/validation/user.error.blocked');
                }

                return $this->sendResponse();
            }
        } catch (\Throwable $th) {
            DB::rollback();
            return $th;
            return $this->processError($th);
        }
    }

    public function testNotification(Request $request)
    {
        // echo $date = \Carbon\Carbon::now()->subDays(30)->format('Y-m-d H:i:s');


        // if (!empty($token)) {
        //     $SERVER_API_KEY = env('FCM_API_KEY');
        //     if (!empty($SERVER_API_KEY)) {
        //         $title = "Test notification";
        //         $body = "Dummy notification content";
        //         $notificationData = array('title' => $title, 'body' => $body, 'sound' => 'default', 'badge' => '1', 'orderId' => false, 'actionId' => config('api.constants.push_notification.actionId.customer_order_details'));
        //         $payload = array('to' => $token, 'data' => $notificationData, 'priority' => 'high');
        //         $json = json_encode($payload);


        //         // $body = array(
        //         //     'registration_ids' => array($token),
        //         //     'notification' => array('body' => 'Hello', 'title' => 'Hello', 'vibrate' => 1, 'sound' => 1)
        //         // );

        //         // $json = json_encode($body);

        //         // echo $json;
        //         // die;



        //         $headers = [
        //             'Authorization: key=' . $SERVER_API_KEY,
        //             'Content-Type: application/json',
        //         ];
        //         $ch = curl_init();
        //         curl_setopt($ch, CURLOPT_URL, 'https://fcm.googleapis.com/fcm/send');
        //         curl_setopt($ch, CURLOPT_POST, true);
        //         curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        //         curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        //         curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        //         curl_setopt($ch, CURLOPT_POSTFIELDS, $json);
        //         $response = curl_exec($ch);
        //         $x = json_decode($response);
        //         dd($x);
        //     } else {
        //         $message = "Please add FCM_API_KEY key";
        //         return $message;
        //     }
        // }

        $token = 'c5i-coOpd0zHoV6qEKeCzw:APA91bHK1xMJssrnrH9Kx4imRb46-ZPYrHHoFHfhkdIhPuSKkB7AXIYLVyDQZVlvgcWWOip6p782OGIdOrfaFHIfe8NSGg3IFb9VamD3bLXR4tk_RowiDWRDrA359KBDg3E4pLM95Ehm';

        if (!empty($token)) {
            $SERVER_API_KEY = env('FCM_API_KEY');
            if (!empty($SERVER_API_KEY)) {
                $notificationData = array('title' => "hello title", 'body' => "hello body", 'sound' => 'default', 'badge' => '1', 'orderId' => false, 'actionId' => config('api.constants.push_notification.actionId.customer_order_details'));

                $payload = array('to' => $token, 'notification' => $notificationData, 'priority' => 'high');


                $json = json_encode($payload);
                $headers = [
                    'Authorization: key=' . $SERVER_API_KEY,
                    'Content-Type: application/json',
                ];
                $ch = curl_init();
                curl_setopt($ch, CURLOPT_URL, 'https://fcm.googleapis.com/fcm/send');
                curl_setopt($ch, CURLOPT_POST, true);
                curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
                curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_POSTFIELDS, $json);
                $response = curl_exec($ch);
                $x = json_decode($response);
                dd($x);
                //save into db

                return true;
            } else {
                $message = "Please add FCM_API_KEY key";
                return $message;
            }
        }
    }

    // function testAndroidNotification(Request $request)
    // {
    //     $parameter = [
    //         "title" => 'New Forum is added',
    //         "body" => 'Test content',
    //         "description" => 'Test desc',
    //         "plateform" => 'Admin',
    //         "pushTag" => 'ForumDetailActivity', //required
    //         "forum_id" => '12' //required
    //     ];

    //     $a = array();

    //     $ids = array("cPsFo1VdSLGsclazNgfS3T:APA91bHFvwNZe7QbGgjy8ukRonYHb0_hZhdGePu9_ERL4veU4w-30GVcfifkCIil6FRRFMQUIUGmHigJLAK8uN1d2XgssP3fTXDoou_JQQo9f2oaor2cB53Ru8bJ-NRNqi7lLIbEBjoU");
    //     $data = [
    //         "registration_ids" => $ids,
    //         "data" => $parameter
    //     ];
    //     $dataString = json_encode($data);

    //     $SERVER_API_KEY = env('FCM_API_KEY');
    //     $headers = [
    //         'Authorization: key=' . $SERVER_API_KEY,
    //         'Content-Type: application/json',
    //     ];

    //     // return $dataString;
    //     $ch = curl_init();
    //     curl_setopt($ch, CURLOPT_URL, 'https://fcm.googleapis.com/fcm/send');
    //     curl_setopt($ch, CURLOPT_POST, true);
    //     curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    //     curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    //     curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    //     curl_setopt($ch, CURLOPT_POSTFIELDS, $dataString);
    //     $response = curl_exec($ch);
    //     $x = json_decode($response);
    //     return $response;
    //     // return true;
    // }
}
