<?php

namespace app\Http\Controllers\Api;

use app\Http\Controllers\Api\BaseController;
use app\Model\Admin\BlockedReason;
use app\Model\Api\BlockedUser;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class BlockedReasons extends BaseController
{
    /**
     * @Author:: NMG
     * @Function::__invoke
     * @Input: optional parameters- page, limit, q(searchString)
     * @Description:: This function will return inappropriate list defualt & as per search term
     * @Output:: A list of inappropriate as per serch string
     * @Dated:: 07/Jan/2020
     * @updated:: 07/Jan/2020
     */
    public function __invoke(Request $request)
    {
        try {

            $page = $request->query('page', 1);
            $limit = $request->query('limit', config('api.constants.DEFAULT_PAGE_LIMIT'));
            $this->const['page'] = $page;
            $this->const['limit'] = $limit;

            $searchString = filter_var($request->q, FILTER_SANITIZE_STRING);
            //Normal Inappropriate list
            $query = BlockedReason::select(
                'blocked_reasons.id',
                'blocked_reasons.reason',
                'blocked_reasons.status'
            )->where('status', 1);

            if (!empty($searchString)) {
                $query->where('blocked_reasons.reason', 'like', "%{$searchString}%");
            }

            $reasonList = $query->get();

            if (!empty($reasonList)) {
                foreach ($reasonList as $key => $reason) {
                    $this->dataList['reasonList'][$key]['id'] = (int) $reason->id;
                    $this->dataList['reasonList'][$key]['reason'] = $reason->reason;
                    $this->dataList['reasonList'][$key]['status'] = $reason->status;
                }

                $this->code = config('api.constants.httpCodes.SUCCESS');
                $this->message = __('api/validation/blocked-reason.success.inappropriateList');
            } else {
                $this->dataList['reasonList'] = [];
                $this->code = config('api.constants.httpCodes.ERROR');
                $this->message = __('api/validation/blocked-reason.error.inappropriateList');
            }

            return $this->sendResponse();
        } catch (\Exception $exception) {
            $this->saveErrorLog($exception);
            $this->code = config('api.constants.httpCodes.SERVER_ERROR');
            $this->message = __('api/common.SERVER_ERROR');
            return $this->sendError();
        }
    }

    /**
     * Author: NMG
     * Function: markIappropriateUser
     * Description: it will return back auth/user details
     * Input: update user details ['userId', 'reasonId']
     * Output: json response of add appropriate details
     * Dated: 07/Jan/2020
     * Updated: 07/Jan/2020
     */
    public function markIappropriateUser(Request $request)
    {
        try {
            if ($request->isMethod('post')) {
                $authUser = Auth::user();
                $auth_id = $authUser->id;

                if ($request->isJson()) {
                    $input = $request->json()->all();
                } else {
                    $input = $request->all();
                }

                if (!empty($input['request'])) {
                    //encrypted request from api
                    $encrypted = $input['request'];
                    $decrypted = $this->apiEncryption('decrypt', $encrypted);
                    $input = (array) json_decode($decrypted);
                } else {
                    $input = $request->only('userId', 'reasonId');
                }

                if (!empty($input['userId'])) {
                    $input['user_id'] = (int) $input['userId'];
                }
                if (!empty($input['reasonId'])) {
                    $input['blocked_reason_id'] = (int) $input['reasonId'];
                }

                // server side validations
                $validation = Validator::make($input, BlockedUser::apiRules(), BlockedUser::messages());
                if ($validation->fails()) {
                    $errorCode = key(current($validation->errors()));
                    $this->code = config('api.constants.httpCodes.ERROR');
                    $this->message = $validation->errors()->first();
                } else {
                    $rollBack = false;
                    $checkUser = BlockedUser::where('user_id', $input['userId'])->first();
                    if (!empty($checkUser)) {
                        BlockedUser::where('user_id', $input['userId'])->delete();
                    }

                    $blockUser = BlockedUser::create([
                        'user_id' => $input['userId'],
                        'blocked_by' => $auth_id,
                        'blocked_reason_id' => $input['reasonId'],
                    ]);

                    if ($blockUser) {
                        $this->dataList = [];
                        $this->code = config('api.constants.httpCodes.SUCCESS');
                        $this->message = __('api/validation/blocked-reason.success.userblockedAdded');
                    } else {
                        $rollBack = true;
                        $this->code = config('api.constants.httpCodes.ERROR');
                        $this->message = __('api/validation/blocked-reason.error.userblockedAdded');
                    }
                }
                return $this->sendResponse();
            }
        } catch (\Exception $exception) {
            $this->saveErrorLog($exception);
            $this->httpCode = config('api.constants.httpCodes.SERVER_ERROR');
            $this->code = config('api.constants.httpCodes.SERVER_ERROR');
            $this->message = __('api/common.SERVER_ERROR');
            return $this->sendError();
        }
    }
}
