<?php

namespace app\Http\Controllers\Admin;

use app\Http\Controllers\Admin\BaseController;
use app\Model\Admin\User;
use app\Model\Admin\UserAvatars;
use app\Traits\FilePathUrl;
use Auth;
use DB;
use File;
use Illuminate\Http\Request;
use Image;
use Validator;

class TestImageController extends BaseController
{
    private $url;
    private $title;
    private $breadCrumb;
    use FilePathUrl;

    /**
     * @Method: __construct()
     * @Scope: public
     * @Params: null
     * @Description: Initialize variables like @avatarDir, @roleId, @roleName, @url within private scope of current class
     * @Created 22/Nov/2019
     * @Updated 22/Nov/2019
     */
    public function __construct()
    {
        try {

            $this->middleware('auth.admin')->except('login');

            $this->url = config('admin.path.ADMIN_BASE_URL');
            $this->title = 'Admin';

            $this->const['url'] = $this->url;
            $this->const['title'] = $this->title;

            //used in adminForm layout to load validation messges in js variable
            $this->const['objValidationMsgs'] = 'user';
            $this->const['objValidationJs'] = 'user';

            $this->const = array_merge($this->adminConst(), $this->const);

            //default breadcrumb option (Dashboard) for every route
            $this->breadCrumb['faClass'] = 'fa-table';
            $this->breadCrumb['breadCrumData'] = [
                0 => [
                    'text' => 'Dashboard',
                    'url' => config('admin.path.ADMIN_BASE_URL'),
                    'breadFaClass' => 'fa-dashboard',
                ],

            ];
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * @Method: myProfile()
     * @Scope: public
     * @Description: Looged in user can view & update certain profile information for himself
     * @Created 22/Nov/2019
     * @Updated 22/Nov/2019
     */
    public function myProfile(Request $request)
    {
        try {
            // method/action specific initializations
            $this->subTitle = 'My Profile';
            $this->actionUri = 'my-profile';
            $formId = 'profileEditForm';

            //pre-populated data
            $this->const['subTitle'] = $this->subTitle;
            $this->const['actionUri'] = $this->actionUri;
            $this->const['formId'] = $formId;

            //push current controller then method in breadcrumb at position 1 & 2
            $this->breadCrumb['breadCrumData'][1] = [
                'text' => $this->subTitle,
                'breadFaClass' => 'active',
            ];

            $user = Auth::guard('admin')->user();
            $avatar = UserAvatars::select('avatar')->where('user_id', $user->id)->latest()->first();
            if (!empty($avatar->avatar)) {
                $this->const['latestAvatar'] = $avatar->avatar;
            } else {
                $this->const['latestAvatar'] = '';
            }

            if ($request->isMethod('put')) {
                $input = $request->only(['first_name', 'last_name', 'avatar']);
                $validator = Validator::make($input, User::profileRules($user), User::messages());
                if ($validator->fails()) {
                    return \Redirect::back()->withInput()->withErrors($validator->errors());
                } else {
                    $rollBack = false;
                    if ($input['first_name'] != $user->first_name) {
                        $user->first_name = $input['first_name'];
                    }
                    if ($input['last_name'] != $user->last_name) {
                        $user->last_name = $input['last_name'];
                    }
                    \DB::beginTransaction();
                    if ($request->hasFile('avatar')) {

                        //Storage::delete("public/pics/{$user->image}");
                        if (!$this->upload_avatar($request->avatar, $user->id)) {
                            $rollBack = true;
                            \DB::rollBack();
                            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.UnableToUploadAvatarImage'));
                        }
                    }
                    if (!$rollBack) {
                        $user->save();
                        \DB::commit();
                        \Session::flash('success', __('admin/flash_msg.success.profileUpdated'));
                        return \Redirect::route('myProfiles');
                    }
                }
            }

            $data = array_merge($this->const, $this->breadCrumb);
            return view('Admin/admin/my_profiles', $data)->withUser($user);
        } catch (\Exception $e) {
            \DB::rollBack();
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * my functions
     */
    public function upload_avatar($file = [], $user_id = null)
    {

        if (empty($file) || empty($user_id)) {
            return false;
        }

        $uploaded = false;
        $uploadDir = config('admin.path.UPLOAD_DIR');
        $avatarDir = config('admin.path.avatar.DIR');
        //concat image name with prefixes
        $resizedPrefix = config('admin.path.avatar.RESIZE_PREFIX');
        $thumbPrefix = config('admin.path.avatar.THUMB_PREFIX');
        // getting image extension
        $ext = $file->getClientOriginalExtension();
        if (empty($ext)) {
            $mime = File::mimeType($file);

            if ($mime == 'image/jpeg') {
                $ext = 'jpeg';
            }
            if ($mime == 'image/png') {
                $ext = 'png';
            }
            if ($mime == 'image/gif') {
                $ext = 'gif';
            }
            if ($mime == 'image/svg+xml') {
                $ext = 'svg';
            }
        }
        //rename image
        $fileName = uniqid() . '.' . $ext;
        $imageObj = Image::make($file->getRealPath());
        //identify greater dimension
        $initialWidth = $imageObj->width();
        $initialHeight = $imageObj->height();

        $aspectByWidth = true;
        $aspectByHeight = false;

        if ($initialWidth < $initialHeight) {
            $aspectByHeight = true;
            $aspectByWidth = false;
        }
        // check if image uploaded on aws s3
        $key = $uploadDir . '/' . $avatarDir . '/' . $user_id . '/' . $fileName;
        $uploaded = \app\Helpers\CommonHelper::imageUpload($key, $file);

        //return response
        if ($uploaded) {
            if (UserAvatars::create([
                'user_id' => $user_id,
                'avatar' => $fileName,
                'width' => $initialWidth,
                'height' => $initialHeight,
            ])) {
                return true;
            }
        }
    }
}
