<?php

namespace app\Http\Controllers\Admin;

use app\Http\Controllers\Admin\BaseController;
use app\Model\Admin\StaticPage;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Input;
use Illuminate\Support\Facades\Validator;

class StaticPagesController extends BaseController
{
    private $const;
    private $breadCrumb;
    private $url;
    private $title;
    private $subTitle;
    private $controllerUri;
    private $actionUri;
    private $data = [];

    /**
     *@Name: Sandeep
     * @Method: __construct()
     * @Scope: public
     * @Params: null
     * @Description: Initialize variable with module scope, breadcrumb
     * @Created 02/Dec/2019
     * @Updated 03/Dec/2019
     */
    public function __construct()
    {
        try {
            parent::__construct();
            $this->middleware('auth.admin');

            $this->url = config('admin.path.ADMIN_BASE_URL') . '/static-pages';
            $this->title = 'Manage Static Pages';
            $this->controllerUri = 'static-pages';

            //Static Pages logo path & base urls
            $this->const['url'] = $this->url;
            $this->const['title'] = $this->title;
            $this->const['controllerUri'] = $this->controllerUri;

            //used in adminForm layout to load validation messges in js variable
            $this->const['objValidationMsgs'] = 'static-page';
            $this->const['objValidationJs'] = 'static-page';

            $this->const = array_merge($this->adminConst(), $this->const);

            //default breadcrumb option (Dashboard) for every route
            $this->breadCrumb['faClass'] = 'fa-table';
            $this->breadCrumb['breadCrumData'] = [
                0 => [
                    'text' => 'Dashboard',
                    'url' => config('admin.path.ADMIN_BASE_URL'),
                    'breadFaClass' => 'fa-dashboard',
                ],

            ];
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * @Name: Sandeep
     * @Method: index()
     * @Scope: public
     * @Params: @page, @limit, @search
     * @returns: Display a listing of the resource
     * @Description: static_pages table pages stores all the static pages.
     * @Created 02/Dec/2019
     * @Updated 03/Dec/2019
     */
    public function index(Request $request)
    {
        try {
            $this->breadCrumb['breadCrumData'][1] = [
                'text' => $this->const['title'],
            ];

            $page = $request->query('page', 1);
            $limit = $request->query('limit', config('admin.constants.DEFAULT_PAGE_LIMIT'));
            $this->const['page'] = $page;
            $this->const['limit'] = $limit;

            $search = !empty($request->search) ? $request->search : '';

            $data = [];

            /* \DB::enableQueryLog(); */
            $query = StaticPage::select(
                'static_pages.id',
                'static_pages.title',
                'static_pages.url',
                'static_pages.slug',
                'static_pages.meta_title',
                'static_pages.meta_description',
                'static_pages.content',
                'static_pages.status',
                'static_pages.created_at',
                'static_pages.updated_at'
            );

            if (!empty($search)) {
                $query->where('title', 'LIKE', "%$search%");
                $query->orWhere('slug', 'LIKE', "%$search%");
                $query->orWhere('meta_title', 'LIKE', "%$search%");
            }

            $staticList = $query->sortable(['created_at' => 'asc'])->paginate($limit);

            $this->data['staticPageList'] = $staticList;

            $data = array_merge($this->data, $this->const, $this->breadCrumb);

            return \View('Admin.' . $this->controllerUri . '.index')->with($data);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * @Name: Sandeep
     * @Method: edit()
     * @Scope: public
     * @params  int  $id
     * @Description: Show the form for editing the specified resource.
     * @Created 02/Dec/2019
     * @Updated 03/Dec/2019
     */
    public function edit($id = null)
    {
        try {
            // method/action specific initializations
            $this->subTitle = 'Edit';
            $this->actionUri = 'edit';
            $formId = 'staticPagesForm';

            //pre-populated data
            $this->const['subTitle'] = $this->subTitle;
            $this->const['actionUri'] = $this->actionUri;
            $this->const['formId'] = $formId;

            //push current controller then method in breadcrumb at position 1 & 2
            $this->breadCrumb['breadCrumData'][1] = [
                'text' => $this->title,
                'url' => $this->url,
                'breadFaClass' => '',
            ];
            $this->breadCrumb['breadCrumData'][2] = [
                'text' => $this->subTitle,
                'breadFaClass' => 'active',
            ];

            $id = \Crypt::decryptString($id);

            $this->data['staticPageDetail'] = StaticPage::FindOrFail($id);

            $data = array_merge($this->data, $this->const, $this->breadCrumb);
            return \View('Admin.' . $this->controllerUri . '.edit')->with($data);

        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * @Name: Sandeep
     * @Method: update()
     * @Scope: public
     * @params: @id
     * @input: @name, @status
     * @Description: Update the specified resource in storage.
     * @Created 02/Dec/2019
     * @Updated 03/Dec/2019
     */
    public function update(Request $request, $id = null)
    {
        try {
            $id = \Crypt::decryptString($id);
            $staticpage = StaticPage::FindOrFail($id);
            if ($request->isMethod('put')) {
                $input = Input::all();
                $validator = Validator::make($input, StaticPage::Rules($id), StaticPage::messages());

                if ($validator->fails()) {
                    return \Redirect::back()->withInput()->withErrors($validator->errors());
                } else {
                    $input['id'] = $staticpage->id;
                    unset($input['title']);

                    $staticpage->fill($input)->save();
                    \Session::flash('success', __('admin/flash_msg.success.StaticPagesUpdated'));
                    return \Redirect::to($this->url);
                }
            }
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * @Method: show()
     * @Scope: public
     * @params: @id
     * @Description: show detail of records
     * @Created 10/Dec/2019
     * @Updated 10/Dec/2019
     */
    public function show($id)
    {
        try {
            // method/action specific initializations
            $this->subTitle = 'View';
            $this->actionUri = 'show';

            //pre-populated data
            $this->const['subTitle'] = $this->subTitle;
            $this->const['actionUri'] = $this->actionUri;

            //push current controller then method in breadcrumb at position 1 & 2
            $this->breadCrumb['breadCrumData'][1] = [
                'text' => $this->title,
                'url' => $this->url,
                'breadFaClass' => '',
            ];
            $this->breadCrumb['breadCrumData'][2] = [
                'text' => $this->subTitle,
                'breadFaClass' => 'active',
            ];

            $id = \Crypt::decryptString($id);
            $staticPage = StaticPage::select(
                'static_pages.id',
                'static_pages.title',
                'static_pages.slug',
                'static_pages.meta_title',
                'static_pages.content',
                'static_pages.status'
            )
                ->where('id', $id)->first();

            $data['staticPageDetail'] = $staticPage;

            $data = array_merge($data, $this->const, $this->breadCrumb);
            return \View('Admin.' . $this->controllerUri . '.show')->with($data);

        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * @Method: uploadEditorImage()
     * @Scope: public
     * @params: $request
     * @Description: upload tinymce image on server
     * @Created 30/Dec/2019
     * @Updated 30/Dec/2019
     */
    public function uploadEditorImage()
    {

        $this->uploadEditorImageUpload($_FILES, 'staticPage');
    }
}
