<?php

namespace app\Http\Controllers\Admin;

use app\Http\Controllers\Admin\BaseController;
use app\Model\Admin\Role;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class RolesController extends BaseController
{

    private $const;
    private $breadCrumb;
    private $url;
    private $title;
    private $subTitle;
    private $controllerUri;
    private $actionUri;

    /**
     * @Method: __construct()
     * @Scope: public
     * @Params: null
     * @Description: Initialize variable with module scope, breadcrumb
     * @Created 28/Nov/2019
     * @Updated 28/Nov/2019
     */
    public function __construct()
    {
        try {
            parent::__construct();
            $this->middleware('auth.admin');
            $this->url = config('admin.path.ADMIN_BASE_URL') . '/roles';
            $this->title = 'Manage Roles';
            $this->controllerUri = 'roles';

            $this->const['url'] = $this->url;
            $this->const['title'] = $this->title;
            $this->const['controllerUri'] = $this->controllerUri;
            $this->const['objValidationMsgs'] = 'role';
            $this->const['objValidationJs'] = 'role';
            $this->const = array_merge($this->adminConst(), $this->const);

            $this->breadCrumb['faClass'] = 'fa-table';
            $this->breadCrumb['breadCrumData'] = [
                0 => [
                    'text' => 'Dashboard',
                    'url' => config('admin.path.ADMIN_BASE_URL'),
                    'breadFaClass' => 'fa-dashboard',
                ],

            ];
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * @Method: index()
     * @Scope: public
     * @Params: @page, @limit, @search
     * @returns: A List roles has been returnd
     * @Description: This method list all the role in the system & only super admin can access it.
     * A super administrator can add, edit , delete & assigns permission to other roles
     * @Created 28/Nov/2019
     * @Updated 28/Nov/2019
     */
    public function index(Request $request)
    {
        try {
            $this->breadCrumb['breadCrumData'][1] = [
                'text' => $this->const['title'],
            ];
            $page = $request->query('page', 1);
            $limit = $request->query('limit', config('admin.constants.DEFAULT_PAGE_LIMIT'));
            $this->const['page'] = $page;
            $this->const['limit'] = $limit;

            $search = !empty($request->search) ? $request->search : '';

            $data = [];

            /* \DB::enableQueryLog(); */
            $query = Role::select(
                'roles.id',
                'roles.name',
                'roles.description',
                'roles.is_super_admin',
                'roles.status',
                'roles.created_at'
            );

            if (!empty($search)) {

                $query->where('name', 'LIKE', "%$search%");
            }

            $roleList = $query->sortable(['created_at' => 'asc'])->paginate($limit);

            ########### download spreadsheet/pdf #######
            $roleListX = $roleList;
            if ($request->input('xlsx') == true) {
                $columnHeaders = [
                    'id' => 'ID',
                    'name' => 'Name',
                    'description' => 'Description',
                    'status' => 'Status',
                ];
                $this->exportXlsx($columnHeaders, $roleListX, 'roles');
            }
            if ($request->input('csv') == true) {
                $columnHeaders = [
                    'id' => 'ID',
                    'name' => 'Name',
                    'description' => 'Description',
                    'status' => 'Status',
                ];
                $this->exportCsv($columnHeaders, $roleListX, 'roles');
            }
            if ($request->input('pdf') == true) {
                $pdfData['title'] = $this->const['title'];
                $pdfData['roleList'] = $roleListX;
                $pdfView = \View('Admin.roles.pdf', $pdfData)->render();
                $this->exportPdf($pdfView);
            }
            #############################

            $data['roleList'] = $roleList;
            $data = array_merge($data, $this->const, $this->breadCrumb);
            return \View('Admin.roles.index')->with($data);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * @Method: create()
     * @Scope: public
     * @Params: null
     * @returns: return variable like breadcrumb, title etc.
     * @Description: Used for pre-populating some field of from & sets up constants used in UI.
     * @Created 28/Nov/2019
     * @Updated 28/Nov/2019
     */
    public function create()
    {
        try {
            // method/action specific initializations
            $this->subTitle = 'Add';
            $this->actionUri = 'create';
            $formId = 'roleForm';

            //pre-populated data
            $this->const['subTitle'] = $this->subTitle;
            $this->const['actionUri'] = $this->actionUri;
            $this->const['formId'] = $formId;

            //push current controller then method in breadcrumb at position 1 & 2
            $this->breadCrumb['breadCrumData'][1] = [
                'text' => $this->title,
                'url' => $this->url,
                'breadFaClass' => '',
            ];
            $this->breadCrumb['breadCrumData'][2] = [
                'text' => $this->subTitle,
                'breadFaClass' => 'active',
            ];

            $data = array_merge($this->const, $this->breadCrumb);
            return \View('Admin.' . $this->controllerUri . '.create')->with($data);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * @Method: store()
     * @Scope: public
     * @input: @name, @description, @status
     * @returns: After successful submission redirected to role listing page
     * @Description: Create new user roles in database
     * @Created 28/Nov/2019
     * @Updated 28/Nov/2019
     */
    public function store(Request $request)
    {
        try {
            if ($request->isMethod('post')) {
                $rollBack = false;
                $input = $request->only(['name', 'description', 'status']);
                $validator = Validator::make($input, Role::Rules(), Role::messages());
                if ($validator->fails()) {
                    return \Redirect::back()->withInput()->withErrors($validator->errors());
                } else {
                    $role = Role::create([
                        'name' => trim($input['name']),
                        'description' => trim($input['description']),
                        'status' => trim($input['status']),
                    ]);
                    if (empty($role->id)) {
                        return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.success.UnableToCreateRole'));
                    }
                    \Session::flash('success', __('admin/flash_msg.success.RoleCreated'));
                    return \Redirect::to($this->url);
                }
            }
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * @Method: edit()
     * @Scope: public
     * @params: @id
     * @returns: Returns pre-populated form data with other constants used in UI
     * @Description: It only fetches role data as per given id & return it to form
     *  for pre-populating it.
     * @Created 28/Nov/2019
     * @Updated 28/Nov/2019
     */
    public function edit($id = null)
    {
        try {

            if (request()->ajax()) {

                if (request()->isMethod('get')) {

                    if (request()->isJson()) {
                        $input = request()->json()->all();
                    } else {
                        $input = request()->all();
                    }
                    $id = \Crypt::decryptString($input['id']);
                    $model = \Crypt::decryptString($input['model']);
                    $modelStatus = \DB::table($model)->select('status')->find($id);
                    $modelData = array('status' => '1');
                    $modelValue = array('status' => '<span style="cursor:pointer" class="glyphicon glyphicon-ok" title="Active"></span>');

                    if ($modelStatus->status == '1') {
                        $modelData = array('status' => '0');
                        $modelValue = array('status' => '<span style="cursor:pointer" class="glyphicon glyphicon-remove" title="Inactive"></span>');

                        $result = \DB::table($model)->where('id', $id)->update($modelData);
                    } else {
                        $result = \DB::table($model)->where('id', $id)->update($modelData);
                    }
                    if ($result) {
                        $status = '1';
                    }
                }
                return response()->json(['success' => $status, 'changeStatus' => $modelValue, 'id' => $id]);
            }

            // method/action specific initializations
            $this->subTitle = 'Edit';
            $this->actionUri = 'edit';
            $formId = 'roleForm';

            //pre-populated data
            $this->const['subTitle'] = $this->subTitle;
            $this->const['actionUri'] = $this->actionUri;
            $this->const['formId'] = $formId;

            //push current controller then method in breadcrumb at position 1 & 2
            $this->breadCrumb['breadCrumData'][1] = [
                'text' => $this->title,
                'url' => $this->url,
                'breadFaClass' => '',
            ];
            $this->breadCrumb['breadCrumData'][2] = [
                'text' => $this->subTitle,
                'breadFaClass' => 'active',
            ];

            $id = \Crypt::decryptString($id);
            $role = Role::select(
                'roles.id',
                'roles.name',
                'roles.description',
                'roles.status',
                'roles.is_super_admin'
            )
                ->where('id', $id)
                ->first();

            if (!empty($role)) {
                $data['roleDetail'] = $role;
            }

            $data = array_merge($data, $this->const, $this->breadCrumb);
            return \View('Admin.' . $this->controllerUri . '.edit')->with($data);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * @Method: update()
     * @Scope: public
     * @params: @id
     * @input: @name, @description, @status
     * @returns: redirect to role listing page after successful submission
     * @Description: Update the specified resource in storage.
     * @Created 28/Nov/2019
     * @Updated 28/Nov/2019
     */
    public function update(Request $request, $id = null)
    {
        try {
            $id = \Crypt::decryptString($id);
            if ($request->isMethod('put')) {
                $input = $request->only(['name', 'description', 'status']);
                $validator = Validator::make($input, Role::Rules($id), Role::messages());
                if ($validator->fails()) {
                    return \Redirect::back()->withInput()->withErrors($validator->errors());
                } else {

                    if (!Role::find($id)->update([
                        'name' => trim($input['name']),
                        'description' => trim($input['description']),
                        'status' => !empty($input['status']) ? trim($input['status']) : 1,
                    ])) {
                        $rollBack = true;
                        return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.UnableToUpdateRole'));
                    }

                    \Session::flash('success', __('admin/flash_msg.success.RoleUpdated'));
                    return \Redirect::to($this->url);
                }
            }
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * @Method: destroy()
     * @Scope: public
     * @params: @id
     * @Description: Remove the specified resource(role) from storage then redirect to listing page
     * @Created 28/Nov/2019
     * @Updated 28/Nov/2019
     */
    public function destroy(Request $request, $id = null)
    {
        try {
            $id = \Crypt::decryptString($id);

            $count_role = \app\Model\Admin\Role::withCount('users')->where('id', $id)->first();

            if (!empty($count_role->users_count) && isset($count_role->users_count)) {
                \Session::flash('error', __('admin/flash_msg.error.RoleNotDeleted'));
                return redirect()->back();
            }

            $user = Role::find($id);
            if ($user->delete()) {
                \Session::flash('success', __('admin/flash_msg.success.RoleDeleted'));
            } else {
                \Session::flash('success', __('admin/flash_msg.success.UnablteToDeleteRole'));
            }
            return \Redirect::to($this->url);

        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

}
