<?php

namespace app\Http\Controllers\Admin;

use Auth;
use app\Model\Admin\Cart;
use app\Model\Admin\Role;
use app\Model\Admin\User;
use app\Model\Admin\Order;
use Illuminate\Http\Request;

use app\Model\Admin\Restaurant;
use Illuminate\Support\Facades\Input;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use app\Http\Controllers\Admin\BaseController;
use app\Components\Api\V1\PushNotification;

class RestaurantController extends BaseController
{
    private $const;
    private $roleId;
    private $roleName;
    private $breadCrumb;
    private $url;
    private $title;
    private $controllerUri;

    /**
     * @Method: __construct()
     * @Scope: public
     * @Params: null
     * @Description: Initialize variables like  @roleId, @roleName, @url within private scope of current class
     * @Created 23/June/2021
     * @Updated 23/June/2021
     */
    public function __construct(Restaurant $restaurant)
    {
        try {
            parent::__construct();
            $this->middleware('auth.admin');

            //fixed role for this module
            $this->roleId = config('admin.constants.RESTAURANT_OWNER_ROLE');
            $role = Role::select('name')->where('id', $this->roleId)->first();
            if ($role->name) {
                $this->roleName = $role->name;
            }

            $this->url = config('admin.path.ADMIN_BASE_URL') . '/restaurant';
            $this->title = 'Manage Restaurants';
            $this->controllerUri = 'restaurant';

            $this->const['url'] = $this->url;
            $this->const['title'] = $this->title;
            $this->const['controllerUri'] = $this->controllerUri;
            $this->const['roleId'] = $this->roleId;
            $this->const['roleName'] = $this->roleName;

            //used in adminForm layout to load validation messges in js variable
            $this->const['objValidationMsgs'] = 'restaurant';
            $this->const['objValidationJs'] = 'restaurant';

            //merge constant into single array with parent controller values
            $this->const = array_merge($this->adminConst(), $this->const);

            //default breadcrumb option (Dashboard) for every route
            $this->breadCrumb['faClass'] = 'fa-table';
            $this->breadCrumb['breadCrumData'] = [
                0 => [
                    'text' => 'Dashboard',
                    'url' => config('admin.path.ADMIN_BASE_URL'),
                    'breadFaClass' => 'fa-dashboard',
                ],

            ];
            $this->restaurant = $restaurant;
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * Author: Bibhash kumar
     * Email: bibhash.shisodiya@nmgtechnologies.com
     * @Method: index()
     * @Scope: public
     * @Params: @page,
     * @returns: A List of Restauurent with owner
     * @Description: This function fetches  restaurent list with owner from database, sets up
     *  breadcrumb & other constants.
     * @Created 23/June/2021
     * @Updated 24/June/2021
     */
    public function index(Request $request)
    {
        try {
            $this->breadCrumb['breadCrumData'][1] = [
                'text' => $this->const['title'],
            ];
            $page = $request->query('page', 1);
            $limit = $request->query('limit', config('admin.constants.DEFAULT_PAGE_LIMIT'));
            $this->const['page'] = $page;
            $this->const['limit'] = $limit;
            $restaurantList = $this->restaurant->list($request);
            $data['restaurantList'] = $restaurantList;
            $data = array_merge($data, $this->const, $this->breadCrumb);
            return \View('Admin.restaurant.index')->with($data);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }
    /**
     * Author: Bibhash kumar
     * Email: bibhash.shisodiya@nmgtechnologies.com
     * @Method: show()
     * @Scope: public
     * @params: @id
     * @Description: show detail of restaurent records
     * @Created 24/June/2021
     * @Updated 24/June/2021
     */
    public function show($id)
    {
        try {
            // method/action specific initializations
            $this->subTitle = 'View';
            $this->actionUri = 'show';

            //pre-populated data
            $this->const['subTitle'] = $this->subTitle;
            $this->const['actionUri'] = $this->actionUri;

            //push current controller then method in breadcrumb at position 1 & 2
            $this->breadCrumb['breadCrumData'][1] = [
                'text' => $this->title,
                'url' => $this->url,
                'breadFaClass' => '',
            ];
            $this->breadCrumb['breadCrumData'][2] = [
                'text' => $this->subTitle,
                'breadFaClass' => 'active',
            ];

            $id = \Crypt::decryptString($id);
            $restaurant = Restaurant::with('user')->where('id', $id)->first();
            $data['restaurantData'] = $restaurant;
            $data = array_merge($data, $this->const, $this->breadCrumb);
            return \View('Admin.' . $this->controllerUri . '.show')->with($data);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * Author: Bibhash kumar
     * Email: bibhash.shisodiya@nmgtechnologies.com
     * @Method: edit()
     * @Scope: public
     * @params: @id
     * @Description: edit detail of restaurent records
     * @Created 24/June/2021
     * @Updated 15/March/2021
     */
    public function edit($id)
    {
        try {

            if (request()->ajax()) {

                if (request()->isMethod('get')) {

                    if (request()->isJson()) {
                        $input = request()->json()->all();
                    } else {
                        $input = request()->all();
                    }
                    $id = \Crypt::decryptString($input['id']);
                    $model = 'users';
                    $modelStatus = \DB::table($model)->select('status')->find($id);
                    $modelData = array('status' => '1');
                    $modelValue = array('status' => '<span style="cursor:pointer" class="glyphicon glyphicon-ok" title="Active"></span>');
                    $restaurantData = Restaurant::with('user')->where('owner_id', $id)->first();
                    if ($modelStatus->status == '1') {
                        $modelData = array('status' => '0');
                        $modelValue = array('status' => '<span style="cursor:pointer" class="glyphicon glyphicon-remove" title="Inactive"></span>');

                        $result = \DB::table($model)->where('id', $id)->update($modelData);


                        ### Account blocked Email#####
                        if (!empty($input['lang'])) {
                            \App::setLocale($input['lang']);
                        }
                        $params['replaceKeywords']['{USER_NAME}'] = $restaurantData->name;

                        $params['toEmail'] = trim($restaurantData->user->email);
                        $params['emailSlug'] = 'account_blocked_for_customer_rider_restaurant';
                        if (!$this->customMailer($params)) {
                            $rollBack = true;
                        }
                        #######################
                    } else {
                        $result = \DB::table($model)->where('id', $id)->update($modelData);
                        ### Account un-blocked Email#####
                        if (!empty($input['lang'])) {
                            \App::setLocale($input['lang']);
                        }
                        $params['replaceKeywords']['{USER_NAME}'] = $restaurantData->name;
                        $params['toEmail'] = trim($restaurantData->user->email);
                        $params['emailSlug'] = 'account_unblocked_for_customer_rider_restaurant';
                        if (!$this->customMailer($params)) {
                            $rollBack = true;
                        }
                        #######################
                    }
                    if ($result) {

                        $title = "Restaurant Status";
                        $body = "Restaurant status changed";
                        $notification = array('title' => $title, 'body' => $body, 'sound' => 'default', 'badge' => '1', 'orderId' => null, 'actionId' => config('api.constants.push_notification.actionId.restaurant_online_offline'));

                        PushNotification::broadcastNotification($notification);
                        $status = '1';
                    }
                }
                return response()->json(['success' => $status, 'changeStatus' => $modelValue, 'id' => $id]);
            }

            // method/action specific initializations
            $this->subTitle = 'Edit';
            $this->actionUri = 'edit';
            $formId = 'createRestaurant';

            //pre-populated data
            $this->const['subTitle'] = $this->subTitle;
            $this->const['actionUri'] = $this->actionUri;
            $this->const['formId'] = $formId;
            $this->const['roles'] = [$this->roleId => $this->roleName];

            //push current controller then method in breadcrumb at position 1 & 2
            $this->breadCrumb['breadCrumData'][1] = [
                'text' => $this->title,
                'url' => $this->url,
                'breadFaClass' => '',
            ];
            $this->breadCrumb['breadCrumData'][2] = [
                'text' => $this->subTitle,
                'breadFaClass' => 'active',
            ];

            $id = \Crypt::decryptString($id);
            $restaurant = Restaurant::with('user')->where('id', $id)->first();
            // dd($restaurant);  exit;
            $data['userDetail'] = $restaurant;

            $data = array_merge($data, $this->const, $this->breadCrumb);
            return \View('Admin.' . $this->controllerUri . '.edit')->with($data);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * Author: Bibhash kumar
     * Email: bibhash.shisodiya@nmgtechnologies.com
     * @Method: update()
     * @Scope: public
     * @params: @id
     * @Description: update detail of restaurent records
     * @Created 24/June/2021
     * @Updated 24/June/2021
     */

    public function update(Request $request, $id)
    {
        //echo "<pre>@@@"; print_r($input = request()->all()); exit;
        try {
            $id = \Crypt::decryptString($id);
            $restaurant = Restaurant::find($id);
            if ($request->isMethod('put')) {
                $rollBack = false;
                \DB::beginTransaction();
                $input = $request->only(['restaurant_name', 'restaurant_bio', 'resturent_type_local', 'resturent_type_continental', 'resturent_type_groceries', 'status', 'country', 'deliverytime', 'email', 'restaurant_phone_number', 'address1', 'address2', 'locality', 'postcode', 'latitude', 'longitude', 'is_include_commission']);
                $validator = Validator::make($input, Restaurant::Rules($restaurant->owner_id), Restaurant::messages());
                if ($validator->fails()) {
                    return \Redirect::back()->withInput()->withErrors($validator->errors());
                } else {
                    $restaurant = Restaurant::find($id);
                    $uid = $restaurant->owner_id;
                    $restaurant->name = trim($input['restaurant_name']);
                    $restaurant->bio = trim($input['restaurant_bio']);
                    if (isset($input['resturent_type_local'])) {
                        $restaurant->resturent_type_local = 1;
                    } else {
                        $restaurant->resturent_type_local = 0;
                    }
                    if (isset($input['resturent_type_continental'])) {
                        $restaurant->resturent_type_continental = 1;
                    } else {
                        $restaurant->resturent_type_continental = 0;
                    }
                    if (isset($input['resturent_type_groceries'])) {
                        $restaurant->resturent_type_groceries = 1;
                    } else {
                        $restaurant->resturent_type_groceries = 0;
                    }

                    $restaurant->address1 = trim($input['address1']);
                    $restaurant->address2 = trim($input['address2']);
                    $restaurant->region = trim($input['country']);
                    $restaurant->city = trim($input['locality']);
                    $restaurant->zipcode = trim($input['postcode']);
                    $restaurant->latitude = trim($input['latitude']);
                    $restaurant->longitude = trim($input['longitude']);
                    $restaurant->delivery_time = trim($input['deliverytime']);
                    $restaurant->is_include_commission = trim($input['is_include_commission']);
                    $recdUpdated = $restaurant->save();

                    $user = User::find($uid);
                    $user->user_name = trim($input['restaurant_name']);
                    $user->status = trim($input['status']);
                    $user->phone_number = trim($input['restaurant_phone_number']);
                    $recdUpdated = $user->save();

                    if ($recdUpdated) {
                        if ($request->hasFile('avatar')) {
                            //Storage::delete("public/pics/{$user->image}");
                            $upload = User::upload_avatar($request->avatar, $uid);
                            if (!empty($upload)) {
                                $user->profile_image = $upload;
                                $recdUpdated = $user->save();
                            } else {
                                return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.UnableToUploadAvatarImage'));
                            }
                        }
                    } else {
                        return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.UnableToUpdateFacilityUser'));
                    }
                    if (!$rollBack) {
                        \DB::commit();
                    } else {
                        \DB::rollBack();
                    }
                }
                if (!$rollBack) {
                    \DB::commit();
                    \Session::flash('success', __('admin/flash_msg.success.RestaurantUpdated'));
                    return \Redirect::to($this->url);
                }
            }
        } catch (\Exception $e) {
            \DB::rollBack();
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * Author: Bibhash kumar
     * Email: bibhash.shisodiya@nmgtechnologies.com
     * @Method: destroy()
     * @Scope: public
     * @params: @id
     * @Description: delete detail of restaurent records
     * @Created 24/June/2021
     * @Updated 24/June/2021
     */
    public function destroy($id)
    {
        try {
            $id = \Crypt::decryptString($id);
            $dirPath = $this->const['avatarStoragePath'] . '/' . $id;

            if (is_dir($dirPath)) {
                deleteDir($dirPath);
            }
            // check if restaurant has order or not
            $user = User::find($id);
            $restaurant = Restaurant::where('owner_id', $id)->first();
            $resOrders = Order::where(['restaurant_id' => $restaurant->id])->count();
            if ($resOrders > 0) {
                return \Redirect::back()->withInput()->with('error', "We cannot remove restaurant because it is already associated with an order.");
            }

            // check if any restaurant items exist in cart
            $resCartItems = Cart::where(['restaurant_id' => $restaurant->id])->count();
            if ($resCartItems > 0) {
                return \Redirect::back()->withInput()->with('error', "Cannot delete this restaurant. Some users has added items in cart of this restaurant");
            }

            $restaurant->delete();
            $user->delete();

            \Session::flash('success', __('admin/flash_msg.success.RestaurantDeleted'));
            return \Redirect::to($this->url);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * Author: Bibhash kumar
     * Email: bibhash.shisodiya@nmgtechnologies.com
     * @Method: restaurantApproval()
     * @Description: To Approve from admin for restaurant record.
     * @Created 24/June/2021
     * @Updated 24/June/2021
     */
    public function restaurantApproval()
    {
        try {
            $input = request()->all();
            if ($input['status'] == 'Approve') {
                $status = 1;
                $msg = 'Approved';
                $emailSlug = 'admin_approves_the_restaurant';
            } else if ($input['status'] == 'Disapprove') {
                $status = 2;
                $msg = "Disapproved";
                $emailSlug = 'admin_disapproved_the_restaurant';
            }
            $result = Restaurant::where('id', $input['id'])->update(['status' => $status]);
            $modelStatus = Restaurant::find($input['id']);



            // broadcast notification to refresh app screen
            $title = "Restaurant Status";
            $body = "Restaurant status changed";
            $notification = array('title' => $title, 'body' => $body, 'sound' => 'default', 'badge' => '1', 'orderId' => null, 'actionId' => config('api.constants.push_notification.actionId.restaurant_online_offline'));

            PushNotification::broadcastNotification($notification);


            if ($modelStatus->user->email) {
                ###Welcome Email#####
                if (!empty($input['lang'])) {
                    \App::setLocale($input['lang']);
                }
                $params['replaceKeywords']['{USER_NAME}'] = $modelStatus->name;
                //$params['replaceKeywords']['{OLD_EMAIL}'] = $oldEmail;
                $params['replaceKeywords']['{REJECT_REASON}'] = $msg;
                $params['toEmail'] = trim($modelStatus->user->email);
                $params['emailSlug'] = $emailSlug;
                if (!$this->customMailer($params)) {
                    $rollBack = true;
                }
                #######################
            }
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }
}
