<?php

namespace app\Http\Controllers\Admin;

use app\Http\Controllers\Admin\BaseController;
use app\Model\Admin\Role;
use app\Model\Admin\User;
use app\Model\Admin\Order;
use app\Model\Admin\OrderStatus;
use Auth;
use DB;
use Exception;
use Illuminate\Http\Request;
use app\Model\Admin\Notification;
use app\Components\Api\V1\PushNotification;
use app\Helpers\CommonHelper;
use app\Model\Api\v1\Restaurant;
use app\Model\Api\V1\RiderEarning;

class OrderController extends BaseController
{
    private $const;
    private $roleId;
    private $roleName;
    private $breadCrumb;
    private $url;
    private $title;
    private $subTitle;
    private $controllerUri;
    private $actionUri;

    /**
     * @Method: __construct()
     * @Scope: public
     * @Params: null
     * @Description: Initialize variables like  @roleId, @roleName, @url within private scope of current class
     * @Created 28/Apr/2021
     * @Updated 28/Apr/2021
     */
    public function __construct(Order $order)
    {
        try {
            parent::__construct();
            $this->middleware('auth.admin');

            //fixed role for this module
            $this->roleId = config('admin.constants.RESTAURANT_OWNER_ROLE');
            $role = Role::select('name')->where('id', $this->roleId)->first();
            if ($role->name) {
                $this->roleName = $role->name;
            }

            $this->url = config('admin.path.ADMIN_BASE_URL') . '/orders';
            $this->customerUrl = config('admin.path.ADMIN_BASE_URL') . '/customer';
            $this->deliveryPersonUrl = config('admin.path.ADMIN_BASE_URL') . '/delivery-persons';
            $this->restaurantUrl = config('admin.path.ADMIN_BASE_URL') . '/restaurant';
            $this->title = 'Manage Order';
            $this->controllerUri = 'order';

            $this->const['url'] = $this->url;
            $this->const['customerUrl'] = $this->customerUrl;
            $this->const['restaurantUrl'] = $this->restaurantUrl;
            $this->const['deliveryPersonUrl'] = $this->deliveryPersonUrl;
            $this->const['title'] = $this->title;
            $this->const['controllerUri'] = $this->controllerUri;
            $this->const['roleId'] = $this->roleId;
            $this->const['roleName'] = $this->roleName;

            //used in adminForm layout to load validation messges in js variable
            $this->const['objValidationMsgs'] = 'order';
            $this->const['objValidationJs'] = 'order';


            //merge constant into single array with parent controller values
            $this->const = array_merge($this->adminConst(), $this->const);

            //default breadcrumb option (Dashboard) for every route
            $this->breadCrumb['faClass'] = 'fa-table';
            $this->breadCrumb['breadCrumData'] = [
                0 => [
                    'text' => 'Dashboard',
                    'url' => config('admin.path.ADMIN_BASE_URL'),
                    'breadFaClass' => 'fa-dashboard',
                ],
            ];
            $this->order = $order;
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }
    /**
     * Author: Jaidev
     * Email: jaidev@nmgtechnologies.com
     * @Method: index()
     * @Scope: public
     * @Params: @page,
     * @returns: A List of Order
     * @Description: This function fetches  order from database, sets up
     *  breadcrumb & other constants.
     * @Created 24/aug/2021
     * @Updated 24/aug/2021
     */
    public function index(Request $request)
    {
        try {
            $this->breadCrumb['breadCrumData'][1] = [
                'text' => $this->const['title'],
            ];
            $page = $request->query('page', 1);
            $limit = $request->query('limit', config('admin.constants.DEFAULT_PAGE_LIMIT'));
            $this->const['page'] = $page;
            $this->const['limit'] = $limit;

            $orderList = $this->order->list($request);
            $orderStatusList = OrderStatus::pluck('name', 'id');
            $data['orderStatusList'] = $orderStatusList;
            $data['orderList'] = $orderList;
            $data['restaurantList'] = \DB::table('restaurants')->orderBy('name', 'asc')->pluck('name', 'id')->toArray();
            $data = array_merge($data, $this->const, $this->breadCrumb);
            return \View('Admin.order.index')->with($data);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    public function show($id)
    {
        try {
            // method/action specific initializations
            $this->subTitle = 'View';
            $this->actionUri = 'show';

            //pre-populated data
            $this->const['subTitle'] = $this->subTitle;
            $this->const['actionUri'] = $this->actionUri;
            $this->const['title'] = 'View order details';

            //push current controller then method in breadcrumb at position 1 & 2
            $this->breadCrumb['breadCrumData'][1] = [
                'text' => $this->title,
                'url' => $this->url,
                'breadFaClass' => '',
            ];
            $this->breadCrumb['breadCrumData'][2] = [
                'text' => $this->subTitle,
                'breadFaClass' => 'active',
            ];
            $id = \Crypt::decryptString($id);
            $orderData = Order::details($id);
            $data['orderData'] = $orderData;
            $data = array_merge($data, $this->const, $this->breadCrumb);
            return \View('Admin.' . $this->controllerUri . '.show')->with($data);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    public function reAssignDeliveryPerson(Request $request, $orderId)
    {
        try {
            $orderId = \Crypt::decryptString($orderId);
            $order = Order::where(['id' => $orderId])->first();

            DB::beginTransaction();
            if ($order->order_status_id == config('admin.constants.ORDER_STATUS_IN_PREP') || $order->order_status_id == config('admin.constants.ORDER_STATUS_ON_THE_WAY')) {

                // mark old delivery person's status rejected
                $currentRiderAssignment = \app\Model\Vendor\DeliveryPersonAssignment::where(['order_id' => $order->id])->latest()->first();

                $currentRiderAssignment->status = 3;
                $currentRiderAssignment->reason = "Rider Re-Assigned by BMF Admin";
                $currentRiderAssignment->save();

                $currentDP = \app\Model\Admin\DeliveryPersons::where(['id' => $currentRiderAssignment->delivery_person_id])->first();
                $currentDP->work_status = 0;
                $currentDP->save();



                // get nearby delivery person
                $availableRider =  \app\Model\Vendor\DeliveryPerson::getClosestAvailableRider($order->latitude, $order->longitude, $order->id);
                if (!empty($availableRider) && $availableRider->distance <= config('admin.constants.DELIVERY_PERSON_AVAILABILITY_RANGE')) {
                    $order->delivery_person_id = $availableRider->id;
                    $order->save();
                    //
                    $dpData['order_id'] = $order->id;
                    $dpData['latitude'] = $availableRider->latitude;
                    $dpData['longitude'] = $availableRider->longitude;
                    $dpData['delivery_person_id'] = $availableRider->id;
                    $dpData['status'] = 1;
                    \app\Model\Vendor\DeliveryPersonAssignment::create($dpData);

                    ### send push notification to customer ###
                    $title = config('api.constants.push_notification.title.rider_updated');
                    $body = config('api.constants.push_notification.description.rider_updated');
                    $notification = array('title' => $title, 'body' => $body, 'sound' => 'default', 'badge' => '1', 'actionId' => config('api.constants.push_notification.actionId.customer_order_details'), 'orderId' => $order->id);
                    \app\Components\Api\V1\PushNotification::sendNotification($order->user_id, $notification, $order->id);
                    ### send push notification to customer ###



                    ### send notification to restaurant ###
                    $title = config('api.constants.push_notification.title.rider_updated');
                    $restaurant = Restaurant::find($order->restaurant_id);
                    Notification::create([
                        'order_id' => $order->id,
                        'user_id' => $restaurant->owner_id,
                        'title' => $title,
                        'description' => 'Delivery person has been changed for Order no. #' . $order->id
                    ]);
                    ### send notification to restaurant ###



                    #### send notification to rider ###
                    $title = config('api.constants.push_notification.title.rider_new_order');
                    $body =  config('api.constants.push_notification.description.rider_new_order');
                    $notification = array('title' => $title, 'body' => $body, 'sound' => 'default', 'badge' => '1', 'actionId' => config('api.constants.push_notification.actionId.rider_order_details'), 'orderId' => $order->id);

                    \app\Components\Api\V1\PushNotification::sendNotification($availableRider->user_id, $notification, $order->id);
                    #### end sending notification to rider ###
                    \Session::flash('success', __('admin/flash_msg.success.orderDeliveryPersonAssigned'));
                } else {
                    // send order in queue for rider assignment after 10 mins
                    $order->delivery_person_id = null;
                    $order->rider_assignment_at = date("Y-m-d H:i:s");
                    $order->save();

                    ### send push notification to customer ###
                    $title = config('api.constants.push_notification.title.rider_in_queue');
                    $body = config('api.constants.push_notification.description.rider_in_queue');
                    $notification = array('title' => $title, 'body' => $body, 'sound' => 'default', 'badge' => '1', 'actionId' => config('api.constants.push_notification.actionId.customer_order_details'), 'orderId' => $order->id);
                    \app\Components\Api\V1\PushNotification::sendNotification($order->user_id, $notification, $order->id);
                    ### send push notification to customer ###



                    // end send order in queue for rider assignment after 10 mins
                    \Session::flash('success', __('admin/flash_msg.success.orderInQueue'));
                }
                DB::commit();
                return \Redirect::back();
            } else {
                return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.UnableToAssignRider'));
            }
        } catch (Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }



    /***
     * Author : Jaidev
     * Email : jaidev@nmgtechnologies.com
     * Method : cancelOrder
     * Desc : cancel the order
     * return : status of cancellation
     * Date : 25/Apr/2022
     */

    public function cancelOrder(Request $request, $orderId)
    {
        try {
            $orderId = \Crypt::decryptString($orderId);
            $order = Order::where(['id' => $orderId])->first();

            DB::beginTransaction();
            // mark old delivery person's status rejected
            $currentRiderAssignment = \app\Model\Vendor\DeliveryPersonAssignment::where(['order_id' => $order->id])->latest()->first();

            // $currentRiderAssignment->status = 3;
            // $currentRiderAssignment->reason = "Rider Re-Assigned by BMF Admin";
            // $currentRiderAssignment->save();

            $currentDP = \app\Model\Admin\DeliveryPersons::where(['id' => $currentRiderAssignment->delivery_person_id])->first();
            $currentDP->work_status = 0;
            $currentDP->save();


            // cancel order from admin end
            // send order cancellation mail
            $order->order_status_id = config('admin.constants.ORDER_STATUS_CANCELED');
            $order->save();
            \app\Model\Vendor\OrderLog::saveLog($order->id, $order->order_status_id);

            #### send push notification to customer ##
            $title = "Order cancelled by BMF";
            $body = "Your order has been cancelled by BMF due to un-availability of rider";
            $notification = array('title' => $title, 'body' => $body, 'sound' => 'default', 'badge' => '1', 'orderId' => $order->id, 'actionId' => config('api.constants.push_notification.actionId.customer_order_details'));
            \app\Components\Api\V1\PushNotification::sendNotification($order->user_id, $notification,  $order->id);

            #### end sending push notification to customer ##


            #### send push notification to rider if assignment status is only accepted ##
            if ($currentRiderAssignment->status == 2) {

                $title = "Order cancelled by BMF";
                $body = "Order has been cancelled by BMF";
                $notification = array('title' => $title, 'body' => $body, 'sound' => 'default', 'badge' => '1', 'orderId' => $order->id, 'actionId' => config('api.constants.push_notification.actionId.rider_order_details'));
                \app\Components\Api\V1\PushNotification::sendNotification($currentDP->user_id, $notification,  $order->id);
            }
            #### end sending push notification to customer ##



            $user = User::find($order->user_id);
            if (!empty($user->email)) {
                ### Email Sending #####
                if (!empty($input['lang'])) {
                    \App::setLocale($input['lang']);
                }
                $params['replaceKeywords']['{USER_NAME}'] = $order->firstname . ' ' . $order->lastname;
                $params['replaceKeywords']['{ORDER_ID}'] = $order->id;
                $params['replaceKeywords']['{AMOUNT}'] = $order->total_amount;
                $params['replaceKeywords']['{DELIVER_TO}'] = $order->firstname . ' ' . $order->lastname;
                $params['replaceKeywords']['{ADDRESS}'] = $order->address1;
                $params['replaceKeywords']['{ADDRESS_TYPE}'] = \app\Helpers\CommonHelper::getAddressType($order->address_type);
                $params['toEmail'] = $user->email;
                $params['emailSlug'] = 'customer_restaurant_due_nonavailability_cancle_order';
                $this->customMailer($params);
            }
            \Session::flash('success', __('admin/flash_msg.success.orderCanceled'));

            DB::commit();
            return \Redirect::back();
        } catch (Exception $e) {
            DB::rollBack();
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }



    /**
     * Author : jaidev
     * Email : jaidev@nmgtechnologies.com
     * Method : completeOrder
     * Create : 30/March/2022
     * Desc : complete order
     */

    public function completeOrder(Request $request, $orderId)
    {
        try {
            $orderId = \Crypt::decryptString($orderId);
            $orderData =  Order::find($orderId);
            if (!empty($orderData)) {
                DB::beginTransaction();
                if ($orderData->order_status_id == config('admin.constants.ORDER_STATUS_ON_THE_WAY')) {
                    $orderData->order_status_id = config('admin.constants.ORDER_STATUS_COMPLETED');
                    $orderData->save();
                    // save notification on behalf of restaurant
                    $restaurant = \app\Model\Admin\Restaurant::find($orderData->restaurant_id);
                    $notification['title'] = 'Order completed';
                    $notification['description'] = 'Order no #' . $orderData->id . ' has been completed';
                    $notification['order_id'] = $orderData->id;
                    $notification['user_id'] = $restaurant->owner_id;
                    Notification::create($notification);


                    #### send order completion notification to customer  ###
                    $title = "Order completed";
                    $body = "Order #" . $orderData->id . " has been delivered";
                    $notification = array('title' => $title, 'body' => $body, 'sound' => 'default', 'badge' => '1', 'orderId' => $orderData->id, 'actionId' => config('api.constants.push_notification.actionId.customer_order_details'));
                    PushNotification::sendNotification($orderData->user_id, $notification, $orderData->id);
                    ## end sending notification to customer ##

                    // send notification to delivery person ##
                    $deliveryPerson = \app\Model\Admin\DeliveryPersons::find($orderData->delivery_person_id);
                    $riderNotification = array('title' => $title, 'body' => $body, 'sound' => 'default', 'badge' => '1', 'orderId' => $orderData->id, 'actionId' => config('api.constants.push_notification.actionId.rider_order_details'));
                    PushNotification::sendNotification($deliveryPerson->user_id, $riderNotification,  $orderData->id);
                    ## end sending notification to rider ##



                    $orderUser = User::find($orderData->user_id);

                    if (!empty($orderUser->email)) {

                        ### Email Sending #####
                        if (!empty($input['lang'])) {
                            \App::setLocale($input['lang']);
                        }
                        $params['replaceKeywords']['{USER_NAME}'] = $orderData->firstname . ' ' . $orderData->lastname;
                        $params['replaceKeywords']['{ORDER_ID}'] = $orderData->id;
                        $params['replaceKeywords']['{AMOUNT}'] = $orderData->total_amount;
                        $params['replaceKeywords']['{DELIVER_TO}'] = $orderData->firstname . ' ' . $orderData->lastname;
                        $params['replaceKeywords']['{ADDRESS}'] = $orderData->address1;
                        $params['replaceKeywords']['{ADDRESS_TYPE}'] = CommonHelper::getAddressType($orderData->address_type);
                        $params['toEmail'] = $orderUser->email;
                        // $params['toEmail'] = 'jaidev@nmgtechnologies.com';
                        $params['emailSlug'] = 'customer_order_completed';
                        $this->customMailer($params);
                    }



                    #### send mail to delivery person####
                    $deliveryPersonUser = User::find($deliveryPerson->user_id);

                    if (!empty($deliveryPersonUser->email)) {

                        ### Email Sending #####
                        if (!empty($input['lang'])) {
                            \App::setLocale($input['lang']);
                        }
                        $params['replaceKeywords']['{USER_NAME}'] = $orderData->firstname . ' ' . $orderData->lastname;
                        $params['replaceKeywords']['{ORDER_ID}'] = $orderData->id;
                        $params['replaceKeywords']['{AMOUNT}'] = $orderData->total_amount;
                        $params['replaceKeywords']['{DELIVER_TO}'] = $orderData->firstname . ' ' . $orderData->lastname;
                        $params['replaceKeywords']['{ADDRESS}'] = $orderData->address1;
                        $params['replaceKeywords']['{ADDRESS_TYPE}'] = CommonHelper::getAddressType($orderData->address_type);
                        $params['toEmail'] = $deliveryPersonUser->email;
                        // $params['toEmail'] = 'jaidev@nmgtechnologies.com';
                        $params['emailSlug'] = 'delivery_person_order_completed';
                        $this->customMailer($params);
                    }




                    ##### send mail to restaurant ###########

                    $restaurant = \app\Model\Api\v1\Restaurant::get($orderData->restaurant_id);
                    if (!empty($restaurant->user->email)) {
                        ### Email Sending #####
                        if (!empty($input['lang'])) {
                            \App::setLocale($input['lang']);
                        }
                        $params['replaceKeywords']['{USER_NAME}'] = $orderData->firstname . ' ' . $orderData->lastname;
                        $params['replaceKeywords']['{ORDER_ID}'] = $orderData->id;
                        $params['replaceKeywords']['{AMOUNT}'] = $orderData->total_amount;
                        $params['replaceKeywords']['{DELIVER_TO}'] = $orderData->firstname . ' ' . $orderData->lastname;
                        $params['replaceKeywords']['{ADDRESS}'] = $orderData->address1;
                        $params['replaceKeywords']['{ADDRESS_TYPE}'] = CommonHelper::getAddressType($orderData->address_type);
                        $params['toEmail'] = $restaurant->user->email;
                        // $params['toEmail'] = 'jaidev@nmgtechnologies.com';
                        $params['emailSlug'] = 'restaurant_order_completed';
                        $this->customMailer($params);
                    }


                    // change delivery person work status to "available"
                    $deliveryPerson->work_status = 0;
                    $deliveryPerson->save();
                    RiderEarning::updateOrCreate(['order_id' => $orderData->id, 'rider_id' => $deliveryPerson->id], ['amount' => $orderData->delivery_person_revenue]);

                    \app\Model\Vendor\OrderLog::saveLog($orderData->id, $orderData->order_status_id);
                    DB::commit();
                    \Session::flash('success', 'Order has been completed');
                }
            } else {
                \Session::flash('success', __('admin/flash_msg.error.orderNotFound'));
            }
            DB::commit();
            return \Redirect::back();
        } catch (\Throwable $th) {
            DB::rollBack();
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }
}
