<?php

namespace app\Http\Controllers\Admin;

use app\Http\Controllers\Admin\BaseController;
use app\Model\Admin\GlobalSetting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class GlobalSettingsController extends BaseController
{

    private $const;
    private $breadCrumb;
    private $url;
    private $title;
    private $subTitle;
    private $controllerUri;
    private $actionUri;

    /**
     * @Method: __construct()
     * @Scope: public
     * @Params: null
     * @Description: Initialize variable with module scope, breadcrumb
     * @Created 28/Nov/2019
     * @Updated 28/Nov/2019
     */
    public function __construct()
    {
        try {
            parent::__construct();
            $this->middleware('auth.admin');
            $this->url = config('admin.path.ADMIN_BASE_URL') . '/global-settings';
            $this->title = 'Manage Global Setting';
            $this->controllerUri = 'global-settings';

            $this->const['url'] = $this->url;
            $this->const['title'] = $this->title;
            $this->const['controllerUri'] = $this->controllerUri;
            $this->const['objValidationMsgs'] = 'global-setting';
            $this->const['objValidationJs'] = 'global-setting';
            $this->const = array_merge($this->adminConst(), $this->const);

            $this->breadCrumb['faClass'] = 'fa-table';
            $this->breadCrumb['breadCrumData'] = [
                0 => [
                    'text' => 'Dashboard',
                    'url' => config('admin.path.ADMIN_BASE_URL'),
                    'breadFaClass' => 'fa-dashboard',
                ],

            ];
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    public function getViewData()
    {
        $viewData['faClass'] = 'fa-table';
        $breadCrumData[0]['text'] = 'Dashboard';
        $breadCrumData[0]['url'] = url('/' . config("admin.constants.ADMIN_ROUTE_NAME") . '/dashboard');
        $breadCrumData[0]['breadFaClass'] = 'fa-dashboard';
        $viewData['breadCrumData'] = $breadCrumData;
        return $viewData;
    }

    /**
     * @Method: index()
     * @Scope: public
     * @Params: @page, @limit, @search
     * @returns: A List grlobal setting has been returnd
     * @Description: This method list all the global setting in the system & only super admin can access it.
     * A super administrator can add, edit , delete & assigns permission to other global settings
     * @Created 28/Nov/2019
     * @Updated 28/Nov/2019
     */
    public function index(Request $request)
    {
        try {
            $this->breadCrumb['breadCrumData'][1] = [
                'text' => $this->const['title'],
            ];
            $page = $request->query('page', 1);
            $limit = $request->query('limit', config('admin.constants.DEFAULT_PAGE_LIMIT'));
            $this->const['page'] = $page;
            $this->const['limit'] = $limit;

            $search = !empty($request->search) ? $request->search : '';

            $data = [];

            /* \DB::enableQueryLog(); */
            $query = GlobalSetting::select(
                'global_settings.id',
                'global_settings.slug',
                'global_settings.title',
                'global_settings.datatype',
                'global_settings.value',
                'global_settings.status',
                'global_settings.created_by',
                'global_settings.updated_by',
                'global_settings.created_at',
                'global_settings.updated_at'
            );

            if (!empty($search)) {

                $query->where('slug', 'LIKE', "%$search%");
            }

            $globalList = $query->sortable(['created_at' => 'asc'])->paginate($limit);

            $data['globalsettings'] = $globalList;
            $data = array_merge($data, $this->const, $this->breadCrumb);
            return \View('Admin.' . $this->controllerUri . '.index')->with($data);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * @Method: edit()
     * @Scope: public
     * @params: @id
     * @returns: Returns pre-populated form data with other constants used in UI
     * @Description: It only fetches global setting data as per given id & return it to form
     *  for pre-populating it.
     * @Created 28/Nov/2019
     * @Updated 28/Nov/2019
     */
    public function edit($id = null)
    {
        try {

            if (request()->ajax()) {

                if (request()->isMethod('get')) {

                    if (request()->isJson()) {
                        $input = request()->json()->all();
                    } else {
                        $input = request()->all();
                    }
                    $id = \Crypt::decryptString($input['id']);
                    $model = \Crypt::decryptString($input['model']);
                    $modelStatus = \DB::table($model)->select('status')->find($id);
                    $modelData = array('status' => '1');
                    $modelValue = array('status' => '<span style="cursor:pointer" class="glyphicon glyphicon-ok" title="Active"></span>');

                    if ($modelStatus->status == '1') {
                        $modelData = array('status' => '0');
                        $modelValue = array('status' => '<span style="cursor:pointer" class="glyphicon glyphicon-remove" title="Inactive"></span>');

                        $result = \DB::table($model)->where('id', $id)->update($modelData);
                    } else {
                        $result = \DB::table($model)->where('id', $id)->update($modelData);
                    }
                    if ($result) {
                        $status = '1';
                    }
                }
                return response()->json(['success' => $status, 'changeStatus' => $modelValue, 'id' => $id]);
            }

            // method/action specific initializations
            $this->subTitle = 'Edit';
            $this->actionUri = 'edit';
            $formId = 'globalsettingForm';

            //pre-populated data
            $this->const['subTitle'] = $this->subTitle;
            $this->const['actionUri'] = $this->actionUri;
            $this->const['formId'] = $formId;

            //push current controller then method in breadcrumb at position 1 & 2
            $this->breadCrumb['breadCrumData'][1] = [
                'text' => $this->title,
                'url' => $this->url,
                'breadFaClass' => '',
            ];
            $this->breadCrumb['breadCrumData'][2] = [
                'text' => $this->subTitle,
                'breadFaClass' => 'active',
            ];

            $id = \Crypt::decryptString($id);
            $globalSetting = GlobalSetting::select(
                'id',
                'slug',
                'title',
                'value',
                'status'
            )
                ->where('id', $id)
                ->first();

            if (!empty($globalSetting)) {
                $data['settingDetail'] = $globalSetting;
            }

            $data = array_merge($data, $this->const, $this->breadCrumb);
            return \View('Admin.' . $this->controllerUri . '.edit')->with($data);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * @Method: update()
     * @Scope: public
     * @params: @id
     * @input: @name, @description, @status
     * @returns: redirect to global setting listing page after successful submission
     * @Description: Update the specified resource in storage.
     * @Created 28/Nov/2019
     * @Updated 28/Nov/2019
     */
    public function update(Request $request, $id = null)
    {
        try {
            $id = \Crypt::decryptString($id);
            if ($request->isMethod('put')) {
                $input = $request->only(['value']);
                $validator = Validator::make($input, GlobalSetting::rules($id), GlobalSetting::messages());
                if ($validator->fails()) {
                    return \Redirect::back()->withInput()->withErrors($validator->errors());
                } else {
                    if (!GlobalSetting::find($id)->update([
                        'value' => trim($input['value']),
                    ])) {
                        $rollBack = true;
                        return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.UnableToUpdateGlobalSetting'));
                    }
                    \Session::flash('success', __('admin/flash_msg.success.GlobalSettingUpdated'));
                    return \Redirect::to($this->url);
                }
            }
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }
}
