<?php

namespace app\Http\Controllers\Admin;

use app\Http\Controllers\Admin\BaseController;
use app\Model\Admin\Role;
use app\Model\Admin\User;
use app\Model\Admin\FoodCategory;
use app\Model\Admin\Restaurant_menu;
use Auth;
use DB;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Input;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;

class FoodCategoryController extends BaseController
{
    private $const;
    private $roleId;
    private $roleName;
    private $breadCrumb;
    private $url;
    private $title;
    private $subTitle;
    private $controllerUri;
    private $actionUri;
    /**
     * @Method: __construct()
     * @Scope: public
     * @Params: null
     * @Description: Initialize variables like  @roleId, @roleName, @url within private scope of current class
     * @Created 28/Apr/2021
     * @Updated 28/Apr/2021
     */
    public function __construct(FoodCategory $foodCategory)
    {
        try {
            parent::__construct();
            $this->middleware('auth.admin');

            //fixed role for this module
            $this->roleId = config('admin.constants.RESTAURANT_OWNER_ROLE');
            $role = Role::select('name')->where('id', $this->roleId)->first();
            if ($role->name) {
                $this->roleName = $role->name;
            }

            $this->url = config('admin.path.ADMIN_BASE_URL') . '/category';
            $this->title = 'Manage Categories';
            $this->controllerUri = 'category';

            $this->const['url'] = $this->url;
            $this->const['title'] = $this->title;
            $this->const['controllerUri'] = $this->controllerUri;
            $this->const['roleId'] = $this->roleId;
            $this->const['roleName'] = $this->roleName;

            //used in adminForm layout to load validation messges in js variable
            $this->const['objValidationMsgs'] = 'category';
            $this->const['objValidationJs'] = 'category';


            //merge constant into single array with parent controller values
            $this->const = array_merge($this->adminConst(), $this->const);

            //default breadcrumb option (Dashboard) for every route
            $this->breadCrumb['faClass'] = 'fa-table';
            $this->breadCrumb['breadCrumData'] = [
                0 => [
                    'text' => 'Dashboard',
                    'url' => config('admin.path.ADMIN_BASE_URL'),
                    'breadFaClass' => 'fa-dashboard',
                ],

            ];
            $this->foodCategory = $foodCategory;
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }
    /**
     * Author: Dipak Verma
     * Email: dipak.verma@nmgtechnologies.com
     * @Method: index()
     * @Scope: public
     * @Params: @page,
     * @returns: A List of Food Category 
     * @Description: This function fetches  food category from database, sets up
     *  breadcrumb & other constants. 
     * @Created 26/Apr/2021
     * @Updated 26/Apr/2021
     */
    public function index(Request $request)
    {
        try {
            $this->breadCrumb['breadCrumData'][1] = [
                'text' => $this->const['title'],
            ];
            $page = $request->query('page', 1);
            $limit = $request->query('limit', config('admin.constants.DEFAULT_PAGE_LIMIT'));
            $this->const['page'] = $page;
            $this->const['limit'] = $limit;
            $foodCategory = $this->foodCategory->category($request);
            $data['userList'] = $foodCategory;
            $data = array_merge($data, $this->const, $this->breadCrumb);
            return \View('Admin.category.index')->with($data);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

/**
     * Author: NMG
     * Email: dipak.verma@nmgtechnologies.com
     * @Method: index()
     * @Scope: public
     * @Params: @page,
     * @returns: Update fav list 
     * @Description:  Update fav list
     * @Created 20/Sep/2022
     * @Updated 20/Sep/2022
     */
    public function makeFavroite(Request $request)
    {
        try {
            if ($request->isMethod('post')) {
                $input = $request->only(['status','id']); 
                FoodCategory::where('id',$input['id'])->update(['isFav'=>$input['status']]);
               return 1;
                 
            }
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }


    

    /**
     * Author: Dipak Verma
     * Email: dipak.verma@nmgtechnologies.com
     * @Method: create()
     * @Scope: public
     * @input: category
     * @Description: Show input form
     * @Created 28/Apr/2021
     * @Updated 28/Apr/2021
     */
    public function create()
    {
        try {
            // method/action specific initializations
            $this->subTitle = 'Add';
            $this->actionUri = 'create';
            $formId = 'createCategory';

            //pre-populated data
            $this->const['subTitle'] = $this->subTitle;
            $this->const['actionUri'] = $this->actionUri;
            $this->const['formId'] = $formId;
            $this->const['roles'] = [$this->roleId => $this->roleName];

            //push current controller then method in breadcrumb at position 1 & 2
            $this->breadCrumb['breadCrumData'][1] = [
                'text' => $this->title,
                'url' => $this->url,
                'breadFaClass' => '',
            ];
            $this->breadCrumb['breadCrumData'][2] = [
                'text' => $this->subTitle,
                'breadFaClass' => 'active',
            ];

            $data = array_merge($this->const, $this->breadCrumb);
            return \View('Admin.' . $this->controllerUri . '.create')->with($data);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }
    /**
     * Author: Dipak Verma
     * Email: dipak.verma@nmgtechnologies.com
     * @Method: store()
     * @Scope: public
     * @input: category
     * @returns: After successful submition of data redirect to listing page 
     * @Description: Create the specified resource in storage.
     * @Created 28/Apr/2021
     * @Updated 28/Apr/2021
     */
    public function store(Request $request)
    {

        try {
            if ($request->isMethod('post')) {
                $input = $request->only(['category']);
                $validator = Validator::make($input, FoodCategory::Rules(), FoodCategory::messages());

                if ($validator->fails()) {
                    return \Redirect::back()->withInput()->withErrors($validator->errors());
                } else {

                    $category = FoodCategory::create([
                        'name' => trim($input['category']),
                        'status' => 1,
                    ]);

                    if ($request->hasFile('image')) {
                        $field = 'image';
                        FoodCategory::upload_doc($request->image, $category->id, $field);
                    }

                    \Session::flash('success', __('admin/flash_msg.success.CategoryCreated'));
                    return \Redirect::to($this->url);
                }
            }
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * Author: Dipak Verma
     * Email: dipak.verma@nmgtechnologies.com
     * @Method: edit()
     * @Scope: public
     * @params: @id
     * @returns: Returns pre-populated form data with other constants used in UI
     * @Description: It only fetches food category detail as per given id & return it to form
     *  for pre-populating it.
     * @Created 28/Apr/2021
     * @Updated 28/Apr/2021
     */
    public function edit($id)
    {
        try {

            if (request()->ajax()) {

                if (request()->isMethod('get')) {

                    if (request()->isJson()) {
                        $input = request()->json()->all();
                    } else {
                        $input = request()->all();
                    }
                    $id = \Crypt::decryptString($input['id']);
                    $model = 'food_categories';
                    $modelStatus = \DB::table($model)->select('status')->find($id);
                    $modelData = array('status' => '1');
                    $modelValue = array('status' => '<span style="cursor:pointer" class="glyphicon glyphicon-ok" title="Active"></span>');

                    if ($modelStatus->status == '1') {
                        $modelData = array('status' => '0');
                        $modelValue = array('status' => '<span style="cursor:pointer" class="glyphicon glyphicon-remove" title="Inactive"></span>');

                        $result = \DB::table($model)->where('id', $id)->update($modelData);
                    } else {
                        $result = \DB::table($model)->where('id', $id)->update($modelData);
                    }
                    if ($result) {
                        $status = '1';
                    }
                }
                return response()->json(['success' => $status, 'changeStatus' => $modelValue, 'id' => $id]);
            }


            // method/action specific initializations
            $this->subTitle = 'Edit';
            $this->actionUri = 'edit';
            $formId = 'createCategory';

            //pre-populated data
            $this->const['subTitle'] = $this->subTitle;
            $this->const['actionUri'] = $this->actionUri;
            $this->const['formId'] = $formId;
            $this->const['roles'] = [$this->roleId => $this->roleName];

            //push current controller then method in breadcrumb at position 1 & 2
            $this->breadCrumb['breadCrumData'][1] = [
                'text' => $this->title,
                'url' => $this->url,
                'breadFaClass' => '',
            ];
            $this->breadCrumb['breadCrumData'][2] = [
                'text' => $this->subTitle,
                'breadFaClass' => 'active',
            ];

            $id = \Crypt::decryptString($id);

            $deliveryCharge = \app\Model\Admin\FoodCategory::select('id', 'name', 'image')
                ->where('id', $id)->first();
            $data['userDetail'] = $deliveryCharge;

            $data = array_merge($data, $this->const, $this->breadCrumb);
            return \View('Admin.' . $this->controllerUri . '.edit')->with($data);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }
    /**
     * Author: Dipak Verma
     * Email: dipak.verma@nmgtechnologies.com
     * @Method: update()
     * @Scope: public
     * @params: @id
     * @input: category
     * @returns: Returns pre-populated form data with other constants used in UI
     * @Description: Update the specified resource in storage.
     * @Created 28/Apr/2021
     * @Updated 28/Apr/2021
     */
    public function update(Request $request, $id)
    {
        try {
            $id = \Crypt::decryptString($id);
            if ($request->isMethod('put')) {
                $input = $request->only(['category']);
                $validator = Validator::make($input, FoodCategory::Rules($id), FoodCategory::messages());
                if ($validator->fails()) {
                    return \Redirect::back()->withInput()->withErrors($validator->errors());
                } else {
                    $category = FoodCategory::find($id);
                    $category->name = trim($input['category']);
                    $recdUpdated = $category->save();
                    if ($request->hasFile('image')) {
                        $field = 'image';
                        FoodCategory::upload_doc($request->image, $category->id, $field);
                    }
                    \Session::flash('success', __('admin/flash_msg.success.CategoryUpdated'));
                    return \Redirect::to($this->url);
                }
            }
        } catch (\Exception $e) {
            // dd($e);
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * Author: Dipak Verma
     * Email: dipak.verma@nmgtechnologies.com
     * @Method: destroy()
     * @Scope: public
     * @params: @id
     * @Description: Remove the specified resource from storage then redirect to listing page
     * @Created 28/Apr/2021
     * @Updated 28/Apr/2021
     */
    public function destroy($id)
    {
        try {
            $id = \Crypt::decryptString($id);
            $category = FoodCategory::find($id);
            // check if category associate with any food items

            $checkLinked = Restaurant_menu::where(['food_category_id' => $id])->count();
            if ($checkLinked > 0) {
                \Session::flash('error', "We cannot remove this category because it is already associated with an food item");
                return \Redirect::to($this->url);
            }
            $category->delete();
            \Session::flash('success', __('admin/flash_msg.success.CategoryDeleted'));
            return \Redirect::to($this->url);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }
}
