<?php

namespace app\Http\Controllers\Admin;

use app\Http\Controllers\Admin\BaseController;
use app\Model\Admin\EmailTemplate;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Input;
use Illuminate\Support\Facades\Validator;
use \app\Model\Admin\EmailType;

class EmailTemplatesController extends BaseController
{

    private $const;
    private $breadCrumb;
    private $url;
    private $title;
    private $subTitle;
    private $controllerUri;
    private $actionUri;
    private $data = [];

    /**
     * @Name: Sandeep
     * @Method: __construct()
     * @Scope: public
     * @Params: null
     * @Description: Initialize variable with module scope, breadcrumb
     * @Created 02/Dec/2019
     * @Updated 02/Dec/2019
     */
    public function __construct()
    {
        try {
            parent::__construct();
            $this->middleware('auth.admin');

            $this->url = config('admin.path.ADMIN_BASE_URL') . '/email-templates';
            $this->title = 'Email Template';
            $this->controllerUri = 'email-templates';

            //department logo path & base urls
            $this->const['url'] = $this->url;
            $this->const['title'] = $this->title;
            $this->const['controllerUri'] = $this->controllerUri;

            //used in adminForm layout to load validation messges in js variable
            $this->const['objValidationMsgs'] = 'email-template';
            $this->const['objValidationJs'] = 'email-template';

            $this->const = array_merge($this->adminConst(), $this->const);

            //default breadcrumb option (Dashboard) for every route
            $this->breadCrumb['faClass'] = 'fa-table';
            $this->breadCrumb['breadCrumData'] = [
                0 => [
                    'text' => 'Dashboard',
                    'url' => config('admin.path.ADMIN_BASE_URL'),
                    'breadFaClass' => 'fa-dashboard',
                ],

            ];
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * @Name: Sandeep
     * @Method: index()
     * @Scope: public
     * @Params: @page, @limit, @search
     * @returns: Display a listing of the resource
     * @Description: email_template table email template stores all the email used in various facilities
     * @Created 04/Dec/2019
     * @Updated 04/Dec/2019
     */
    public function index(Request $request)
    {
        try {
            $this->breadCrumb['breadCrumData'][1] = [
                'text' => $this->const['title'],
            ];

            $page = $request->query('page', 1);
            $limit = $request->query('limit', config('admin.constants.DEFAULT_PAGE_LIMIT'));
            $this->const['page'] = $page;
            $this->const['limit'] = $limit;

            $search = !empty($request->search) ? $request->search : '';

            $data = [];

            /* \DB::enableQueryLog(); */
            $query = EmailTemplate::select(
                'email_templates.id',
                'email_templates.email_type_id',
                'email_templates.locale',
                'email_templates.subject',
                'email_templates.content',
                'email_templates.status',

                'email_templates.created_at',
                'email_templates.updated_at'
            );

            if (!empty($search)) {
                $query->where('subject', 'LIKE', "%$search%");
                $query->orWhere('content', 'LIKE', "%$search%");
            }

            $emailList = $query->sortable(['created_at' => 'asc'])->paginate($limit);

            $this->data['emailTemplateList'] = $emailList;

            $data = array_merge($this->data, $this->const, $this->breadCrumb);
            return \View('Admin.email-templates.index')->with($data);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * @Name: Sandeep
     * @Method: edit()
     * @Scope: public
     * @params  int  $id
     * @Description: Show the form for editing the specified resource.
     * @Created 02/Dec/2019
     * @Updated 03/Dec/2019
     */
    public function edit($id = null)
    {
        try {

            if (request()->ajax()) {

                if (request()->isMethod('get')) {

                    if (request()->isJson()) {
                        $input = request()->json()->all();
                    } else {
                        $input = request()->all();
                    }
                    $id = \Crypt::decryptString($input['id']);
                    $model = \Crypt::decryptString($input['model']);
                    $modelStatus = \DB::table($model)->select('status')->find($id);
                    $modelData = array('status' => '1');
                    $modelValue = array('status' => '<span style="cursor:pointer" class="glyphicon glyphicon-ok" title="Active"></span>');

                    if ($modelStatus->status == '1') {
                        $modelData = array('status' => '0');
                        $modelValue = array('status' => '<span style="cursor:pointer" class="glyphicon glyphicon-remove" title="Inactive"></span>');

                        $result = \DB::table($model)->where('id', $id)->update($modelData);
                    } else {
                        $result = \DB::table($model)->where('id', $id)->update($modelData);
                    }
                    if ($result) {
                        $status = '1';
                    }
                }
                return response()->json(['success' => $status, 'changeStatus' => $modelValue, 'id' => $id]);
            }

            // method/action specific initializations
            $this->subTitle = 'Edit';
            $this->actionUri = 'edit';
            $formId = 'emailTemplateForm';

            //pre-populated data
            $this->const['subTitle'] = $this->subTitle;
            $this->const['actionUri'] = $this->actionUri;
            $this->const['formId'] = $formId;

            //push current controller then method in breadcrumb at position 1 & 2
            $this->breadCrumb['breadCrumData'][1] = [
                'text' => $this->title,
                'url' => $this->url,
                'breadFaClass' => '',
            ];
            $this->breadCrumb['breadCrumData'][2] = [
                'text' => $this->subTitle,
                'breadFaClass' => 'active',
            ];

            $id = \Crypt::decryptString($id);

            $this->data['emailTemplateDetail'] = EmailTemplate::FindOrFail($id);
            $this->data['emailTypes'] = EmailType::fetchAllTypes();

            $data = array_merge($this->data, $this->const, $this->breadCrumb);
            return \View('Admin.' . $this->controllerUri . '.edit')->with($data);

        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * @Name: Sandeep
     * @Method: update()
     * @Scope: public
     * @params: @id
     * @input: @name, @status
     * @Description: Update the specified resource in storage.
     * @Created 02/Dec/2019
     * @Updated 03/Dec/2019
     */
    public function update(Request $request, $id = null)
    {
        try {
            $id = \Crypt::decryptString($id);
            $emailTemplate = EmailTemplate::FindOrFail($id);
            if ($request->isMethod('put')) {
                $input = Input::all();
                $validator = Validator::make($input, EmailTemplate::Rules($id), EmailTemplate::messages());

                if ($validator->fails()) {
                    return \Redirect::back()->withInput()->withErrors($validator->errors());
                } else {
                    $input['id'] = $emailTemplate->id;
                    unset($input['email_type_id']);
                    $emailTemplate->fill($input)->save();

                    \Session::flash('success', __('admin/flash_msg.success.EmailTemplateUpdated'));
                    return \Redirect::to($this->url);
                }
            }
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * @Method: show()
     * @Scope: public
     * @params: @id
     * @Description: show detail of records
     * @Created 10/Dec/2019
     * @Updated 10/Dec/2019
     */
    public function show($id)
    {
        try {
            // method/action specific initializations
            $this->subTitle = 'View';
            $this->actionUri = 'show';

            //pre-populated data
            $this->const['subTitle'] = $this->subTitle;
            $this->const['actionUri'] = $this->actionUri;

            //push current controller then method in breadcrumb at position 1 & 2
            $this->breadCrumb['breadCrumData'][1] = [
                'text' => $this->title,
                'url' => $this->url,
                'breadFaClass' => '',
            ];
            $this->breadCrumb['breadCrumData'][2] = [
                'text' => $this->subTitle,
                'breadFaClass' => 'active',
            ];

            $id = \Crypt::decryptString($id);
            $emailTemplate = EmailTemplate::select(
                'email_templates.id',
                'email_templates.email_type_id',
                'email_templates.locale',
                'email_templates.subject',
                'email_templates.content',
                'email_templates.status',
                'email_templates.created_by'
            )->with('emailType')->where('id', $id)->first();
            $data['emailTemplateDetail'] = $emailTemplate;

            $data = array_merge($data, $this->const, $this->breadCrumb);
            return \View('Admin.' . $this->controllerUri . '.show')->with($data);

        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * @Method: uploadEditorImage()
     * @Scope: public
     * @params: $request
     * @Description: upload tinymce image on server
     * @Created 30/Dec/2019
     * @Updated 30/Dec/2019
     */
    public function uploadEditorImage()
    {
        $this->uploadEditorImageUpload($_FILES, 'email');
    }

}
