<?php

/**
 * @Description: The `DeliveryPersonController` is used for manage the delivery person user in MCA
 * application. User Can Register from mobile app as customer but he/she become a delivery person so update other detail and after this Admin approve
 * @auther: NMG - Jaidev
 * @dated: 26/June/2021
 */

namespace app\Http\Controllers\Admin;

use app\Http\Controllers\Admin\BaseController;
use app\Model\Admin\Role;
use app\Model\Admin\User;
use app\Model\Admin\DeliveryPersons;
use app\Model\Admin\Order;
use Auth;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Input;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use app\Components\Api\V1\PushNotification;

class DeliveryPersonController extends BaseController
{
    private $const;
    private $roleId;
    private $roleName;
    private $breadCrumb;
    private $url;
    private $title;
    private $subTitle;
    private $controllerUri;
    private $actionUri;

    /**
     * @Method: __construct()
     * @Scope: public
     * @Params: null
     * @Description: Initialize variables like  @roleId, @roleName, @url within private scope of current class
     * @Created 28/Apr/2021
     * @Updated 28/Apr/2021
     */

    public function __construct(User $u)
    {
        try {
            parent::__construct();
            $this->middleware('auth.admin');

            //fixed role for this module
            $this->roleId = config('admin.constants.DELIVERY_PERSON_ROLE');
            $role = Role::select('name')->where('id', $this->roleId)->first();
            if ($role->name) {
                $this->roleName = $role->name;
            }

            $this->url = config('admin.path.ADMIN_BASE_URL') . '/delivery-persons';
            $this->title = 'Manage Delivery Persons';
            $this->controllerUri = 'delivery-persons';

            $this->const['url'] = $this->url;
            $this->const['title'] = $this->title;
            $this->const['controllerUri'] = $this->controllerUri;
            $this->const['roleId'] = $this->roleId;
            $this->const['roleName'] = $this->roleName;

            //used in adminForm layout to load validation messges in js variable
            $this->const['objValidationMsgs'] = 'delivery-person';
            $this->const['objValidationJs'] = 'delivery-person';

            //merge constant into single array with parent controller values
            $this->const = array_merge($this->adminConst(), $this->const);

            //default breadcrumb option (Dashboard) for every route
            $this->breadCrumb['faClass'] = 'fa-table';
            $this->breadCrumb['breadCrumData'] = [
                0 => [
                    'text' => 'Dashboard',
                    'url' => config('admin.path.ADMIN_BASE_URL'),
                    'breadFaClass' => 'fa-dashboard',
                ],

            ];
            $this->u = $u;
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * Author: Dipak Verma
     * Email: dipak.verma@nmgtechnologies.com
     * @Method: index()
     * @Scope: public
     * @Params: @page,
     * @returns: A List of delivery person users identified by Delivery Person role(id:4)
     * @Description: This function fetches  delivery person users from database, sets up
     *  breadcrumb & other constants.
     * @Created 26/Apr/2021
     * @Updated 26/Apr/2021
     */
    public function index(Request $request)
    {
        try {
            $this->breadCrumb['breadCrumData'][1] = [
                'text' => $this->const['title'],
            ];
            $page = $request->query('page', 1);
            $limit = $request->query('limit', config('admin.constants.DEFAULT_PAGE_LIMIT'));
            $this->const['page'] = $page;
            $this->const['limit'] = $limit;
            $deliveryPersons = DeliveryPersons::list($request);
            $riderList = DeliveryPersons::pluck('id', 'id')->toArray();
            // dd($riderList);
            $data['riderList'] = !empty($riderList) ? $riderList : [];
            $data['userList'] = $deliveryPersons;
            $data = array_merge($data, $this->const, $this->breadCrumb);
            return view('Admin.delivery-persons.index')->with($data);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * Author: Dipak Verma
     * Email: dipak.verma@nmgtechnologies.com
     * @Method: show()
     * @Scope: public
     * @params: @id
     * @Description: show detail of records
     * @Created 28/Apr/2021
     * @Updated 28/Apr/2021
     */
    public function show($id)
    {
        try {
            // method/action specific initializations
            $this->subTitle = 'View';
            $this->actionUri = 'show';

            //pre-populated data
            $this->const['subTitle'] = $this->subTitle;
            $this->const['actionUri'] = $this->actionUri;

            //push current controller then method in breadcrumb at position 1 & 2
            $this->breadCrumb['breadCrumData'][1] = [
                'text' => $this->title,
                'url' => $this->url,
                'breadFaClass' => '',
            ];
            $this->breadCrumb['breadCrumData'][2] = [
                'text' => $this->subTitle,
                'breadFaClass' => 'active',
            ];

            $id = \Crypt::decryptString($id);
            $user = DeliveryPersons::select(
                'delivery_persons.id',
                'delivery_persons.user_id',
                'delivery_persons.age',
                'delivery_persons.photo',
                'delivery_persons.id_proof',
                'delivery_persons.vehicle_info',
                'delivery_persons.driving_license',
                'delivery_persons.bank_name',
                'delivery_persons.account_number',
                'delivery_persons.status',
                'delivery_persons.created_at'
            )
                ->with('user')
                ->where('id', $id)
                ->first();


            //echo "<pre>@".$id; print_r($user);  exit;
            $data['userDetail'] = $user;
            $data = array_merge($data, $this->const, $this->breadCrumb);
            return \View('Admin.' . $this->controllerUri . '.show')->with($data);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }
    /**
     * Author: Dipak Verma
     * Email: dipak.verma@nmgtechnologies.com
     * @Method: edit()
     * @Scope: public
     * @params: @id
     * @returns: Returns pre-populated form data with other constants used in UI
     * @Description: It only fetches delivery person detail as per given id & return it to form
     *  for pre-populating it.
     * @Created 28/Apr/2021
     * @Updated 28/Apr/2021
     */
    public function edit($id)
    {
        try {

            if (request()->ajax()) {

                if (request()->isMethod('get')) {

                    if (request()->isJson()) {
                        $input = request()->json()->all();
                    } else {
                        $input = request()->all();
                    }
                    $id = \Crypt::decryptString($input['id']);
                    $model = 'delivery_persons';
                    $modelStatus = \DB::table($model)->select('status')->find($id);
                    $modelData = array('status' => '1');
                    $modelValue = array('status' => '<span style="cursor:pointer" class="glyphicon glyphicon-ok" title="Active"></span>');

                    if ($modelStatus->status == '1') {

                        // prevent to block if any active order assigned to delivery person
                        $activeOrderStatus[] = config('admin.constants.ORDER_STATUS_IN_PREP');
                        $activeOrderStatus[] = config('admin.constants.ORDER_STATUS_ON_THE_WAY');
                        // $activeOrderStatus[] = config('admin.constants.ORDER_STATUS_COMPLETED');
                        $activeOrderCount = Order::whereIn('order_status_id', $activeOrderStatus)
                            ->where(['delivery_person_id' => $id])->get()->count();
                        if ($activeOrderCount > 0) {
                            return response()->json(['success' => false, 'msg' => 'We cannot block this user because it is already associated with an active order']);
                        }
                        $modelData = array('status' => '0');
                        $modelValue = array('status' => '<span style="cursor:pointer" class="glyphicon glyphicon-remove" title="Inactive"></span>');

                        $result = \DB::table($model)->where('id', $id)->update($modelData);
                    } else {
                        $result = \DB::table($model)->where('id', $id)->update($modelData);
                    }
                    if ($result) {
                        $status = '1';

                        $deliveryPerson = DeliveryPersons::find($id);
                        // send push notification to refresh screen
                        $title = "Rider Status";
                        $body = "Account status has changed";
                        $notification = array('title' => $title, 'body' => $body, 'sound' => 'default', 'badge' => '1', 'orderId' => null, 'actionId' => config('api.constants.push_notification.actionId.rider_online_offline'));
                        PushNotification::sendNotification($deliveryPerson->user_id,  $notification, null);
                    }
                }
                return response()->json(['success' => $status, 'changeStatus' => $modelValue, 'id' => $id]);
            }

            // method/action specific initializations
            $this->subTitle = 'Edit';
            $this->actionUri = 'edit';
            $formId = 'editDeliveryPerson';

            //pre-populated data
            $this->const['subTitle'] = $this->subTitle;
            $this->const['actionUri'] = $this->actionUri;
            $this->const['formId'] = $formId;
            $this->const['roles'] = [$this->roleId => $this->roleName];

            //push current controller then method in breadcrumb at position 1 & 2
            $this->breadCrumb['breadCrumData'][1] = [
                'text' => $this->title,
                'url' => $this->url,
                'breadFaClass' => '',
            ];
            $this->breadCrumb['breadCrumData'][2] = [
                'text' => $this->subTitle,
                'breadFaClass' => 'active',
            ];

            $id = \Crypt::decryptString($id);

            $user = DB::table('users')
                ->join('delivery_persons', 'users.id', '=', 'delivery_persons.user_id')
                ->select('users.*', 'delivery_persons.*', 'delivery_persons.id as delivery_person_id')
                ->where('delivery_persons.id', $id)
                ->first();
            $data['userDetail'] = $user;

            $data = array_merge($data, $this->const, $this->breadCrumb);
            return \View('Admin.' . $this->controllerUri . '.edit')->with($data);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }


    /**
     * Author: Dipak Verma
     * Email: dipak.verma@nmgtechnologies.com
     * @Method: update()
     * @Scope: public
     * @params: @id
     * @input: @user_name,@email,@phone_no,@mobile_no,first_name,@last_name,@staus,@avatar,
     * @age,@id_proof,@vehicle_info,@druving_license,@account_details
     * @returns: Returns pre-populated form data with other constants used in UI
     * @Description: Update the specified resource in storage.
     * @Created 28/Apr/2021
     * @Updated 28/Apr/2021
     */
    public function update(Request $request, $id)
    {
        try {
            $id = \Crypt::decryptString($id);
            if ($request->isMethod('put')) {
                $rollBack = false;
                \DB::beginTransaction();
                $input = $request->only(['user_name', 'email', 'phone_number', 'mobile_no', 'first_name', 'last_name', 'status', 'avatar', 'age', 'id_proof', 'vehicle_info', 'driving_license', 'account_number', 'bank_name']);
                $validator = Validator::make($input, DeliveryPersons::Rules($id), DeliveryPersons::messages());
                if ($validator->fails()) {
                    return \Redirect::back()->withInput()->withErrors($validator->errors());
                } else {
                    $deliveryPerson = DeliveryPersons::find($id);
                    $uid = $deliveryPerson->user_id;
                    $deliveryPerson->age = trim($input['age']);
                    $deliveryPerson->account_number = trim($input['account_number']);
                    $deliveryPerson->bank_name = trim($input['bank_name']);
                    $deliveryPerson->status = trim($input['status']);

                    $facilityUser = User::find($uid);
                    $oldEmail = $facilityUser->email;
                    $facilityUser->role_id = $this->roleId;
                    if (!empty($input['email'])) {
                        $facilityUser->email = trim($input['email']);
                    }
                    $facilityUser->phone_number = trim($input['phone_number']);
                    $facilityUser->first_name = trim($input['first_name']);
                    $facilityUser->last_name = trim($input['last_name']);

                    if ($facilityUser->isDirty('email')) {
                        ###Welcome Email#####
                        if (!empty($input['lang'])) {
                            \App::setLocale($input['lang']);
                        }
                        $params['replaceKeywords']['{USER_NAME}'] = $facilityUser->user_name;
                        $params['replaceKeywords']['{OLD_EMAIL}'] = $oldEmail;
                        $params['replaceKeywords']['{NEW_EMAIL}'] = trim($input['email']);
                        $params['toEmail'] = trim($input['email']);
                        $params['emailSlug'] = 'change_email';
                        if (!$this->customMailer($params)) {
                            $rollBack = true;
                        }
                        #######################
                    }
                    $delpUpdated = $deliveryPerson->save();
                    $recdUpdated = $facilityUser->save();

                    if ($delpUpdated) {
                        if ($request->hasFile('id_proof')) {
                            $field = 'id_proof';
                            if (!DeliveryPersons::upload_doc($request->id_proof, $id, $field)) {
                                return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.UnableToUploadAvatarImage'));
                            }
                        }
                    } else {
                        return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.UnableToUpdateFacilityUser'));
                    }

                    if ($delpUpdated) {
                        if ($request->hasFile('vehicle_info')) {
                            $field = 'vehicle_info';
                            if (!DeliveryPersons::upload_doc($request->vehicle_info, $id, $field)) {
                                return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.UnableToUploadAvatarImage'));
                            }
                        }
                    } else {
                        return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.UnableToUpdateFacilityUser'));
                    }

                    if ($delpUpdated) {
                        if ($request->hasFile('driving_license')) {
                            $field = 'driving_license';
                            if (!DeliveryPersons::upload_doc($request->driving_license, $id, $field)) {
                                return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.UnableToUploadAvatarImage'));
                            }
                        }
                    } else {
                        return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.UnableToUpdateFacilityUser'));
                    }

                    if ($recdUpdated) {
                        if ($request->hasFile('avatar')) {
                            if (!User::upload_profile_image($request->avatar, $uid)) {
                                return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.UnableToUploadAvatarImage'));
                            }
                        }
                    } else {
                        return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.UnableToUpdateFacilityUser'));
                    }
                    if (!$rollBack) {
                        \DB::commit();
                    } else {
                        \DB::rollBack();
                    }
                }
                if (!$rollBack) {
                    \DB::commit();
                    \Session::flash('success', __('admin/flash_msg.success.DeliveryPersonUpdated'));
                    return \Redirect::to($this->url);
                }
            }
        } catch (\Exception $e) {
            \DB::rollBack();
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }
    /**
     * Author: Dipak Verma
     * Email: dipak.verma@nmgtechnologies.com
     * @Method: destroy()
     * @Scope: public
     * @params: @id
     * @Description: Remove the specified resource from storage then redirect to listing page
     * @Created 20/Nov/2019
     * @Updated 20/Nov/2019
     */
    public function destroy($id)
    {
        try {
            $id = \Crypt::decryptString($id);
            // check order associate with this delivery person or not
            $assignedOrderCount = Order::where(['delivery_person_id' => $id])->get()->count();
            if ($assignedOrderCount > 0) {
                return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.orderAssociateWithDeliveryPerson'));
            }
            $dirPath = $this->const['avatarStoragePath'] . '/' . $id;

            if (is_dir($dirPath)) {
                deleteDir($dirPath);
            }
            $user = DeliveryPersons::find($id);
            $uid = $user->user_id;
            $user->delete();
            $public_roleId = config('admin.constants.PUBLIC_USERS_ROLE');
            $update_role = User::find($uid);
            $update_role->role_id = $public_roleId;
            $update_role->is_delivery_person = 0;
            $update_role->save();
            \Session::flash('success', __('admin/flash_msg.success.DeliveryPersonDeleted'));
            return \Redirect::to($this->url);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }



    /**
     * Author: Jaidev
     * Email: jaidev@nmgtechnologies.com
     * @Method: requestStatusHandler()
     * @Scope: public
     * @params: @id
     * @returns: Returns status and response  of executed sql code
     * @Description: This method is used to handle delivery person approval status from admin end
     * @Created 23/June/2021
     * @Updated 23/June/2021
     */


    public function requestStatusHandler(Request $request)
    {
        if (request()->ajax()) {

            if (request()->isMethod('post')) {

                if (request()->isJson()) {
                    $input = request()->json()->all();
                } else {
                    $input = request()->all();
                }
                $status = 0;
                $id = \Crypt::decryptString($input['id']);
                $is_approve_status = $input['status'];

                $model = 'delivery_persons';
                $adminUrl = config('admin.path.ADMIN_BASE_URL');
                $rejectUrl = $adminUrl . '/' . $this->controllerUri . '/rejectRequestForm/' . $input['id'];
                $approveUrl = $adminUrl . '/' . $this->controllerUri . '/requestStatusHandler';

                $disapproveBtn = '<a href="javascript:void(0)"  data="' . $input['id'] . '"
            baseUrl="' . $rejectUrl . '"
            model="' . \Crypt::encryptString($this->controllerUri) . '" class="change-approve-status"
            msg="' . __('admin/flash_msg.error.ConfirmChangeStatus') . '"
            status="reject"
            >
            <button type="button" class="btn btn-primary btn-sm buttonload" title="Click to Disapprove Request" style="padding: 2px 10px;">Disapprove</button>
        </a>';

                $approveBtn = '<a href="javascript:void(0)"  data="' . $input['id'] . '"
            baseUrl="' . $approveUrl . '"
            model="' . \Crypt::encryptString($this->controllerUri) . '" class="change-approve-status"
            msg="' . __('admin/flash_msg.error.ConfirmChangeStatus') . '"
            status="approve"
            >
            <button type="button" class="btn btn-success btn-sm buttonload" title="Click to Approve Request" style="padding: 2px 16px;">Approve &nbsp;&nbsp;</button>
        </a>';

                if ($is_approve_status == 'approve') {
                    // approve request
                    $modelData = array('is_approved' => '1', 'status' => 1);
                    $modelValue = array('status' => ' <button type="button" class="btn btn-success btn-sm" title="Approved" disabled="disabled" class="change-status-confirm" style="padding: 2px 20px; margin-bottom: 10px; font-size: 14px;">Approved </button><br>' . $disapproveBtn);
                    $result = \DB::table($model)->where('id', $id)->update($modelData);
                    $deliveryPerson = DeliveryPersons::find($id);
                    $user = User::find($deliveryPerson->user_id);
                    $user->is_delivery_person = 1;
                    $user->save();

                    // send push notification to delivery person
                    $title = "Profile approved";
                    $body = "Your request for delivery person has been approved";
                    $notification = array('title' => $title, 'body' => $body, 'sound' => 'default', 'badge' => '1', 'orderId' => null, 'actionId' => config('api.constants.push_notification.actionId.rider_profile'));

                    PushNotification::sendNotification($user->id, $notification,  null);

                    if (!empty($user->email)) {

                        ###Approved  Email#####
                        if (!empty($input['lang'])) {
                            \App::setLocale($input['lang']);
                        }
                        $params['replaceKeywords']['{USER_NAME}'] = $user->first_name . ' ' . $user->last_name;
                        $params['toEmail'] = $user->email;
                        $params['emailSlug'] = 'rider_account_approved';
                        $this->customMailer($params);
                        #############################
                    }
                } else {
                    // save reject request with rejection reason
                    $modelData = array('status' => '0', 'reject_reason' => $input['reject_reason']);

                    $modelValue = array('status' => '<button type="button" class="btn btn-primary btn-sm" title="Disapproved" disabled="disabled" style="padding: 2px 9px; font-size: 14px; margin-bottom: 10px;">Disapproved</button><br>' . $approveBtn);




                    $deliveryPerson = DeliveryPersons::find($id);
                    // check if its first time then set status to inactive
                    if ($deliveryPerson->is_approved == 0) {
                        $modelData['is_approved'] = 2;
                        $title = "Profile rejected";
                        $body = "Your request for delivery person has been rejected";
                    } else {
                        $title = "Account De-Activated";
                        $body = "Your account has been de-active by admin";
                    }

                    $result = \DB::table($model)->where('id', $id)->update($modelData);
                    $user = User::find($deliveryPerson->user_id);
                    $user->is_delivery_person = 0;
                    $user->save();
                    // send push notification to delivery person

                    $notification = array('title' => $title, 'body' => $body, 'sound' => 'default', 'badge' => '1', 'orderId' => null, 'actionId' => config('api.constants.push_notification.actionId.rider_profile'));


                    PushNotification::sendNotification($user->id, $notification, null);

                    if (!empty($user->email)) {

                        ###Disapproved  Email#####
                        if (!empty($input['lang'])) {
                            \App::setLocale($input['lang']);
                        }
                        $params['replaceKeywords']['{USER_NAME}'] = $user->first_name . ' ' . $user->last_name;
                        $params['replaceKeywords']['{REJECT_REASON}'] = $modelData['reject_reason'];
                        $params['toEmail'] = $user->email;
                        $params['emailSlug'] = 'rider_account_disapproved';
                        $this->customMailer($params);
                        #############################
                    }
                }
                if ($result) {
                    $status = '1';
                }
            }
            return response()->json(['success' => $status, 'changeStatus' => $modelValue, 'id' => $id]);
        } else {
            return response()->json(['success' => false, 'msg' => "Invalid Request"]);
        }
    }
}
