<?php

/**
 * @Description: The `CustomerController` is used for manage the public user or customer in BMF
 * application. 
 * @auther: NMG 
 * @dated: 28/Apr/2021
 */

namespace app\Http\Controllers\Admin;

use app\Http\Controllers\Admin\BaseController;
use app\Model\Admin\Role;
use app\Model\Admin\User;
use app\Model\Admin\UserAvatars;
use app\Model\Admin\Customer;
use app\Model\Admin\Order;
use app\Model\Admin\DeliveryPersons;
use app\Model\Admin\Notification;

use Auth;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Input;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use app\Model\Admin\User_address;
// use Notification;

class CustomerController extends BaseController
{
    private $const;
    private $roleId;
    private $roleName;
    private $breadCrumb;
    private $url;
    private $title;
    private $subTitle;
    private $controllerUri;
    private $actionUri;

    /**
     * Author: Dipak Verma
     * Email: dipak.verma@nmgtechnologies.com
     * @Method: __construct()
     * @Scope: public
     * @Params: null
     * @Description: Initialize variables like @avatarStoragePath, @roleId, @roleName, @url within private scope of current class
     * @Created 28/Apr/2021
     * @Updated 28/Apr/2021
     */

    public function __construct(Customer $customer)
    {
        try {
            parent::__construct();
            $this->middleware('auth.admin');

            //fixed role for this module
            $this->roleId = config('admin.constants.PUBLIC_USERS_ROLE');
            $role = Role::select('name')->where('id', $this->roleId)->first();
            if ($role->name) {
                $this->roleName = $role->name;
            }

            $this->url = config('admin.path.ADMIN_BASE_URL') . '/customer';
            $this->title = 'Manage Customers';
            $this->controllerUri = 'customer';

            $this->const['url'] = $this->url;
            $this->const['title'] = $this->title;
            $this->const['controllerUri'] = $this->controllerUri;
            $this->const['roleId'] = $this->roleId;
            $this->const['roleName'] = $this->roleName;

            //used in adminForm layout to load validation messges in js variable
            $this->const['objValidationMsgs'] = 'user';
            $this->const['objValidationJs'] = 'user';

            //merge constant into single array with parent controller values
            $this->const = array_merge($this->adminConst(), $this->const);

            //default breadcrumb option (Dashboard) for every route
            $this->breadCrumb['faClass'] = 'fa-table';
            $this->breadCrumb['breadCrumData'] = [
                0 => [
                    'text' => 'Dashboard',
                    'url' => config('admin.path.ADMIN_BASE_URL'),
                    'breadFaClass' => 'fa-dashboard',
                ],

            ];
            $this->customer = $customer;
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }
    /**
     * Author: Dipak Verma
     * Email: dipak.verma@nmgtechnologies.com
     * @Method: index()
     * @Scope: public
     * @Params: @page, @limit, @search
     * @returns: A List of  customer users identified by Public User role(id:3)
     * @Description: This function fetches customers list from database, sets up
     *  breadcrumb & other constants. 
     * @Created 28/Apr/2021
     * @Updated 17/Jun/2021
     */
    public function index(Request $request)
    {
        try {
            $this->breadCrumb['breadCrumData'][1] = [
                'text' => $this->const['title'],
            ];
            $page = $request->query('page', 1);
            $limit = $request->query('limit', config('admin.constants.DEFAULT_PAGE_LIMIT'));
            $this->const['page'] = $page;
            $this->const['limit'] = $limit;
            $customerList = $this->customer->list($request);
            //dd($customerList);
            $data['userList'] = $customerList;
            $data = array_merge($data, $this->const, $this->breadCrumb);
            return \View('Admin.customer.index')->with($data);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * Author: Dipak Verma
     * Email: dipak.verma@nmgtechnologies.com
     * @Method: show()
     * @Scope: public
     * @params: @id
     * @Description: show detail of records
     * @Created 28/Apr/2021
     * @Updated 28/Apr/2021
     */
    public function show($id)
    {
        try {
            // method/action specific initializations
            $this->subTitle = 'View';
            $this->actionUri = 'show';

            //pre-populated data
            $this->const['subTitle'] = $this->subTitle;
            $this->const['actionUri'] = $this->actionUri;

            //push current controller then method in breadcrumb at position 1 & 2
            $this->breadCrumb['breadCrumData'][1] = [
                'text' => $this->title,
                'url' => $this->url,
                'breadFaClass' => '',
            ];
            $this->breadCrumb['breadCrumData'][2] = [
                'text' => $this->subTitle,
                'breadFaClass' => 'active',
            ];

            $id = \Crypt::decryptString($id);
            $user = $this->customer->customer($id);
            $data['userDetail'] = $user;
            $data = array_merge($data, $this->const, $this->breadCrumb);
            return \View('Admin.' . $this->controllerUri . '.show')->with($data);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * Author: Dipak Verma
     * Email: dipak.verma@nmgtechnologies.com
     * @Method: edit()
     * @Scope: public
     * @params: @id
     * @returns: Returns pre-populated form data with other constants used in UI
     * @Description: It only fetches customer detail as per given id & return it to form
     *  for pre-populating it.
     * @Created 28/Apr/2021
     * @Updated 17/Jun/2021
     */
    public function edit($id)
    {
        try {

            if (request()->ajax()) {

                if (request()->isMethod('get')) {

                    if (request()->isJson()) {
                        $input = request()->json()->all();
                    } else {
                        $input = request()->all();
                    }
                    $id = \Crypt::decryptString($input['id']);
                    $model = 'users';
                    $modelStatus = \DB::table($model)->select('status')->find($id);
                    $modelData = array('status' => '1');
                    $modelValue = array('status' => '<span style="cursor:pointer" class="glyphicon glyphicon-ok" title="Active"></span>');

                    $userData = User::find($id);
                    if ($modelStatus->status == '1') {
                        $modelData = array('status' => '0');
                        $modelValue = array('status' => '<span style="cursor:pointer" class="glyphicon glyphicon-remove" title="Inactive"></span>');

                        // prevent to block if user has any active order 
                        $activeOrderStatus[] = config('admin.constants.ORDER_STATUS_PENDING');
                        $activeOrderStatus[] = config('admin.constants.ORDER_STATUS_IN_PREP');
                        $activeOrderStatus[] = config('admin.constants.ORDER_STATUS_ON_THE_WAY');
                        // $activeOrderStatus[] = config('admin.constants.ORDER_STATUS_COMPLETED');
                        $activeOrderCount = Order::whereIn('order_status_id', $activeOrderStatus)
                            ->where(['user_id' => $id])->get()->count();
                        if ($activeOrderCount > 0) {
                            return response()->json(['success' => false, 'msg' => 'We cannot block this user because it is already associated with an active order']);
                        }


                        $result = \DB::table($model)->where('id', $id)->update($modelData);

                        if (!empty($userData->email)) {
                            ### Account blocked Email#####
                            if (!empty($input['lang'])) {
                                \App::setLocale($input['lang']);
                            }
                            $params['replaceKeywords']['{USER_NAME}'] = $userData->first_name;

                            $params['toEmail'] = trim($userData->email);
                            $params['emailSlug'] = 'account_blocked_for_customer_rider_restaurant';
                            if (!$this->customMailer($params)) {
                                $rollBack = true;
                            }
                            #######################
                        }
                    } else {
                        $result = \DB::table($model)->where('id', $id)->update($modelData);

                        if (!empty($userData->email)) {
                            ### Account un-blocked Email#####
                            if (!empty($input['lang'])) {
                                \App::setLocale($input['lang']);
                            }
                            $params['replaceKeywords']['{USER_NAME}'] = $userData->first_name;
                            $params['toEmail'] = trim($userData->email);
                            $params['emailSlug'] = 'account_unblocked_for_customer_rider_restaurant';
                            if (!$this->customMailer($params)) {
                                $rollBack = true;
                            }
                            #######################
                        }
                    }
                    if ($result) {
                        $status = '1';
                    }
                }
                return response()->json(['success' => $status, 'changeStatus' => $modelValue, 'id' => $id]);
            }

            // method/action specific initializations
            $this->subTitle = 'Edit';
            $this->actionUri = 'edit';
            $formId = 'createUser';

            //pre-populated data
            $this->const['subTitle'] = $this->subTitle;
            $this->const['actionUri'] = $this->actionUri;
            $this->const['formId'] = $formId;
            $this->const['roles'] = [$this->roleId => $this->roleName];

            //push current controller then method in breadcrumb at position 1 & 2
            $this->breadCrumb['breadCrumData'][1] = [
                'text' => $this->title,
                'url' => $this->url,
                'breadFaClass' => '',
            ];
            $this->breadCrumb['breadCrumData'][2] = [
                'text' => $this->subTitle,
                'breadFaClass' => 'active',
            ];

            $id = \Crypt::decryptString($id);
            $user = $this->customer->customer($id);
            $data['userDetail'] = $user;

            $data = array_merge($data, $this->const, $this->breadCrumb);
            return \View('Admin.' . $this->controllerUri . '.edit')->with($data);
        } catch (\Exception $e) {
            // dd($e);
            if (request()->ajax()) {
                $this->saveErrorLog($e);
                return abort(500, $e->getMessage());
            } else {
                $this->saveErrorLog($e);
                return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
            }
        }
    }

    /**
     * Author: Dipak Verma
     * Email: dipak.verma@nmgtechnologies.com
     * @Method: update()
     * @Scope: public
     * @params: @id
     * @input: @commission,
     * @returns: Returns pre-populated form data with other constants used in UI
     * @Description: Update the specified resource in storage.
     * @Created 28/Apr/2021
     * @Updated 28/Apr/2021
     */
    public function update(Request $request, $id)
    {
        try {
            $id = \Crypt::decryptString($id);
            if ($request->isMethod('put')) {
                $rollBack = false;
                \DB::beginTransaction();
                $input = $request->only(['user_name',  'phone_number', 'first_name', 'last_name', 'status', 'avatar']);
                $validator = Validator::make($input, User::Rules($id), User::messages());
                if ($validator->fails()) {
                    return \Redirect::back()->withInput()->withErrors($validator->errors());
                } else {
                    $customer = User::find($id);
                    $oldEmail = $customer->email;
                    $customer->role_id = $this->roleId;
                    $customer->user_name = trim($input['user_name']);
                    // $customer->email = trim($input['email']);
                    $customer->phone_number = trim($input['phone_number']);
                    $customer->first_name = trim($input['first_name']);
                    $customer->last_name = trim($input['last_name']);
                    $customer->status = trim($input['status']);
                    $recdUpdated = $customer->save();

                    if ($recdUpdated) {
                        if ($request->hasFile('avatar')) {
                            //Storage::delete("public/pics/{$user->image}");
                            if (!User::upload_profile_image($request->avatar, $id)) {
                                return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.UnableToUploadAvatarImage'));
                            }
                        }
                    } else {
                        return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.UnableToUpdateFacilityUser'));
                    }
                    if (!$rollBack) {
                        \DB::commit();
                    } else {
                        \DB::rollBack();
                    }
                }
                if (!$rollBack) {
                    \DB::commit();
                    \Session::flash('success', __('admin/flash_msg.success.CustomerUpdated'));
                    return \Redirect::to($this->url);
                }
            }
        } catch (\Exception $e) {
            \DB::rollBack();
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * Author: Dipak Verma
     * Email: dipak.verma@nmgtechnologies.com
     * @Method: destroy()
     * @Scope: public
     * @params: @id
     * @Description: Remove the specified resource from storage then redirect to listing page
     * @Created 20/Nov/2019
     * @Updated 20/Nov/2019
     */
    public function destroy($id)
    {
        try {
            $id = \Crypt::decryptString($id);

            // check if user associate with orders
            $userOrderCount = Order::where(['user_id' => $id])->get()->count();
            if ($userOrderCount > 0) {
                return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.orderAssociateWithUser'));
            }

            $user = User::find($id);
            if ($user->is_delivery_person == 1) {
                $deliveryPerson = DeliveryPersons::where(['user_id' => $user->id])->first();
                if (!empty($deliveryPerson)) {
                    // check any order associate with user or not 
                    $assignedOrderCount = Order::where(['delivery_person_id' => $deliveryPerson->id])->get()->count();
                    if ($assignedOrderCount > 0) {
                        return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.orderAssociateWithDeliveryPerson'));
                    }
                }
            }


            $userOrderCount = Order::where(['user_id' => $id])->get()->count();
            if ($userOrderCount > 0) {
                return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.orderAssociateWithUser'));
            }

            $dirPath = $this->const['avatarStoragePath'] . '/' . $id;

            if (is_dir($dirPath)) {
                deleteDir($dirPath);
            }
            UserAvatars::where('user_id', $id)->delete();

            Notification::where('user_id', $id)->delete();

            $hasEntryAsDeliveryPerson =  DeliveryPersons::where('user_id', $user->id)->first();
            if ($hasEntryAsDeliveryPerson) {
                DeliveryPersons::where('user_id', $user->id)->delete();
            }
            $user->delete();
            \Session::flash('success', __('admin/flash_msg.success.CustomerDeleted'));
            return \Redirect::to($this->url);
        } catch (\Exception $e) {
            // dd($e);
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }
}
