<?php

namespace app\Http\Controllers\Admin;

use app\Http\Controllers\Admin\BaseController;
use app\Model\Admin\Role;
use app\Model\Admin\User;
use app\Model\Admin\ContactTitle;
use Auth;
use DB;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Input;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use OpenApi\Annotations\Contact;

class ContactTitleController extends BaseController
{
    private $const;
    private $roleId;
    private $roleName;
    private $breadCrumb;
    private $url;
    private $title;
    private $subTitle;
    private $controllerUri;
    private $actionUri;
    /**
     * @Method: __construct()
     * @Scope: public
     * @Params: null
     * @Description: Initialize variables like  @roleId, @roleName, @url within private scope of current class
     * @Created 14/Jul/2021
     * @Updated 14/Jul/2021
     */
    public function __construct(ContactTitle $contactTitle)
    {
        try {
            parent::__construct();
            $this->middleware('auth.admin');

            //fixed role for this module
            $this->roleId = config('admin.constants.SUPER_ADMIN_ROLE');
            $role = Role::select('name')->where('id', $this->roleId)->first();
            if ($role->name) {
                $this->roleName = $role->name;
            }

            $this->url = config('admin.path.ADMIN_BASE_URL') . '/contact-title';
            $this->title = 'Manage Contact Title';
            $this->controllerUri = 'contact-title';

            $this->const['url'] = $this->url;
            $this->const['title'] = $this->title;
            $this->const['controllerUri'] = $this->controllerUri;
            $this->const['roleId'] = $this->roleId;
            $this->const['roleName'] = $this->roleName;

            //used in adminForm layout to load validation messges in js variable
            $this->const['objValidationMsgs'] = 'contact-title';
            $this->const['objValidationJs'] = 'contact-title';


            //merge constant into single array with parent controller values
            $this->const = array_merge($this->adminConst(), $this->const);

            //default breadcrumb option (Dashboard) for every route
            $this->breadCrumb['faClass'] = 'fa-table';
            $this->breadCrumb['breadCrumData'] = [
                0 => [
                    'text' => 'Dashboard',
                    'url' => config('admin.path.ADMIN_BASE_URL'),
                    'breadFaClass' => 'fa-dashboard',
                ],

            ];
            $this->contactTitle = $contactTitle;
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }
    /**
     * Author: Jaidev
     * Email: jaidev@nmgtechnologies.com
     * @Method: index()
     * @Scope: public
     * @Params: @page,
     * @returns: A List of Contact Us Titles 
     * @Description: This function fetches  contact us titles from database, sets up
     *  breadcrumb & other constants. 
     * @Created 14/jul/2021
     * @Updated 14/jul/2021
     */
    public function index(Request $request)
    {
        try {
            $this->breadCrumb['breadCrumData'][1] = [
                'text' => $this->const['title'],
            ];
            $page = $request->query('page', 1);
            $limit = $request->query('limit', config('admin.constants.DEFAULT_PAGE_LIMIT'));
            $this->const['page'] = $page;
            $this->const['limit'] = $limit;
            $contactTitle = $this->contactTitle->title($request);
            $data['titleList'] = $contactTitle;
            $data = array_merge($data, $this->const, $this->breadCrumb);
            return \View('Admin.contact-title.index')->with($data);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * Author: Jaidev
     * Email: jaidev@nmgtechnologies.com
     * @Method: create()
     * @Scope: public
     * @input: name
     * @Description: Show input form
     * @Created 14/Jul/2021
     * @Updated 14/Jul/2021
     */
    public function create()
    {
        try {
            // method/action specific initializations
            $this->subTitle = 'Add';
            $this->actionUri = 'create';
            $formId = 'createContactTitle';

            //pre-populated data
            $this->const['subTitle'] = $this->subTitle;
            $this->const['actionUri'] = $this->actionUri;
            $this->const['formId'] = $formId;
            $this->const['roles'] = [$this->roleId => $this->roleName];

            //push current controller then method in breadcrumb at position 1 & 2
            $this->breadCrumb['breadCrumData'][1] = [
                'text' => $this->title,
                'url' => $this->url,
                'breadFaClass' => '',
            ];
            $this->breadCrumb['breadCrumData'][2] = [
                'text' => $this->subTitle,
                'breadFaClass' => 'active',
            ];

            $data = array_merge($this->const, $this->breadCrumb);
            return \View('Admin.' . $this->controllerUri . '.create')->with($data);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }
    /**
     * Author: Jaidev
     * Email: jaidev@nmgtechnologies.com
     * @Method: store()
     * @Scope: public
     * @input: category
     * @returns: After successful submition of data redirect to listing page 
     * @Description: Create the specified resource in storage.
     * @Created 14/Jul/2021
     * @Updated 14/Jul/2021
     */
    public function store(Request $request)
    {

        try {
            if ($request->isMethod('post')) {
                $input = $request->only(['name']);
                $validator = Validator::make($input, ContactTitle::Rules(), ContactTitle::messages());

                if ($validator->fails()) {
                    return \Redirect::back()->withInput()->withErrors($validator->errors());
                } else {
                    $user = ContactTitle::create([
                        'name' => trim($input['name']),
                        'status' => 1,
                    ]);
                    \Session::flash('success', __('admin/flash_msg.success.contactTitleCreated'));
                    return \Redirect::to($this->url);
                }
            }
        } catch (\Exception $e) {
            // dd($e->getMessage());
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * Author: Jaidev
     * Email: jaidev@nmgtechnologies.com
     * @Method: edit()
     * @Scope: public
     * @params: @id
     * @returns: Returns pre-populated form data with other constants used in UI
     * @Description: It only fetches food category detail as per given id & return it to form
     *  for pre-populating it.
     * @Created 14/Jul/2021
     * @Updated 14/Jul/2021
     */
    public function edit($id)
    {
        try {

            if (request()->ajax()) {

                if (request()->isMethod('get')) {

                    if (request()->isJson()) {
                        $input = request()->json()->all();
                    } else {
                        $input = request()->all();
                    }
                    $id = \Crypt::decryptString($input['id']);
                    $model = 'contact_titles';
                    $modelStatus = \DB::table($model)->select('status')->find($id);
                    $modelData = array('status' => '1');
                    $modelValue = array('status' => '<span style="cursor:pointer" class="glyphicon glyphicon-ok" title="Active"></span>');

                    if ($modelStatus->status == '1') {
                        $modelData = array('status' => '0');
                        $modelValue = array('status' => '<span style="cursor:pointer" class="glyphicon glyphicon-remove" title="Inactive"></span>');

                        $result = \DB::table($model)->where('id', $id)->update($modelData);
                    } else {
                        $result = \DB::table($model)->where('id', $id)->update($modelData);
                    }
                    if ($result) {
                        $status = '1';
                    }
                }
                return response()->json(['success' => $status, 'changeStatus' => $modelValue, 'id' => $id]);
            }


            // method/action specific initializations
            $this->subTitle = 'Edit';
            $this->actionUri = 'edit';
            $formId = 'createContactTitle';

            //pre-populated data
            $this->const['subTitle'] = $this->subTitle;
            $this->const['actionUri'] = $this->actionUri;
            $this->const['formId'] = $formId;
            $this->const['roles'] = [$this->roleId => $this->roleName];

            //push current controller then method in breadcrumb at position 1 & 2
            $this->breadCrumb['breadCrumData'][1] = [
                'text' => $this->title,
                'url' => $this->url,
                'breadFaClass' => '',
            ];
            $this->breadCrumb['breadCrumData'][2] = [
                'text' => $this->subTitle,
                'breadFaClass' => 'active',
            ];

            $id = \Crypt::decryptString($id);
            $titleData  = ContactTitle::where(['id' => $id])->first();
            $data['userDetail'] = $titleData;

            $data = array_merge($data, $this->const, $this->breadCrumb);
            return \View('Admin.' . $this->controllerUri . '.edit')->with($data);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }
    /**
     * Author: Jaidev
     * Email: jaidev@nmgtechnologies.com
     * @Method: update()
     * @Scope: public
     * @params: @id
     * @input: category
     * @returns: Returns pre-populated form data with other constants used in UI
     * @Description: Update the specified resource in storage.
     * @Created 14/Jul/2021
     * @Updated 14/Jul/2021
     */
    public function update(Request $request, $id)
    {
        try {
            $id = \Crypt::decryptString($id);
            if ($request->isMethod('put')) {
                $input = $request->only(['name']);
                $validator = Validator::make($input, ContactTitle::Rules($id), ContactTitle::messages());
                if ($validator->fails()) {
                    return \Redirect::back()->withInput()->withErrors($validator->errors());
                } else {
                    $category = ContactTitle::find($id);
                    $category->name = trim($input['name']);
                    $recdUpdated = $category->save();
                    \Session::flash('success', __('admin/flash_msg.success.contactTitleUpdated'));
                    return \Redirect::to($this->url);
                }
            }
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * Author: Jaidev
     * Email: jaidev@nmgtechnologies.com
     * @Method: destroy()
     * @Scope: public
     * @params: @id
     * @Description: Remove the specified resource from storage then redirect to listing page
     * @Created 14/Jul/2021
     * @Updated 14/Jul/2021
     */
    public function destroy($id)
    {
        try {
            $id = \Crypt::decryptString($id);
            $category = ContactTitle::find($id);
            $category->delete();
            \Session::flash('success', __('admin/flash_msg.success.contactTitleDeleted'));
            return \Redirect::to($this->url);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }
}