<?php

namespace app\Http\Controllers\Admin;

use app\Http\Controllers\Admin\BaseController;
use app\Model\Admin\BlockedReason;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class BlockedReasonsController extends BaseController
{
    private $const;
    private $breadCrumb;
    private $url;
    private $title;
    private $subTitle;
    private $controllerUri;
    private $actionUri;

    /**
     * @Method: __construct()
     * @Scope: public
     * @Params: null
     * @Description: Initialize variable with module scope, breadcrumb
     * @Created 02/Dec/2019
     * @Updated 02/Dec/2019
     */
    public function __construct()
    {
        try {
            parent::__construct();
            $this->middleware('auth.admin');

            $this->url = config('admin.path.ADMIN_BASE_URL') . '/blocked-reasons';
            $this->title = 'Manage Blocked Reason';
            $this->controllerUri = 'blocked-reasons';

            //department logo path & base urls
            $this->const['url'] = $this->url;
            $this->const['title'] = $this->title;
            $this->const['controllerUri'] = $this->controllerUri;

            //used in adminForm layout to load validation messges in js variable
            $this->const['objValidationMsgs'] = 'blocked-reason';
            $this->const['objValidationJs'] = 'blocked-reason';

            $this->const = array_merge($this->adminConst(), $this->const);
            //dd($this->const);
            //default breadcrumb option (Dashboard) for every route
            $this->breadCrumb['faClass'] = 'fa-table';
            $this->breadCrumb['breadCrumData'] = [
                0 => [
                    'text' => 'Dashboard',
                    'url' => config('admin.path.ADMIN_BASE_URL'),
                    'breadFaClass' => 'fa-dashboard',
                ],

            ];
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * @Method: index()
     * @Scope: public
     * @Params: @page, @limit, @search
     * @returns: Display a listing of the resource
     * @Description: master table department stores all the department used in various facilities
     * @Created 02/Dec/2019
     * @Updated 02/Dec/2019
     */
    public function index(Request $request)
    {
        try {
            $this->breadCrumb['breadCrumData'][1] = [
                'text' => $this->const['title'],
            ];
            $page = $request->query('page', 1);
            $limit = $request->query('limit', config('admin.constants.DEFAULT_PAGE_LIMIT'));
            $this->const['page'] = $page;
            $this->const['limit'] = $limit;

            $search = !empty($request->search) ? $request->search : '';

            $data = [];
            $query = BlockedReason::select(
                'blocked_reasons.id',
                'blocked_reasons.reason',
                'blocked_reasons.status',
                'blocked_reasons.created_at'
            );
            if (!empty($search)) {
                $query->where('reason', 'LIKE', "%$search%");
            }

            $blockReasonList = $query->sortable(['created_at' => 'asc'])->paginate($limit);

            $data['blockReasonList'] = $blockReasonList;
            $data = array_merge($data, $this->const, $this->breadCrumb);
            return \View('Admin.' . $this->controllerUri . '.index')->with($data);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * @Method: create()
     * @Scope: public
     * @Params: null
     * @Description: Show the form for creating a new resource.
     * @Created 02/Dec/2019
     * @Updated 02/Dec/2019
     */
    public function create()
    {
        try {
            // method/action specific initializations
            $this->subTitle = 'Add';
            $this->actionUri = 'create';
            $formId = 'blockedReasonForm';

            //pre-populated data
            $this->const['subTitle'] = $this->subTitle;
            $this->const['actionUri'] = $this->actionUri;
            $this->const['formId'] = $formId;

            //push current controller then method in breadcrumb at position 1 & 2
            $this->breadCrumb['breadCrumData'][1] = [
                'text' => $this->title,
                'url' => $this->url,
                'breadFaClass' => '',
            ];
            $this->breadCrumb['breadCrumData'][2] = [
                'text' => $this->subTitle,
                'breadFaClass' => 'active',
            ];

            $data = [];

            $data = array_merge($this->const, $this->breadCrumb);
            return \View('Admin.' . $this->controllerUri . '.create')->with($data);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * @Method: store()
     * @Scope: public
     * @input: @name,@status
     * @Description: Store a newly created resource in storage.
     * @Created 02/Dec/2019
     * @Updated 02/Dec/2019
     */
    public function store(Request $request)
    {
        try {
            if ($request->isMethod('post')) {
                $input = $request->only(['reason', 'status']);
                $validator = Validator::make($input, BlockedReason::Rules(), BlockedReason::messages());
                if ($validator->fails()) {
                    return \Redirect::back()->withInput()->withErrors($validator->errors());
                } else {
                    $blockReason = BlockedReason::create([
                        'reason' => $input['reason'],
                        'status' => $input['status'],
                    ]);
                    if (empty($blockReason->id)) {
                        $rollBack = true;
                        return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.UnableToCreateBlockReason'));
                    }
                    \Session::flash('success', __('admin/flash_msg.success.BlockReasonCreated'));
                    return \Redirect::to($this->url);
                }
            }
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * @Method: edit()
     * @Scope: public
     * @params  int  $id
     * @Description: Show the form for editing the specified resource.
     * @Created 02/Dec/2019
     * @Updated 02/Dec/2019
     */
    public function edit($id)
    {
        try {

            if (request()->ajax()) {

                if (request()->isMethod('get')) {

                    if (request()->isJson()) {
                        $input = request()->json()->all();
                    } else {
                        $input = request()->all();
                    }
                    $id = \Crypt::decryptString($input['id']);
                    $model = \Crypt::decryptString($input['model']);
                    $modelStatus = \DB::table($model)->select('status')->find($id);
                    $modelData = array('status' => '1');
                    $modelValue = array('status' => '<span style="cursor:pointer" class="glyphicon glyphicon-ok" title="Active"></span>');

                    if ($modelStatus->status == '1') {
                        $modelData = array('status' => '0');
                        $modelValue = array('status' => '<span style="cursor:pointer" class="glyphicon glyphicon-remove" title="Inactive"></span>');

                        $result = \DB::table($model)->where('id', $id)->update($modelData);
                    } else {
                        $result = \DB::table($model)->where('id', $id)->update($modelData);
                    }
                    if ($result) {
                        $status = '1';
                    }
                }
                return response()->json(['success' => $status, 'changeStatus' => $modelValue, 'id' => $id]);
            }

            // method/action specific initializations
            $this->subTitle = 'Edit';
            $this->actionUri = 'edit';
            $formId = 'blockedReasonForm';

            //pre-populated data
            $this->const['subTitle'] = $this->subTitle;
            $this->const['actionUri'] = $this->actionUri;
            $this->const['formId'] = $formId;

            //push current controller then method in breadcrumb at position 1 & 2
            $this->breadCrumb['breadCrumData'][1] = [
                'text' => $this->title,
                'url' => $this->url,
                'breadFaClass' => '',
            ];
            $this->breadCrumb['breadCrumData'][2] = [
                'text' => $this->subTitle,
                'breadFaClass' => 'active',
            ];

            $id = \Crypt::decryptString($id);
            $blockReason = BlockedReason::select(
                'blocked_reasons.id',
                'blocked_reasons.reason',
                'blocked_reasons.status'
            )
                ->where('id', $id)
                ->first();

            $data['blockReasonDetail'] = $blockReason;
            $data = array_merge($data, $this->const, $this->breadCrumb);
            return \View('Admin.' . $this->controllerUri . '.edit')->with($data);

        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * @Method: update()
     * @Scope: public
     * @params: @id
     * @input: @name, @status
     * @Description: Update the specified resource in storage.
     * @Created 02/Dec/2019
     * @Updated 02/Dec/2019
     */
    public function update(Request $request, $id)
    {
        try {
            $id = \Crypt::decryptString($id);
            if ($request->isMethod('put')) {
                $input = $request->only(['reason', 'status']);
                $validator = Validator::make($input, BlockedReason::Rules($id), BlockedReason::messages());
                if ($validator->fails()) {
                    return \Redirect::back()->withInput()->withErrors($validator->errors());
                } else {
                    if (!BlockedReason::find($id)->update([
                        'reason' => $input['reason'],
                        'status' => $input['status'],
                    ])) {
                        return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.UnableToUpdateblockedReason'));
                    }

                    \Session::flash('success', __('admin/flash_msg.success.blockedReasonUpdated'));
                    return \Redirect::to($this->url);
                }
            }
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * @Method: destroy()
     * @Scope: public
     * @params: @id
     * @Description: Remove the specified resource from storage.
     * @Created 02/Dec/2019
     * @Updated 02/Dec/2019
     */
    public function destroy(Request $request, $id = null)
    {
        try {
            $id = \Crypt::decryptString($id);

            if(!empty($id)){
            $count_blocked = \app\Model\Admin\BlockedReason::withCount('blocked_users')->where('id', $id)->first();

            if (!empty($count_blocked->blocked_users_count) && isset($count_blocked->blocked_users_count)) {
                \Session::flash('error', __('admin/flash_msg.error.blockedReasonNotDeleted'));
                return redirect()->back();
            }

            $recordDel = BlockedReason::find($id)->delete();

            if (!empty($recordDel)) {
                \Session::flash('success', __('admin/flash_msg.success.blockedReasonDeleted'));
            }
        }

            return \Redirect::to($this->url);

        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }
}
