<?php
/**
 * @Description: The `FacilityUsersController` is used for manage the facility admin user in MCA
 * application. Platform Admin User will have the access & use this module to add & then assign
 * a facility to manage for facility admin.
 * @auther: NMG - PR
 * @dated: 20/Nov/2019
 */
namespace app\Http\Controllers\Admin;

use app\Http\Controllers\Admin\BaseController;
use app\Model\Admin\Role;
use app\Model\Admin\User;
use Auth;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Input;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;

class AdminUsersController extends BaseController
{
    private $const;
    private $roleId;
    private $roleName;
    private $breadCrumb;
    private $url;
    private $title;
    private $subTitle;
    private $controllerUri;
    private $actionUri;

    /**
     * @Method: __construct()
     * @Scope: public
     * @Params: null
     * @Description: Initialize variables like @avatarStoragePath, @roleId, @roleName, @url within private scope of current class
     * @Created 20/Nov/2019
     * @Updated 20/Nov/2019
     */
    public function __construct()
    {
        try {
            parent::__construct();
            $this->middleware('auth.admin');

            //fixed role for this module
            $this->roleId = config('admin.constants.SUB_ADMIN_ROLE');
            $role = Role::select('name')->where('id', $this->roleId)->first();
            if ($role->name) {
                $this->roleName = $role->name;
            }

            $this->url = config('admin.path.ADMIN_BASE_URL') . '/admin-users';
            $this->title = 'Admin Users';
            $this->controllerUri = 'admin-users';

            $this->const['url'] = $this->url;
            $this->const['title'] = $this->title;
            $this->const['controllerUri'] = $this->controllerUri;
            $this->const['roleId'] = $this->roleId;
            $this->const['roleName'] = $this->roleName;

            //used in adminForm layout to load validation messges in js variable
            $this->const['objValidationMsgs'] = 'user';
            $this->const['objValidationJs'] = 'user';

            //merge constant into single array with parent controller values
            $this->const = array_merge($this->adminConst(), $this->const);

            //default breadcrumb option (Dashboard) for every route
            $this->breadCrumb['faClass'] = 'fa-table';
            $this->breadCrumb['breadCrumData'] = [
                0 => [
                    'text' => 'Dashboard',
                    'url' => config('admin.path.ADMIN_BASE_URL'),
                    'breadFaClass' => 'fa-dashboard',
                ],

            ];
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * @Method: index()
     * @Scope: public
     * @Params: @page, @limit, @search
     * @returns: A List of facility admin users identified by Facility Admin role(id:2)
     * @Description: This function fetches facility admin users from database, sets up
     *  breadcrumb & other constants. It has the search utility using username, first_name,
     *  last_name & email.
     * @Created 20/Nov/2019
     * @Updated 20/Nov/2019
     */
    public function index(Request $request)
    {
        try {
            $this->breadCrumb['breadCrumData'][1] = [
                'text' => $this->const['title'],
            ];

            $page = $request->query('page', 1);
            $limit = $request->query('limit', config('admin.constants.DEFAULT_PAGE_LIMIT'));
            $this->const['page'] = $page;
            $this->const['limit'] = $limit;

            $search = !empty($request->search) ? $request->search : '';

            $data = [];
            /* \DB::enableQueryLog(); */
            $query = User::select(
                'users.id',
                'users.first_name',
                'users.last_name',
                'users.user_name',
                'users.email',
                'users.phone_number',
                'users.mobile_no',
                'users.profile_image',
                'users.email_verified_at',
                'users.status',
                'users.created_at'
            )
                ->with([
                    'latestAvatar' => function ($query) {
                        $query->select('id', 'user_id', 'avatar');
                    }])
                ->where('users.role_id', $this->roleId);
            if (!empty($search)) {

                $query->where(function ($q) use ($search) {
                    $q->orWhere('users.email', 'LIKE', "%$search%");
                    //$q->orWhereRaw("concat(first_name, ' ', last_name) like '%$search%'");
                    $q->orWhere('users.user_name', 'like', "%{$search}%");
                    $q->orWhere('users.first_name', 'like', "%{$search}%");
                    $q->orWhere("users.user_name", 'like', "%$search%");
                });
            }

            $users = $query->sortable(['created_at' => 'desc'])->paginate($limit);

            $data['userList'] = $users;
            $data = array_merge($data, $this->const, $this->breadCrumb);
            return \View('Admin.admin-users.index')->with($data);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * @Method: create()
     * @Scope: public
     * @Params: null
     * @returns: This method returns an array having breadcrumb data & other constants
     *  like title, sub-title, formId, static roles array of facility admin only etc.
     * @Description: Used for pre-populating some field of from & sets up constants used in UI.
     * @Created 20/Nov/2019
     * @Updated 20/Nov/2019
     */
    public function create()
    {
        try {
            // method/action specific initializations
            $this->subTitle = 'Add';
            $this->actionUri = 'create';
            $formId = 'createUser';

            //pre-populated data
            $this->const['subTitle'] = $this->subTitle;
            $this->const['actionUri'] = $this->actionUri;
            $this->const['formId'] = $formId;
            $this->const['roles'] = [$this->roleId => $this->roleName];

            //push current controller then method in breadcrumb at position 1 & 2
            $this->breadCrumb['breadCrumData'][1] = [
                'text' => $this->title,
                'url' => $this->url,
                'breadFaClass' => '',
            ];
            $this->breadCrumb['breadCrumData'][2] = [
                'text' => $this->subTitle,
                'breadFaClass' => 'active',
            ];

            $data = array_merge($this->const, $this->breadCrumb);
            return \View('Admin.' . $this->controllerUri . '.create')->with($data);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * @Method: store()
     * @Scope: public
     * @input: @role_id, @user_name, @password, @email, @first_name, @last_name, @status, @avatar
     * @returns: After successful submission redirected to facility admin listing page
     * @Description: This function store the input data for facility admin in `users` database table,
     *  also corresponding new entry has been made in `user_avatars` table. The data has been proccessed
     *  only after approved by validation rules defined in user model. Avatar image is optional here.
     * @Created 20/Nov/2019
     * @Updated 20/Nov/2019
     */
    public function store(Request $request)
    {

        try {
            if ($request->isMethod('post')) {
                $rollBack = false;
                $input = $request->only(['facility_id', 'user_name', 'email', 'phone_number', 'mobile_no', 'first_name', 'last_name', 'status', 'avatar','city','pincode']);
                $validator = Validator::make($input, User::Rules(), User::messages());

                if ($validator->fails()) {

                    return \Redirect::back()->withInput()->withErrors($validator->errors());
                } else {

                    $plainText = $this->randomPassword();
                     $input['password'] = bcrypt($plainText);
                    \DB::beginTransaction();
                    $user = User::create([
                        'role_id' => $this->roleId,
                        'user_name' => trim($input['user_name']),
                        'email' => trim($input['email']),
                        'phone_number' => trim($input['phone_number']),
                        'mobile_no' => trim($input['mobile_no']),
                        'password' => trim($input['password']),
                        'first_name' => trim($input['first_name']),
                        'last_name' => trim($input['last_name']),
                        'status' => trim($input['status']),
                    ]);

                    if (!empty($user->id)) {
                        if ($request->hasFile('avatar')) {
                            //Storage::delete("public/pics/{$user->image}");
                            if (!User::upload_avatar($request->avatar, $user->id)) {
                                /* if (User::find($user->id)->update([
                                'profile_image' => $input['image'],
                                ])) {
                                } */
                                $rollBack = true;
                                \DB::rollBack();
                                return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.UnableToUploadAvatarImage'));
                            }
                        }
                    } else {
                        $rollBack = true;
                        return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.success.UnableToCreateFacilityUser'));
                    }

                    if (!$rollBack) {

                        ###Welcome Email#####
                        if (!empty($input['lang'])) {
                            \App::setLocale($input['lang']);
                        }
                        $params['replaceKeywords']['{USER_NAME}'] = $user->full_name;
                        $params['replaceKeywords']['{LOGIN_ID}'] = $user->email;
                        $params['replaceKeywords']['{PASSWORD}'] = $plainText;
                        $params['toEmail'] = $user->email;
                        $params['emailSlug'] = 'app_registration';
                        $this->customMailer($params);

                        #######################
                        \DB::commit();
                        \Session::flash('success', __('admin/flash_msg.success.FacilityUserCreated'));
                        return \Redirect::to($this->url);
                    }
                }
            }
        } catch (\Exception $e) {
            \DB::rollBack();
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * @Method: edit()
     * @Scope: public
     * @params: @id
     * @returns: Returns pre-populated form data with other constants used in UI
     * @Description: It only fetches user data as per given id & return it to form
     *  for pre-populating it.
     * @Created 20/Nov/2019
     * @Updated 20/Nov/2019
     */
    public function edit($id)
    {
        try {

            if (request()->ajax()) {

                if (request()->isMethod('get')) {

                    if (request()->isJson()) {
                        $input = request()->json()->all();
                    } else {
                        $input = request()->all();
                    }
                    $id = \Crypt::decryptString($input['id']);
                    $model = 'users';
                    $modelStatus = \DB::table($model)->select('status')->find($id);
                    $modelData = array('status' => '1');
                    $modelValue = array('status' => '<span style="cursor:pointer" class="glyphicon glyphicon-ok" title="Active"></span>');

                    if ($modelStatus->status == '1') {
                        $modelData = array('status' => '0');
                        $modelValue = array('status' => '<span style="cursor:pointer" class="glyphicon glyphicon-remove" title="Inactive"></span>');

                        $result = \DB::table($model)->where('id', $id)->update($modelData);
                    } else {
                        $result = \DB::table($model)->where('id', $id)->update($modelData);
                    }
                    if ($result) {
                        $status = '1';
                    }
                }
                return response()->json(['success' => $status, 'changeStatus' => $modelValue, 'id' => $id]);
            }

            // method/action specific initializations
            $this->subTitle = 'Edit';
            $this->actionUri = 'edit';
            $formId = 'createUser';

            //pre-populated data
            $this->const['subTitle'] = $this->subTitle;
            $this->const['actionUri'] = $this->actionUri;
            $this->const['formId'] = $formId;
            $this->const['roles'] = [$this->roleId => $this->roleName];

            //push current controller then method in breadcrumb at position 1 & 2
            $this->breadCrumb['breadCrumData'][1] = [
                'text' => $this->title,
                'url' => $this->url,
                'breadFaClass' => '',
            ];
            $this->breadCrumb['breadCrumData'][2] = [
                'text' => $this->subTitle,
                'breadFaClass' => 'active',
            ];

            $id = \Crypt::decryptString($id);
            $user = User::select(
                'users.id',
                'users.first_name',
                'users.last_name',
                'users.user_name',
                'users.email',
                'users.phone_number',
                'users.mobile_no',
                'users.status'
            )
                ->where('id', $id)
                ->with([
                    'latestAvatar' => function ($query) {
                        $query->select('id', 'user_id', 'avatar');
                    }])
                ->first();

            $data['userDetail'] = $user;

            $data = array_merge($data, $this->const, $this->breadCrumb);
            return \View('Admin.' . $this->controllerUri . '.edit')->with($data);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * @Method: update()
     * @Scope: public
     * @params: @id
     * @input: @role_id, @user_name, @email, @first_name, @last_name, @status, @avatar
     * @returns: Returns pre-populated form data with other constants used in UI
     * @Description: Update the specified resource in storage.
     * @Created 20/Nov/2019
     * @Updated 20/Nov/2019
     */
    public function update(Request $request, $id)
    {
        try {
            $id = \Crypt::decryptString($id);
            if ($request->isMethod('put')) {
                $rollBack = false;
                \DB::beginTransaction();
                $input = $request->only(['user_name', 'email', 'phone_number', 'mobile_no', 'first_name', 'last_name', 'status', 'avatar']);
                $validator = Validator::make($input, User::Rules($id), User::messages());
                if ($validator->fails()) {
                    return \Redirect::back()->withInput()->withErrors($validator->errors());
                } else {
                    $facilityUser = User::find($id);
                    $oldEmail = $facilityUser->email;
                    $facilityUser->role_id = $this->roleId;
                    $facilityUser->user_name = trim($input['user_name']);
                    $facilityUser->email = trim($input['email']);
                    $facilityUser->phone_number = trim($input['phone_number']);
                    $facilityUser->mobile_no = trim($input['mobile_no']);
                    $facilityUser->first_name = trim($input['first_name']);
                    $facilityUser->last_name = trim($input['last_name']);
                    $facilityUser->status = trim($input['status']);

                    if ($facilityUser->isDirty('email')) {
                        ###Welcome Email#####
                        if (!empty($input['lang'])) {
                            \App::setLocale($input['lang']);
                        }
                        $params['replaceKeywords']['{USER_NAME}'] = $facilityUser->user_name;
                        $params['replaceKeywords']['{OLD_EMAIL}'] = $oldEmail;
                        $params['replaceKeywords']['{NEW_EMAIL}'] = trim($input['email']);
                        $params['toEmail'] = trim($input['email']);
                        $params['emailSlug'] = 'change_email';
                        if (!$this->customMailer($params)) {
                            $rollBack = true;
                        }
                        #######################
                    }

                    $recdUpdated = $facilityUser->save();

                    if ($recdUpdated) {
                        if ($request->hasFile('avatar')) {
                            //Storage::delete("public/pics/{$user->image}");
                            if (!User::upload_avatar($request->avatar, $id)) {
                                return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.UnableToUploadAvatarImage'));
                            }
                        }
                    } else {
                        return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.UnableToUpdateFacilityUser'));
                    }
                    if (!$rollBack) {
                        \DB::commit();
                    } else {
                        \DB::rollBack();
                    }
                }
                if (!$rollBack) {
                    \DB::commit();
                    \Session::flash('success', __('admin/flash_msg.success.FacilityUserUpdated'));
                    return \Redirect::to($this->url);
                }
            }
        } catch (\Exception $e) {
            \DB::rollBack();
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }

    }

    /**
     * @Method: destroy()
     * @Scope: public
     * @params: @id
     * @Description: Remove the specified resource from storage then redirect to listing page
     * @Created 20/Nov/2019
     * @Updated 20/Nov/2019
     */
    public function destroy($id)
    {
        try {
            $id = \Crypt::decryptString($id);
            $dirPath = $this->const['avatarStoragePath'] . '/' . $id;

            if (is_dir($dirPath)) {
                deleteDir($dirPath);
            }
            $user = User::find($id);
            $user->delete();
            \Session::flash('success', __('admin/flash_msg.success.FacilityUserDeleted'));
            return \Redirect::to($this->url);

        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * @Method: show()
     * @Scope: public
     * @params: @id
     * @Description: show detail of records
     * @Created 10/Dec/2019
     * @Updated 10/Dec/2019
     */
    public function show($id)
    {
        try {
            // method/action specific initializations
            $this->subTitle = 'View';
            $this->actionUri = 'show';

            //pre-populated data
            $this->const['subTitle'] = $this->subTitle;
            $this->const['actionUri'] = $this->actionUri;

            //push current controller then method in breadcrumb at position 1 & 2
            $this->breadCrumb['breadCrumData'][1] = [
                'text' => $this->title,
                'url' => $this->url,
                'breadFaClass' => '',
            ];
            $this->breadCrumb['breadCrumData'][2] = [
                'text' => $this->subTitle,
                'breadFaClass' => 'active',
            ];

            $id = \Crypt::decryptString($id);
            $user = User::select(
                'users.id',
                'users.role_id',
                'users.first_name',
                'users.last_name',
                'users.user_name',
                'users.email',
                'users.phone_number',
                'users.mobile_no',
                'users.status',
                'users.created_at'
            )
                ->where('id', $id)
                ->with([
                    'latestAvatar' => function ($query) {
                        $query->select('id', 'user_id', 'avatar');
                    }])
                ->first();

            $data['userDetail'] = $user;
            $data = array_merge($data, $this->const, $this->breadCrumb);
            return \View('Admin.' . $this->controllerUri . '.show')->with($data);

        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    } 
}
