<?php

namespace app\Http\Controllers\Admin;

use app\Http\Controllers\Admin\BaseController;
use app\Model\Admin\Role;
use app\Model\Admin\User;
use app\Model\Admin\Order;
use app\Model\Admin\UserAvatars;
use app\Model\Admin\DeliveryPersons;

use Auth;
use DB;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Input;
use Validator;

class AdminController extends BaseController
{
    private $url;
    private $title;
    private $breadCrumb;

    /**
     * @Method: __construct()
     * @Scope: public
     * @Params: null
     * @Description: Initialize variables like @avatarDir, @roleId, @roleName, @url within private scope of current class
     * @Created 22/June/2021
     * @Updated 22/June/2021
     */
    public function __construct()
    {
        try {

            $this->middleware('auth.admin')->except('login');

            $this->url = config('admin.path.ADMIN_BASE_URL');
            $this->title = 'Admin';

            $this->const['url'] = $this->url;
            $this->const['title'] = $this->title;

            //used in adminForm layout to load validation messges in js variable
            $this->const['objValidationMsgs'] = 'user';
            $this->const['objValidationJs'] = 'user';

            $this->const = array_merge($this->adminConst(), $this->const);

            //default breadcrumb option (Dashboard) for every route
            $this->breadCrumb['faClass'] = 'fa-table';
            $this->breadCrumb['breadCrumData'] = [
                0 => [
                    'text' => 'Dashboard',
                    'url' => config('admin.path.ADMIN_BASE_URL'),
                    'breadFaClass' => 'fa-dashboard',
                ],

            ];
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * @Method: login()
     * @Scope: public
     * @Input: @email, @password
     * @returns: redirect user to admin dashboard after successful login
     * @Description: Admin users with role - Plateform Admin & Facility Can login admin panel
     *  using their credentials like- email, password.
     * @Created 23/June/2021
     * @Updated 23/June/2021
     */
    public function login(Request $request)
    {
        try {
            if (Auth::guard('admin')->check()) {
                return \Redirect::to($this->url . '/dashboard');
            }

            // method/action specific initializations
            $this->subTitle = 'Login';
            $formId = 'loginForm';

            $this->const['subTitle'] = $this->subTitle;
            $this->const['formId'] = $formId;

            $data = array_merge($this->const);

            if ($request->isMethod('post')) {
                $input = $request->only(['email', 'password', 'remember']);
                $validator = Validator::make($input, User::loginRules(), User::messages());
                if ($validator->fails()) {
                    return \Redirect::back()->withInput()->withErrors($validator->errors());
                } else {
                    $remember = (Input::has('remember')) ? true : false;
                    if (Auth::guard('admin')->attempt(['email' => $request->email, 'password' => $request->password, 'role_id' => [1]], $remember)) {

                        if (Auth::guard('admin')->user()->status == '0') {
                            Auth::guard('admin')->logout();
                            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.userNotActivate'));
                        }

                        if ($remember) {
                            $arr['email'] = $request->email;
                            $arr['password'] = $request->password;
                            $arr['rememberMe'] = true;
                            $cookie_name = "admin_remember";
                            setcookie($cookie_name, base64_encode(json_encode($arr)), time() + (86400 * 30), "/"); // 86400 = 1 day
                        } else {
                            if (!empty($_COOKIE['admin_remember'])) {
                                $cookie_name = "admin_remember";
                                setcookie($cookie_name, "", time() + (86400 * 30), "/");
                            }
                        }
                        return \Redirect::to($this->url . '/dashboard');
                    } else {
                        return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.InavlidLoginCredentials'));
                        // dd($request->all());
                    }
                }
            }
            return view('Admin/admin/login')->with($data);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * @Method: logout()
     * @Scope: public
     * @returns: redirect login screen after successful logout
     * @Description: Ends the user authentication session
     * @Created 23/June/2021
     * @Updated 23/June/2021
     */
    public function logout()
    {
        try {
            Auth::guard('admin')->logout();
            \Session::flash('success', __('admin/flash_msg.success.logout'));
            return \Redirect::to($this->url);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * @Method: forgotPassword()
     * @Scope: public
     * @Input: @email
     * @returns: send otp for reset password to registerd email inbox
     * @Description: Reset password otp has been sent to registered email inbox
     *  with an encrypted tokn expires few minutes set in admin configuartion.
     * @Created 23/June/2021
     * @Updated 23/June/2021
     */

    public function forgotPassword(Request $request)
    {
        try {
            // method/action specific initializations
            $this->subTitle = 'Forgot Password';
            $formId = 'forgotPassword';

            $this->const['subTitle'] = $this->subTitle;
            $this->const['formId'] = $formId;

            $data = array_merge($this->const);
            if ($request->isMethod('post')) {
                $input = $request->only(['email']);
                $validator = Validator::make($input, User::forgotPasswordRules(), User::messages());
                if ($validator->fails()) {
                    return \Redirect::back()->withInput()->withErrors($validator->errors());
                } else {
                    // $user = User::where('email', $request->email)->where('role_id', '<>', config('admin.constants.PUBLIC_USERS_ROLE'))->select('id', 'email', 'first_name', 'last_name', 'status')->first();
                    $user = User::where('email', $request->email)->select('id', 'user_name', 'email', 'first_name', 'last_name', 'status')->first();

                    if (empty($user)) {
                        return \Redirect::back()->withInput()->withErrors(__('admin/flash_msg.error.emailNotExists'));
                    } else {

                        if ($user->status == 1) {
                            $rollBack = false;
                            \DB::beginTransaction();
                            $plainText = $this->randomPassword();
                            $user->reset_password_token = bcrypt($plainText);
                            $userId = $this->encryptDecryptId('encrypt', $user->id);
                            $email = $this->encryptDecryptId('encrypt', $user->email);

                            $plainText = $this->generatePIN(4);
                            $reset_password_token = $user->id . ':' . $user->email . ':' . Carbon::now()->timestamp;
                            $token = $this->encryptDecryptId('encrypt', $reset_password_token);
                            $user->reset_password_token = $token;
                            $user->reset_password_sent_at = Carbon::now();
                            $user->verification_otp = $plainText;
                            $user->verification_otp_sent_at = Carbon::now();

                            if ($user->save()) {
                                ###Reset Password Email#####
                                if (!empty($input['lang'])) {
                                    \App::setLocale($input['lang']);
                                }
                                $params['replaceKeywords']['{USER_NAME}'] = $user->user_name;
                                $params['replaceKeywords']['{LOGIN_ID}'] = $user->email;
                                $params['replaceKeywords']['{OTP}'] = $plainText;
                                $params['toEmail'] = $user->email;
                                // $params['emailSlug'] = 'reset_password';
                                $params['emailSlug'] = 'forgot_password';
                                if (!$this->customMailer($params)) {
                                    $rollBack = true;
                                }
                            }
                            if (!$rollBack) {
                                \DB::commit();
                                // return \Redirect::back()->withInput()->with('success', __('admin/validation/user.success.reset'));
                                \Session::flash('success', \Config::get('flash_msg.OTPSENT'));
                                return redirect()->route('ResetPassword');
                            } else {
                                \DB::rollBack();
                                return \Redirect::back()->withInput()->with('error', __('admin/validation/user.error.reset'));
                            }
                        } else {
                            return \Redirect::back()->withInput()->with('error', __('admin/validation/user.error.blocked'));
                        }
                    }
                }
            }
            return view('Admin/admin/forgot_password')->with($data);
        } catch (\Exception $e) {
            \DB::rollBack();
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.EmailWrong'));
        }
    }

    /**
     * @Method: resetPassword()
     * @Scope: public
     * @Input: @email, @new_password, @new_password_confirmation
     * @returns: redirect to login screen after successful operation
     * @Description: Reset password after-
     *   - validating token, check user email & user status
     * @Created 23/June/2021
     * @Updated 23/June/2021
     */
    public function resetPassword(Request $request, $token = null)
    {
        try {
            $this->subTitle = 'Reset Password';
            $formId = 'resetPasswordForm';
            $this->const['subTitle'] = $this->subTitle;
            $this->const['formId'] = $formId;
            if (!empty($request->otp)) {
                $token = $request->otp;
                $user = User::select('id', 'reset_password_sent_at', 'email', 'status', 'verification_otp_sent_at')->where('verification_otp', $token)->first();
                // dd($user['reset_password_sent_at']);
                //dd($toeknParts);
                if ($user) {
                    $userId = $user->id;
                    $email = $user->email;
                    $otpCreatedTime = strtotime($user->verification_otp_sent_at);
                    $currentTime = strtotime(date('Y-m-d H:i:s'));
                    //Calculate the difference.
                    $difference = $currentTime - $otpCreatedTime;
                    $difference = floor($difference / (60));
                    if (empty($user)) {
                        return \Redirect::back()->withInput()->withErrors(__('admin/flash_msg.error.invalidResetToken'));
                    } elseif ($difference > \Config::get('admin.constants.RESET_TOKEN_EXPIRE')) {
                        return \Redirect::back()->withInput()->withErrors(__('admin/flash_msg.error.invalidResetToken'));
                    } else {
                        //$updateVal = array('reset_password_token' => '', 'password' => bcrypt($request->new_password));
                        $updateVal = [
                            'reset_password_sent_at' => null,
                            'reset_password_token' => null,
                            'verification_otp'     => null,
                            'verification_otp_sent_at' => null,
                            'email_verified_at' => null,
                            'password' => bcrypt($request->new_password),
                        ];
                        $pswdUpdate = \DB::table('users')->where('id', '=', $user->id)->update($updateVal);
                        if ($pswdUpdate) {
                            \Session::flash('success', __('admin/flash_msg.success.passwordReset'));
                            return \Redirect::to(\Config::get('admin.path.ADMIN_BASE_URL'));
                        } else {
                            return \Redirect::back()->withInput()->withErrors(__('admin/flash_msg.error.resetPassword'));
                        }
                    }
                } else {
                    return \Redirect::back()->withInput()->withErrors(__('admin/flash_msg.error.wrongOTP'));
                    return \Redirect::to(\Config::get('admin.path.ADMIN_BASE_URL'));
                }
            }
            $data = array_merge($this->const, $this->breadCrumb);
            return view('Admin/admin/reset_password', $data);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * @Method: changePassword()
     * @Scope: public
     * @Input: @current_password, @new_password, @new_password_confirmation
     * @returns: redirect to login screen after successful operation
     * @Description: It will destroy auth session & prompt user to login again
     * @Created 22/Nov/2019
     * @Updated 22/Nov/2019
     */
    public function changePassword(Request $request)
    {
        try {
            $this->user = Auth::guard('admin')->user();
            if (!$this->user) {
                return \Redirect::to(config("admin.constants.ADMIN_ROUTE_NAME"));
            }
            $this->subTitle = 'Change Password';
            $formId = 'changePassword';

            $this->const['subTitle'] = $this->subTitle;
            $this->const['formId'] = $formId;

            $this->breadCrumb['breadCrumData'][1] = [
                'text' => $this->const['subTitle'],
                'breadFaClass' => 'fa-th-list',
            ];

            if ($request->isMethod('post')) {
                $user = Auth::guard('admin')->user();
                $input = $request->only(['password', 'new_password', 'new_password_confirmation']);
                $validator = Validator::make($input, User::changePasswordRules($user), User::messages());
                if ($validator->fails()) {
                    return \Redirect::back()->withInput()->withErrors($validator->errors());
                } else {
                    if (!\Hash::check(Input::get('password'), $user->password)) {
                        return \Redirect::route('changePassword')->withErrors(__('admin/flash_msg.error.wrongCPassword'));
                    } else {
                        $plainText = Input::get('new_password');
                        $user->password = Input::get('new_password');
                        $rollBack = false;
                        \DB::beginTransaction();
                        if ($user->save()) {

                            ###Reset Password Email#####
                            if (!empty($input['lang'])) {
                                \App::setLocale($input['lang']);
                            }
                            $params['replaceKeywords']['{USER_NAME}'] = $user->user_name;
                            $params['replaceKeywords']['{LOGIN_ID}'] = $user->email;
                            $params['replaceKeywords']['{PASSWORD}'] = $plainText;
                            $params['toEmail'] = $user->email;
                            $params['emailSlug'] = 'change_password';

                            if (!$this->customMailer($params)) {
                                $rollBack = true;
                            }

                            if (!$rollBack) {
                                \DB::commit();
                                \Session::flash('success', __('admin/flash_msg.success.passwordChanged'));
                                Auth::guard('admin')->logout();
                                return \Redirect::to($this->url);
                            } else {
                                \DB::rollBack();
                                \Session::flash('error', __('admin/flash_msg.error.passwordMailActivate'));
                            }
                        } else {
                            \Session::flash('error', __('admin/flash_msg.error.passwordChanged'));
                        }
                    }
                }
            }
            $data = array_merge($this->const, $this->breadCrumb);
            return view('Admin/admin/change_password')->with($data);
        } catch (\Exception $e) {
            \DB::rollBack();
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * @Method: dashboard()
     * @Scope: public
     * @Description: Display the recent statistics of CMS & serve all links as per user access right
     * @Created 22/June/2021
     * @Updated 22/June/2021
     */
    public function dashboard(Request $request)
    {
        try {
            if (!Auth::guard('admin')->check()) {
                \Session::flash('error', __('admin/flash_msg.error.loginFirst'));
                return \Redirect::to($this->url);
            }
            $this->title = 'Dashboard';
            $this->subTitle = 'Control panel';
            $this->const['title'] = $this->title;
            $this->const['subTitle'] = $this->subTitle;

            $role = Role::select('is_super_admin')->where('id', Auth::guard('admin')->user()->role_id)->first();
            $data = [];
            $faQ = User::select(
                'users.id',
                'users.first_name',
                'users.last_name',
                'users.user_name',
                'users.email',
                'users.profile_image',
                'users.email_verified_at',
                'users.status',
                'users.created_at'
            )
                ->with([
                    'latestAvatar' => function ($query) {
                        $query->select('id', 'user_id', 'avatar');
                    }
                ]);
            $facilityUsers = $faQ->where('users.role_id', config('admin.constants.SUB_ADMIN_ROLE'))->sortable(['created_at' => 'desc'])->paginate(5);
            $mwQ = User::select(
                'users.id',
                'users.first_name',
                'users.last_name',
                'users.user_name',
                'users.email',
                'users.profile_image',
                'users.email_verified_at',
                'users.status',
                'users.created_at'
            )
                ->with([
                    'latestAvatar' => function ($query) {
                        $query->select('id', 'user_id', 'avatar');
                    }
                ]);



            if ($role->is_super_admin) {
                $publicCustomerCount = User::where('users.role_id', config('admin.constants.PUBLIC_USERS_ROLE'))->count();
                $deliveryPersonCount = DeliveryPersons::count();
                $restaurantOwnerCount = User::where('users.role_id', config('admin.constants.RESTAURANT_OWNER_ROLE'))->count();
                $totalOrderCount = Order::count();
                $restaurantData = User::where(['users.role_id' => config('admin.constants.RESTAURANT_OWNER_ROLE')])->orderBy('id', 'DESC')->limit(5)->get();
                $orderData = Order::orderBy('id', 'DESC')->limit(5)->get();
                $customerData = User::where(['users.role_id' => config('admin.constants.PUBLIC_USERS_ROLE')])->orderBy('id', 'DESC')->limit(5)->get();
            } else {
                $publicCustomerCount = User::where('users.role_id', config('admin.constants.PUBLIC_USERS_ROLE'))->count();
                $deliveryPersonCount = DeliveryPersons::count();
                $restaurantOwnerCount = User::where('users.role_id', config('admin.constants.RESTAURANT_OWNER_ROLE'))->count();
                $totalOrderCount = Order::count();
                $restaurantData = User::where(['users.role_id' => config('admin.constants.RESTAURANT_OWNER_ROLE')])->get();
                $orderData = Order::orderBy('id', 'DESC')->limit(5)->get();
                $customerData = User::where(['users.role_id' => config('admin.constants.PUBLIC_USERS_ROLE')])->orderBy('id', 'DESC')->limit(5)->get();
            }

            //facility logo path & base urls
            $this->const['defaultLogoUrl'] = config('admin.path.facilityLogo.DEFAULT_URL');
            $this->const['logoBaseUrl'] = config('admin.path.facilityLogo.BASE_URL');
            $this->const['logoStoragePath'] = config('admin.path.STORAGE_DIR') .
                config('admin.path.UPLOAD_DIR') . '/' .
                config('admin.path.facilityLogo.DIR');

            $data['publicCustomerCount'] = $publicCustomerCount;
            $data['deliveryPersonCount'] = $deliveryPersonCount;
            $data['restaurantOwnerCount'] = $restaurantOwnerCount;
            $data['totalOrderCount'] = $totalOrderCount;
            $data['restaurantData'] = $restaurantData;
            $data['orderData'] = $orderData;
            $data['customerData'] = $customerData;

            if ($role->is_super_admin) {
                $view = 'Admin/admin/dashboard';
            } else {
                $view = 'Admin/admin/facility-dashboard';
            }

            $data = array_merge($data, $this->const, $this->breadCrumb);
            return view($view)->with($data);
        } catch (\Exception $e) {
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }

    /**
     * @Method: myProfile()
     * @Scope: public
     * @Description: Looged in user can view & update certain profile information for himself
     * @Created 22/June/2021
     * @Updated 22/June/2021
     */
    public function myProfile(Request $request)
    {
        try {
            // method/action specific initializations
            $this->subTitle = 'My Profile';
            $this->actionUri = 'my-profile';
            $formId = 'profileEditForm';
            //pre-populated data
            $this->const['subTitle'] = $this->subTitle;
            $this->const['actionUri'] = $this->actionUri;
            $this->const['formId'] = $formId;
            //push current controller then method in breadcrumb at position 1 & 2
            $this->breadCrumb['breadCrumData'][1] = [
                'text' => $this->subTitle,
                'breadFaClass' => 'active',
            ];

            $user = Auth::guard('admin')->user();
            $avatar = UserAvatars::select('avatar')->where('user_id', $user->id)->latest()->first();
            if (!empty($avatar->avatar)) {
                $this->const['latestAvatar'] = $avatar->avatar;
            } else {
                $this->const['latestAvatar'] = '';
            }

            if ($request->isMethod('put')) {
                $input = $request->only(['first_name', 'last_name', 'avatar']);
                $validator = Validator::make($input, User::profileRules($user), User::messages());
                if ($validator->fails()) {
                    return \Redirect::back()->withInput()->withErrors($validator->errors());
                } else {
                    $rollBack = false;
                    if ($input['first_name'] != $user->first_name) {
                        $user->first_name = $input['first_name'];
                        $user->user_name = $input['first_name'];
                    }
                    if ($input['last_name'] != $user->last_name) {
                        $user->last_name = $input['last_name'];
                    }
                    \DB::beginTransaction();
                    if ($request->hasFile('avatar')) {

                        $upload = User::upload_avatar($request->avatar, $user->id);
                        if (!empty($upload)) {
                            $user->profile_image = $upload;
                        } else {
                            $rollBack = true;
                            \DB::rollBack();
                            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.UnableToUploadAvatarImage'));
                        }
                    }
                    if (!$rollBack) {
                        $user->save();
                        \DB::commit();
                        \Session::flash('success', __('admin/flash_msg.success.profileUpdated'));
                        return \Redirect::route('myProfile');
                    }
                }
            }

            $data = array_merge($this->const, $this->breadCrumb);
            return view('Admin/admin/my_profile', $data)->withUser($user);
        } catch (\Exception $e) {
            \DB::rollBack();
            $this->saveErrorLog($e);
            return \Redirect::back()->withInput()->with('error', __('admin/flash_msg.error.SomethingWrong'));
        }
    }
}
