<?php

namespace app\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use app\Model\Api\V1\Order;
use app\Model\Api\V1\DeliveryPersonAssignment;
use app\Model\Vendor\DeliveryPerson as DeliveryPersonVendor;
use app\Model\Vendor\OrderLog;
use app\Model\Api\V1\User;
use app\Components\Api\V1\PushNotification;
use app\Helpers\CommonHelper;
use app\Traits\OrderStatusTrait;



class ReassignRider extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'rider:reassign';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'This command re-assign rider on orders which is not accepted/rejected by rider within 60 seconds';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     *
     * @return mixed
     */
    public function handle()
    {
        // re-assign rider who didn't accepted order requested withing 1 min
        $data = \DB::table('delivery_person_assignments')->where(['status' => 1])->where('created_at', '<', \Carbon\Carbon::now()->subSeconds(60)->toDateTimeString())->get();
        if (!empty($data)) {
            foreach ($data as $assignData) {
                $orderData =  Order::find($assignData->order_id);
                if (!empty($orderData)) {
                    // get order assignment
                    $delAssignData = DeliveryPersonAssignment::where(['order_id' => $orderData->id, 'delivery_person_id' => $orderData->delivery_person_id])->first();

                    if ($delAssignData->status == 1) {
                        $delAssignData->status = 3; // reject status
                        $delAssignData->reason = 'Not accepted within 60 seconds';
                        $delAssignData->save();

                        // assign new delivery person or cancel order if no rider available
                        $availableRider =  DeliveryPersonVendor::getClosestAvailableRider($orderData->latitude, $orderData->longitude, $orderData->id);
                        if (!empty($availableRider)) {
                            $orderData->delivery_person_id = $availableRider->id;
                            $orderData->save();
                            //
                            $dpData['order_id'] = $orderData->id;
                            $dpData['latitude'] = $availableRider->latitude;
                            $dpData['longitude'] = $availableRider->longitude;
                            $dpData['delivery_person_id'] = $availableRider->id;
                            $dpData['status'] = 1;
                            DeliveryPersonAssignment::create($dpData);


                            ### send push notification to customer ###
                            $title = config('api.constants.push_notification.title.rider_updated');
                            $body = config('api.constants.push_notification.description.rider_updated');
                            $notification = array('title' => $title, 'body' => $body, 'sound' => 'default', 'badge' => '1', 'actionId' => config('api.constants.push_notification.actionId.customer_order_details'), 'orderId' => $orderData->id);
                            \app\Components\Api\V1\PushNotification::sendNotification($orderData->user_id, $notification, $orderData->id);
                            ### send push notification to customer ###



                            #### send notification to rider ###
                            $title = config('api.constants.push_notification.title.rider_new_order');
                            $body =  config('api.constants.push_notification.description.rider_new_order');
                            $notification = array('title' => $title, 'body' => $body, 'sound' => 'default', 'badge' => '1', 'actionId' => config('api.constants.push_notification.actionId.rider_order_details'), 'orderId' => $orderData->id);

                            \app\Components\Api\V1\PushNotification::sendNotification($availableRider->user_id, $notification, $orderData->id);
                            #### end sending notification to rider ###


                        } else {
                            // send order in queue for rider assignment after 10 mins
                            $orderData->delivery_person_id = null;
                            $orderData->rider_assignment_at = date("Y-m-d H:i:s");
                            $orderData->save();


                            ### send push notification to customer ###
                            $title = config('api.constants.push_notification.title.rider_in_queue');
                            $body = config('api.constants.push_notification.description.rider_in_queue');
                            $notification = array('title' => $title, 'body' => $body, 'sound' => 'default', 'badge' => '1', 'actionId' => config('api.constants.push_notification.actionId.customer_order_details'), 'orderId' => $orderData->id);
                            \app\Components\Api\V1\PushNotification::sendNotification($orderData->user_id, $notification, $orderData->id);

                            // end send order in queue for rider assignment after 10 mins
                        }
                    }
                } else {
                    // order id not found
                }
            }
        }




        // re-assign rider on those orders where rider wasn't available 10 mins ago
        $data = \DB::table('orders')
            ->whereIn('order_status_id', [config('admin.constants.ORDER_STATUS_IN_PREP'), config('admin.constants.ORDER_STATUS_ON_THE_WAY')])
            ->whereNull('delivery_person_id')
            ->where('rider_assignment_at', '<', \Carbon\Carbon::now()->toDateTimeString())
            ->get();


        if (!empty($data)) {
            foreach ($data as $assignData) {
                $orderData =  Order::find($assignData->id);
                if (!empty($orderData)) {

                    // check rider re-assignment time difference //
                    // if its more than 10 mins then cancel the order
                    $assignAt = strtotime($assignData->rider_assignment_at);
                    $now = time();
                    $minuteDiff = round(($now - $assignAt) / 60);
                    if ($minuteDiff > 10) {
                        // cancel order
                        // cancel order due to non-availibity of rider
                        // send order cancellation mail
                        $orderData->order_status_id = config('admin.constants.ORDER_STATUS_CANCELED');
                        $orderData->delivery_person_id = null;
                        $orderData->rider_assignment_at = null;
                        $orderData->save();
                        OrderLog::saveLog($orderData->id, $orderData->order_status_id);

                        $user = User::find($orderData->user_id);

                        #### send push notification to customer ##
                        $title = "Order cancelled";
                        $body = "Your order has been cancelled due to un-availability of rider";
                        $notification = array('title' => $title, 'body' => $body, 'sound' => 'default', 'badge' => '1', 'orderId' => $orderData->id, 'actionId' => config('api.constants.push_notification.actionId.customer_order_details'));
                        PushNotification::sendNotification($orderData->user_id, $notification,  $orderData->id);

                        #### end sending push notification to customer ##

                        if (!empty($user->email)) {
                            ### Email Sending #####
                            $params['replaceKeywords']['{USER_NAME}'] = $orderData->firstname . ' ' . $orderData->lastname;
                            $params['replaceKeywords']['{ORDER_ID}'] = $orderData->id;
                            $params['replaceKeywords']['{AMOUNT}'] = $orderData->total_amount;
                            $params['replaceKeywords']['{DELIVER_TO}'] = $orderData->firstname . ' ' . $orderData->lastname;
                            $params['replaceKeywords']['{ADDRESS}'] = $orderData->address1;
                            $params['replaceKeywords']['{ADDRESS_TYPE}'] = CommonHelper::getAddressType($orderData->address_type);
                            $params['toEmail'] = $user->email;
                            $params['emailSlug'] = 'customer_restaurant_due_nonavailability_cancle_order';
                            $this->customMailer($params);
                        }
                    } else {
                        // find rider if order status in prep

                        if ($orderData->order_status_id == config('admin.constants.ORDER_STATUS_IN_PREP') || $orderData->order_status_id == config('admin.constants.ORDER_STATUS_ON_THE_WAY')) {
                            // assign new delivery person or cancel order if no rider available
                            $availableRider =  DeliveryPersonVendor::getClosestAvailableRider($orderData->latitude, $orderData->longitude, $orderData->id);
                            if (!empty($availableRider)) {
                                $orderData->delivery_person_id = $availableRider->id;
                                $orderData->save();
                                //
                                $dpData['order_id'] = $orderData->id;
                                $dpData['latitude'] = $availableRider->latitude;
                                $dpData['longitude'] = $availableRider->longitude;
                                $dpData['delivery_person_id'] = $availableRider->id;
                                $dpData['status'] = 1;
                                DeliveryPersonAssignment::create($dpData);

                                #### send notification to rider ###
                                $title = config('api.constants.push_notification.title.rider_new_order');
                                $body =  config('api.constants.push_notification.description.rider_new_order');
                                $notification = array('title' => $title, 'body' => $body, 'sound' => 'default', 'badge' => '1', 'actionId' => config('api.constants.push_notification.actionId.rider_order_details'), 'orderId' => $orderData->id);

                                \app\Components\Api\V1\PushNotification::sendNotification($availableRider->user_id, $notification, $orderData->id);
                                #### end sending notification to rider ###
                            }
                        }
                    }
                }
            }
        }

        echo "reassigned riders";
    }
}
